package csbase.client.ias;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.DesktopTask;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.srvproxies.UserProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.logic.UserInfo;

/**
 * Dilogo de modificao de dados de um usurio.
 * 
 * <p>
 * Esta classe  utilizada somente por usurios comuns desejando alterar seus
 * dados. O administrador deve utilizar a classe
 * {@link csbase.client.ias.AdminModifyUserInfoDialog} para modificar tanto seus
 * dados como o de outros usurios.
 * </p>
 * 
 * @author Leonardo Barros
 */
public class ModifyUserInfoDialog extends UserInfoDialog {
  /** Usurio sendo modificado */
  protected User user;

  /** Boto de alterao de dados do usurio */
  protected JButton modifyUserButton;

  /** Boto de alterao de senha do usurio */
  protected JButton passwordButton;

  /** Indica se uma nova senha foi fornecida para o usurio */

  // #TODO Verificar se esse campo no poderia ser substitudo por uma 
  // verificao no atributo PASSWORD.
  protected boolean passwordGiven;

  /**
   * Cria o dilogo de modificao de dados de um usurio.
   * 
   * @param ownerWindow janela que criou este dilogo.
   * @param userId identificador do usurio a ser editado.
   * 
   * @throws IllegalArgumentException no foi possvel obter o usurio.
   */
  public ModifyUserInfoDialog(Window ownerWindow, Object userId) {
    super(ownerWindow, ClientUtilities.addSystemNameToTitle(LNG
      .get("ias.user.update_title")));
    if (userId == null) {
      throw new IllegalArgumentException("userId == null");
    }
    user =
      UserProxy.getUser(ownerWindow, dialogTitle,
        LNG.get("IAS_WAITING_USER_INFO"), userId);
    passwordGiven = false;
  }

  /**
   * {@inheritDoc}<br>
   * Se o usurio confirmar a modificao, verifica se os dados requeridos esto
   * preenchidos (ver {@link UserInfoDialog#requiredFieldsAreFilled()}. Se no
   * estiverem, pede o preenchimento e cancela o fechamento.
   */
  @Override
  protected boolean confirmClose() {
    Object[] options =
      { LNG.get("IAS_UPDATE"), LNG.get("IAS_NOT_UPDATE"), LNG.get("IAS_CANCEL") };

    // #TODO Usar constantes para as tags de linguagem.
    int selected =
      JOptionPane.showOptionDialog(userInfoDialog,
        LNG.get("IAS_USER_UPDATE_CONFIRMATION"),
        LNG.get("IAS_USER_UPDATE_NOT_DONE_TITLE"), JOptionPane.DEFAULT_OPTION,
        JOptionPane.INFORMATION_MESSAGE, null, options, options[0]);
    if (options[selected].equals(LNG.get("IAS_UPDATE"))) {
      if (requiredFieldsAreFilled()) {
        return modifyUser();
      }
      StandardErrorDialogs.showErrorDialog(userInfoDialog, dialogTitle,
        LNG.get("IAS_FILL_ALL_FIELDS"));
      return false;
    }
    if (options[selected].equals(LNG.get("IAS_NOT_UPDATE"))) {
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}<br>
   * Desabilita a edio do login (toda a estrutura de diretrios dos projetos 
   * baseada no login dos usurios, portanto no so permitidas alteraes no
   * login) e preenche os campos com os dados atuais do usurio para edio.
   */
  @Override
  protected JPanel makeUpperPanel() {
    JPanel panel = super.makeUpperPanel();
    login.setEditable(false);
    fillFieldsWithUserData();
    return panel;
  }

  /**
   * Preenche os campos com os dados do usurio.
   */
  protected void fillFieldsWithUserData() {
    // Obtm os dados do usurio (a obteno dos dados  realizada dentro de
    // uma DesktopTask para possibilitar o repaint da tela, pois o processo
    // pode ser demorado).
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        userInfo.setAttribute(User.NAME, user.getName());
        userInfo.setAttribute(User.LOGIN, user.getLogin());
        userInfo.setAttribute(User.EMAILS, user.getEmails());
        userInfo.setAttribute(UserGroup.USERGROUP_ID,
          user.getAttribute(UserGroup.USERGROUP_ID));
        userInfo.setAttribute(User.ROLE_IDS, user.getAttribute(User.ROLE_IDS));
        userInfo.setAttribute(User.PERMISSION_IDS,
          user.getAttribute(User.PERMISSION_IDS));
        userInfo.setAttribute(User.FORCE_LOCAL_LOGIN,
          user.getAttribute(User.FORCE_LOCAL_LOGIN));
        setResult(userInfo);
      }
    };
    task.start(userInfoDialog, dialogTitle, LNG.get("IAS_WAITING_USER_INFO"));
    Exception exception = task.getException();
    if (exception != null) {
      StandardErrorDialogs.showErrorDialog(userInfoDialog, LNG.get("ERRO")
        + " - " + dialogTitle, exception);
      return;
    }
    userInfo = (UserInfo) task.getResult();

    // Embora todos os atributos sejam recuperados, somente os visveis so 
    // copiados para a interface.
    // #TODO No parece ser necessrio recuperar todos os atributos aqui.
    // Verificar com testes.
    login.setText(userInfo.getLogin());
    name.setText((String) userInfo.getAttribute(User.NAME));
    String[] emails = (String[]) userInfo.getAttribute(User.EMAILS);
    if (emails != null) {
      for (String email : emails) {
        emailsModel.addElement(email);
      }
    }
  }

  /**
   * Cria o boto de modificao de senha de usurio, atribuindo-lhe como ao o
   * mtodo {@link ModifyUserInfoDialog#modifyPassword()}.
   * 
   * @return o boto de modificao de senha de usurio.
   */
  protected JButton makePasswordButton() {
    passwordButton = new JButton(LNG.get("IAS_USER_PASSWORD_UPDATE"));
    passwordButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        modifyPassword();
      }
    });
    return passwordButton;
  }

  /**
   * Exibe um dilogo de modificao da senha do usurio. Se uma senha for
   * passada e todos os campos requeridos estiverem preenchidos, habilita o
   * boto de modificao.
   */
  private void modifyPassword() {
    IncludePasswordDialog dialog =
      new IncludePasswordDialog(userInfoDialog, user);
    String password = dialog.getNewPassword();
    if (password == null) {
      return;
    }
    userInfo.setAttribute(User.PASSWORD, password);
    passwordGiven = true;
    enableModifyUserButtonIfAllowed();
  }

  /**
   * Cria o boto de modificao de dados do usurio. Esse boto somente 
   * habilitado quando alguma modificao  feita nos campos. O mtodo
   * {@link ModifyUserInfoDialog#modifyUser()}  associado como ao do boto.
   * 
   * @return o boto de modificao de dados do usurio.
   */
  private JButton makeModifyUserButton() {
    JButton button = new JButton(LNG.get("IAS_USER_UPDATE"));
    button.setEnabled(false);
    button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        // #TODO No  mais necessria essa validao, o processo de 
        // habilitao/desabilitao do boto j se encarrega disso.
        if (requiredFieldsAreFilled()) {
          if (modifyUser()) {
            userInfoDialog.close();
          }
        }
        else {
          StandardErrorDialogs.showErrorDialog(userInfoDialog, dialogTitle,
            LNG.get("IAS_FILL_ALL_FIELDS"));
        }
      }
    });
    return button;
  }

  /**
   * Executa a ao de modificar o usurio, usando os dados preenchidos nos
   * campos da janela. O mtodo assume que qualquer erro ocorrido na alterao
   * do usurio dever ser reportado no ponto em que ocorrer, portanto no se
   * preocupa com a exibio de mensagem de erro, apenas de sucesso.
   * 
   * @return <code>true</code> se os dados do usurios tiverem sido modificados
   *         com sucesso, <code>false</code> caso contrrio.
   */
  protected boolean modifyUser() {
    updateUserInfo();

    // #TODO Verificar se esse digest precisa ser atribudo sempre aqui.
    userInfo.setAttribute(User.PASSWORD_DIGEST, user.getPasswordDigest());
    User modifiedUser =
      UserProxy.modifyUser(userInfoDialog, dialogTitle,
        LNG.get("IAS_WAITING_USER_MODIFICATION"), user.getId(), userInfo);
    if (modifiedUser == null) {
      return false;
    }
    ClientRemoteMonitor.getInstance().setPassword(
      (String) this.userInfo.getAttribute(User.PASSWORD));
    StandardDialogs.showInfoDialog(userInfoDialog, dialogTitle, MessageFormat
      .format(LNG.get("IAS_USER_MODIFIED_WITH_SUCCESS"),
        new Object[] { user.getLogin() }));
    this.user = modifiedUser;
    passwordGiven = false;
    modifyUserButton.setEnabled(false);

    return true;
  }

  /**
   * {@inheritDoc}<br>
   * Adiciona um boto de modificao de dados do usurio ao painel.
   */
  @Override
  protected JPanel makeLowerPanel() {
    JPanel buttonPanel = super.makeLowerPanel();
    modifyUserButton = makeModifyUserButton();
    modifyUserButton.setEnabled(false);
    buttonPanel.add(modifyUserButton, 0);
    return buttonPanel;
  }

  /**
   * {@inheritDoc}<br>
   * Se a troca de senhas for permitida ou o usurio sendo editado tiver o
   * atributo de login local forado, um boto de alterao de senhas ser
   * adicionado ao painel.
   */
  @Override
  protected JPanel makeMiddlePanel() {
    JPanel panel = super.makeMiddlePanel();
    boolean forceLocalLogin = false;
    Object attribute = user.getAttribute(User.FORCE_LOCAL_LOGIN);
    if (attribute != null) {
      forceLocalLogin = ((Boolean) attribute).booleanValue();
    }
    if (ClientRemoteMonitor.getInstance().canChangePasswords()
      || forceLocalLogin) {
      passwordButton = makePasswordButton();
      panel.add(passwordButton);
    }
    return panel;
  }

  /**
   * {@inheritDoc}<br>
   * Verifica se houve alguma mudana nos dados do usurio (login, nome, e-mail
   * ou senha). No caso da senha, como esta no  guardada e portanto no pode
   * ser verificada, qualquer senha entrada  considerada como nova senha, mesmo
   * que ela seja idntica  anterior.
   */
  @Override
  protected boolean unsavedDataInput() {
    if (!login.getText().trim().equals(user.getLogin())) {
      return true;
    }
    else if (userInfo.getAttribute(User.PASSWORD) != null) {
      return true;
    }
    else if (!name.getText().trim().equals(user.getName())) {
      return true;
    }

    String[] currentEmails = (String[]) userInfo.getAttribute(User.EMAILS);
    if (currentEmails == null) {
      return emailsModel.getSize() > 0;
    }
    if (currentEmails.length != emailsModel.getSize()) {
      return true;
    }
    for (int inx = 0; inx < currentEmails.length; inx++) {
      if (!currentEmails[inx].equals(emailsModel.get(inx))) {
        return true;
      }
    }

    return false;
  }

  /**
   * Habilita o boto de modificao se as condies requeridas forem atendidas.
   * Caso contrrio, desabilita.
   */
  protected void enableModifyUserButtonIfAllowed() {
    modifyUserButton
      .setEnabled(unsavedDataInput() && requiredFieldsAreFilled());
  }

  /**
   * {@inheritDoc}<br>
   * Chama o {@link #enableModifyUserButtonIfAllowed() mtodo} que trata o
   * estado habilitado do boto de modificar usurio.
   */
  @Override
  protected void fireOnDataChanged() {
    enableModifyUserButtonIfAllowed();
  }
}
