package csbase.client.util.filechooser.filetablepanel;

import java.awt.Component;
import java.awt.Window;
import java.util.ArrayList;
import java.util.List;

import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.Task;
import csbase.client.util.filechooser.ClientFileChooser;
import csbase.client.util.filechooser.filters.ClientFileAllFilter;
import csbase.client.util.filechooser.filters.ClientFileFilterInterface;
import csbase.client.util.filechooser.util.ClientFileChooserUtils;
import csbase.logic.ClientFile;

/**
 * Modelo de arquivos locais.
 *
 * @author Tecgraf/PUC-Rio
 */
class ClientFileTableModel extends AbstractTableModel {

  /**
   * Lista de arquivos.
   */
  final private List<ClientFile> list = new ArrayList<ClientFile>();

  /**
   * Arquivo.
   */
  private ClientFile clientFile;

  /**
   * Filtro.
   */
  private ClientFileFilterInterface filter;

  /**
   * Nomes das colunas
   */
  final private String[] colunms;

  /**
   * Indicativo de que arquivos considerados ocultos devem ser vistos.
   */
  private boolean showHiddenFiles;

  /**
   * Componente de GUI de referncia. (Usado para a execuo da
   * <code>RemoteTask</code>).
   */
  final private Component parent;

  /**
   * Construtor
   *
   * @param file objeto raiz da tabela
   * @param filter filtro
   * @param parent componente de referncia do modelo.
   */
  public ClientFileTableModel(final ClientFile file,
    final ClientFileAllFilter filter, final Component parent) {
    final String typeText = getString("type.column");
    final String nameText = getString("name.column");
    final String sizeText = getString("size.column");
    final String modifiedText = getString("modified.column");
    this.colunms = new String[] { typeText, nameText, sizeText, modifiedText };
    this.clientFile = file;
    this.filter = filter;
    this.parent = parent;
    refresh();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(final int col) {
    return ClientFile.class;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public int getColumnCount() {
    return colunms.length;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public String getColumnName(final int col) {
    return colunms[col];
  }

  /**
   * Consulta o atributo (ver documentao de {@link #clientFile}).
   *
   * @return o atributo.
   */
  public final ClientFile getDirectory() {
    return clientFile;
  }

  /**
   * @return a lista
   */
  final public List<ClientFile> getFileList() {
    return list;
  }

  /**
   * Consulta o atributo (ver documentao de {@link #filter}).
   *
   * @return o atributo.
   */
  public final ClientFileFilterInterface getFilter() {
    return filter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public int getRowCount() {
    if (list == null) {
      return 0;
    }
    return list.size();
  }

  /**
   * Consulta uma string internacionalizada.
   *
   * @param tag tag
   * @return o texto
   */
  final private String getString(final String tag) {
    final Class<ClientFileChooser> clazz = ClientFileChooser.class;
    final String prefix = clazz.getSimpleName();
    return LNG.get(prefix + "." + tag);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public ClientFile getValueAt(final int row, final int col) {
    if (row < 0) {
      return null;
    }
    final ClientFile f = list.get(row);
    return f;
  }

  /**
   * Obtm um arquivo da lista atual cujo nome comece com um determinado prefixo
   * (no sensvel ao case).
   *
   * @param prefix o prefixo.
   * @return a lista de arquivos que comeam com o prefixo definido (a lista
   *         pode ser vazia).
   */
  final public List<ClientFile> getFilesWithPrefix(String prefix) {
    List<ClientFile> files = new ArrayList<ClientFile>();
    if (prefix != null) {
      for (ClientFile file : list) {
        String fileName = file.getName().toLowerCase();
        if (fileName.startsWith(prefix.toLowerCase())) {
          files.add(file);
        }
      }
    }
    return files;
  }

  /**
   * Atualiza a lista interna de objetos.
   */
  final public void refresh() {
    list.clear();
    if (clientFile == null) {
      return;
    }

    Task<ClientFile[]> task = new Task<ClientFile[]>() {
      @Override
      protected void performTask() throws Exception {
        final ClientFile[] children = clientFile.getChildren();
        setResult(children);
      }
    };

    final String taskMessage = clientFile.getName();
    final String taskTitle = "";
    Window window = null;
    if (parent != null) {
      window = SwingUtilities.getWindowAncestor(parent);
    }
    final boolean executed = task.execute(window, taskTitle, taskMessage);
    final ClientFile[] children;
    if (!executed || task.getError() != null) {
      children = null;
    }
    else {
      children = task.getResult();
    }

    if (children != null) {
      for (final ClientFile f : children) {
        final boolean isHidden = ClientFileChooserUtils.isHiddenFile(f);
        if (!showHiddenFiles && isHidden) {
          continue;
        }
        if (filter.accept(f)) {
          list.add(f);
        }
      }
    }
    fireTableDataChanged();
  }

  /**
   * Ajusta o atributo (ver documentao de {@link #clientFile}).
   *
   * @param file o novo valor a ser ajustado.
   */
  public final void setDirectory(final ClientFile file) {
    this.clientFile = file;
    refresh();
  }

  /**
   * Ajusta o atributo (ver documentao de {@link #filter}).
   *
   * @param filter o novo valor a ser ajustado.
   */
  public final void setFilter(final ClientFileFilterInterface filter) {
    this.filter = filter;
    refresh();
  }

  /**
   * Indica que arquivos ocultos devem ser exibidos.
   *
   * @param flag indicativo
   */
  final public void showHiddenFiles(boolean flag) {
    this.showHiddenFiles = flag;
    refresh();
  }

  /**
   * Indica se arquivos ocultos devem ser vistos.
   *
   * @return indicativo
   */
  public final boolean isHiddenFilesShown() {
    return this.showHiddenFiles;
  }
}
