package csbase.server.services.administrationservice;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.logic.SuperUserPermission;
import csbase.logic.User;
import csbase.logic.UserInfo;
import csbase.logic.UserOutline;
import csbase.logic.UserUpdate;

/**
 * Implementao padro do DAO de usurio. Utiliza armazenamento em arquivo
 * (serializao de objetos).
 * 
 * @see DefaultAdministrationDAOFactory
 * @author Jorge Alexandre
 * 
 */
public class DefaultUserDAO implements UserDAO {

  private UserIO userRepository;

  public DefaultUserDAO() {
    userRepository = new UserIO();
  }

  @Override
  public User createUser(UserInfo info) throws DAOException {
    User user;
    try {
      user = userRepository.write(info);
    }
    catch (Exception e) {
      throw new DAOException(e, info.getAttributes().toString());
    }
    if (user == null) {
      throw new DAOException("Falha em UserIO.write", info.getAttributes()
        .toString());
    }
    return user;
  }

  @Override
  public void deleteUser(Object id) throws DAOException {
    try {
      userRepository.delete(id);
    }
    catch (Exception e) {
      throw new DAOException(e, "id: " + id);
    }
  }

  @Override
  public User modifyUser(Object id, UserInfo info) throws DAOException {
    User user;
    try {
      user = userRepository.write(info);
    }
    catch (Exception e) {
      throw new DAOException(e, info.getAttributes().toString()
        + " - para o usurio: " + id);
    }
    if (user == null) {
      throw new DAOException("Falha em UserIO.write", info.getAttributes()
        .toString()
        + " - para o usurio: " + id);
    }
    return user;
  }

  @Override
  public Set<Object> readAllUserIds() throws DAOException {
    try {
      List<User> users = userRepository.readAll();
      Set<Object> idSet = new HashSet<Object>();
      for (User user : users) {
        idSet.add(user.getId());
      }
      return idSet;
    }
    catch (Exception e) {
      throw new DAOException(e);
    }
  }

  @Override
  public List<UserOutline> readAllUserOutlines() throws DAOException {
    try {
      List<User> users = userRepository.readAll();
      List<UserOutline> outlines = new Vector<UserOutline>();
      for (User user : users) {
        outlines.add(user.getOutline());
      }
      return outlines;
    }
    catch (Exception e) {
      throw new DAOException(e);
    }
  }

  @Override
  public List<UserUpdate> readAllUserUpdates() throws DAOException {
    try {
      List<User> users = userRepository.readAll();
      List<UserUpdate> updates = new Vector<UserUpdate>();
      for (User user : users) {
        updates.add(new UserUpdate(user));
      }
      return updates;
    }
    catch (Exception e) {
      throw new DAOException(e);
    }
  }

  @Override
  public List<User> readAllUsers() throws DAOException {
    try {
      return userRepository.readAll();
    }
    catch (Exception e) {
      throw new DAOException(e);
    }
  }

  @Override
  public User readUser(Object id) throws DAOException {
    try {
      return userRepository.read(id);
    }
    catch (Exception e) {
      throw new DAOException(e, "id: " + id);
    }
  }

  /**
   * {@inheritDoc} Se o usurio for um super-usurio, cria uma instncia de
   * <code>User</code> para o usurio delegado por ele. OBS: O
   * Administrador(admin) no precisa de permisso de super-usurio para delegar
   * um outro usurio e, por questes de segurana, no pode ser um usurio
   * delegado por nenhum super-usurio.
   * 
   * @param superuser usurio no qual ser verificado a permisso de
   *        super-usurio.
   * @param delegatedLogin login do usurio que ser delegado pelo super-usurio
   *        sem a necessidade de senha.
   * 
   * @return uma instncia de <code>User</code> para o usurio delegado ou nulo
   *         se <code>delegatedLogin</code> for nulo.
   * 
   * @throws PermissionException se o usurio no tiver permisso de
   *         super-usurio ou se o usurio delegado for o
   *         Administrador(login=admin).
   * @throws ServiceFailureException se ocorrer erro na criao do usurio
   *         delegado.
   */
  @Override
  public User changeUser(User superuser, String delegatedLogin)
    throws DAOException {
    SuperUserPermission permission =
      superuser.getPermission(SuperUserPermission.class);
    if (!superuser.isAdmin() && permission == null) {
      throw new PermissionException("O usurio '" + superuser.getLogin()
        + "' no tem permisso de super-usurio");
    }
    User delegatedUser = readUser(delegatedLogin);

    if (delegatedUser == null) {
      String msg = "Login de usurio inexistente: '" + delegatedLogin + "'";

      throw new DAOException(msg, null);
    }
    if (delegatedUser.isAdmin()) {
      throw new PermissionException(
        "O Administrador no pode ser delegado por um super-usurio");
    }
    if (!superuser.isAdmin()) {
      Map roleMap = permission.getAttributes();
      Collection roleKeySet = roleMap.keySet();
      boolean hasRole = false;
      for (Iterator i = roleKeySet.iterator(); i.hasNext();) {
        Collection roleIds = (Collection) roleMap.get(i.next());
        for (Iterator j = roleIds.iterator(); j.hasNext();) {
          Object id = j.next();
          if (delegatedUser.getRole(id) != null) {
            hasRole = true;
            break;
          }
        }
      }
      if (!hasRole) {
        throw new PermissionException(
          "O usurio '"
            + delegatedLogin
            + "' no tem o perfil necessrio para autenticao por delegao de '"
            + superuser.getLogin() + "'");
      }
    }
    return delegatedUser;
  }

}
