/**
 * $Id: CommandPersistenceCleanupThread.java 116325 2011-03-18 03:17:58Z cassino
 * $
 */
package csbase.server.services.commandpersistenceservice;

import java.io.File;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Set;

import csbase.logic.ClientProjectFile;
import csbase.logic.UserProjectInfo;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.projectservice.ProjectService;

/**
 * Thread para efetuar a "limpeza" de arquivos associados  execuo de comandos
 * que tenham sido modificados h mais do que um nmero especfico de dias.
 * <p>
 * Esta thread atua apenas uma vez por dia, em horrio determinado pelo
 * administrador via propriedades.
 * <p>
 * As seguintes propriedades controlam o funcionamento desta thread:
 * <ul>
 * <li> <code>SGAService.cleanup.maxDays</code> nmero mximo de dias para manter
 * os arquivos. Arquivos modificados pela ltima vez h mais tempo sero
 * removidos. Default = 0 = "pra sempre" (arquivos no sero removidos)
 * <li> <code>SGAService.cleanup.hour</code> hora para execuo da limpeza
 * peridica, no formato 24h (0-23h). Default = 0h. A limpeza ser executada na
 * hora "cheia", i.e. 0min.
 * </ul>
 * 
 * @see #EXPIRATION_MAX_DAYS
 * @see #CLEANUP_HOUR_PROPERTY
 * 
 * @author Tecgraf
 */
class CommandPersistenceCleanupThread extends Thread {

  /**
   * Hora para execuo da limpeza (default = 0h), no formato 24h (0-23h).
   */
  private static final String CLEANUP_HOUR_PROPERTY = "cleanup.hour";

  /**
   * Nmero mximo de dias para manter os arquivos. Arquivos modificados pela
   * ltima vez h mais tempo sero removidos.
   * <p>
   * dafault = 0 dias = "pra sempre" (os arquivos no sero removidos)
   */
  private static final String EXPIRATION_MAX_DAYS = "cleanup.maxDays";

  /**
   * Nmero de milisegundos em um dia.
   */
  private static final long MILLISECONDS_PER_DAY = 24 * 60 * 60 * 1000L;

  /**
   * Nome do diretrio de persistncia da execuo dos comandos.
   */
  private final String CMDS_DIR_NAME;

  /**
   * Valor da propriedade {@link #EXPIRATION_MAX_DAYS}.
   */
  private int maxDays = 0;

  /**
   * Valor da propriedade {@link #CLEANUP_HOUR_PROPERTY}.
   */
  private int cleanupHour = 0;

  /**
   * Flag usado para definir quando a thread deve ser interrompida.
   */
  private volatile boolean quit = false;

  /**
   * Construtor.
   * 
   * @throws ServerException em caso de erro (propriedades obrigatrias
   *         indefinidas, valores incorretos)
   */
  CommandPersistenceCleanupThread() throws ServerException {
    super(CommandPersistenceCleanupThread.class.getSimpleName());
    CommandPersistenceService cmdPersService =
      CommandPersistenceService.getInstance();
    CMDS_DIR_NAME = cmdPersService.getCommandsDirectoryName();
    maxDays = cmdPersService.getIntProperty(EXPIRATION_MAX_DAYS);
    // DEBUG
    //      maxDays = 1;
    if (maxDays < 0) {
      throw new ServerException(String.format("%s deve ser >= 0 [%d].",
        EXPIRATION_MAX_DAYS, maxDays));
    }
    if (maxDays > 0) {
      cleanupHour = cmdPersService.getIntProperty(CLEANUP_HOUR_PROPERTY);
      // DEBUG
      //        cleanupHour = 0;
      if (cleanupHour < 0 || cleanupHour > 23) {
        throw new ServerException(CLEANUP_HOUR_PROPERTY
          + " deve estar no intervalo [0, 23].");
      }
    }
    else {
      Server.logInfoMessage(String.format(
        "Thread de limpeza desativada (%s.%s = 0)",
        CommandPersistenceService.SERVICE_NAME, EXPIRATION_MAX_DAYS));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void run() {
    if (maxDays == 0) {
      /*
       * no devemos fazer limpeza; a thread nem  iniciada
       */
      return;
    }
    Server.logInfoMessage("Thread de limpeza iniciada.");
    Server
      .logInfoMessage(String.format("Hora para cleanup: %dh.", cleanupHour));
    Server.logInfoMessage("Comandos com mais de " + maxDays
      + " dias sero removidos.");
    doCleanup();
  }

  /**
   * 
   */
  private void doCleanup() {
    try {
      while (!quit) {
        synchronized (this) {
          wait(getWaitDelta());
          removeExpiredFiles();
        }
      }
    }
    catch (InterruptedException e) {
      Server.logInfoMessage("Thread de limpeza interrompida.");
    }
    Server.logInfoMessage("Thread de limpeza encerrada.");
  }

  /**
   * Remove arquivos expirados.
   * 
   * @see #maxDays
   */
  private void removeExpiredFiles() {
    Server.logInfoMessage("Iniciando limpeza de comandos expirados.");
    ProjectService projectService = ProjectService.getInstance();
    AdministrationService adminService = AdministrationService.getInstance();
    Set<Object> usersId = adminService.getAllUserIds();
    String[] filePath = new String[] { CMDS_DIR_NAME };
    try {
      for (Object userId : usersId) {
        Service.setUserId(userId);
        List<UserProjectInfo> projects = projectService.getProjectsFromUser(userId);
        for (UserProjectInfo project : projects) {
          Object projectId = project.getProjectId();
          cleanupCmdsDir(projectId, filePath, userId);
        }
      }
    }
    finally {
      Service.setUserId(null);
    }
    Server.logInfoMessage("Limpeza de comandos expirados terminada.");
  }

  /**
   * Executa limpeza em um diretrio <code>.cmds</code>. Como este diretrio
   * possui apenas subdiretrios no formato
   * 
   * <pre>
   * admi_proj_MAASdEUCUp
   * admi_proj_MAASdEURWc
   * ...
   * </pre>
   * 
   * e no faz sentido removermos apenas parte destes diretrios, vamos
   * trat-los de forma atmica, i.e. consultaremos apenas a data da ltima
   * modificao do diretrio. Esta  atualizada sempre que um arquivo (ou
   * diretrio)  criado ou removido; ela <b>no</b>  atualizada quando um
   * arquivo  atualizado ou quando um arquivo  criado em um subdiretrio, mas
   * estes casos sero desconsiderados.
   * 
   * @param projectId Identificador do projeto.
   * @param cmdsPath Caminho para o diretrio de persistncia de comandos.
   * @param userId Identificador do usurio dono do projeto.
   */
  private void cleanupCmdsDir(Object projectId, String[] cmdsPath, Object userId) {
    ProjectService projectService = ProjectService.getInstance();
    if (!projectService.existsFile(projectId, cmdsPath)) {
      return;
    }
    long maxMillis = maxDays * MILLISECONDS_PER_DAY;
    long expirationDate = Calendar.getInstance().getTimeInMillis() - maxMillis;
    ClientProjectFile[] cmds = projectService.getChildren(projectId, cmdsPath);
    for (ClientProjectFile cmd : cmds) {
      if (cmd.getModificationDate() < expirationDate) {
        projectService.removeFile(projectId, cmd.getPath());
        Server.logInfoMessage("Diretrio " + cmd.getName() + " do usurio "
          + userId + " removido.");
      }
    }
  }

  /**
   * Encerra a thread.
   */
  synchronized void shutdown() {
    quit = true;
    Server.logInfoMessage("Shutdown da thread de limpeza solicitado.");
    interrupt();
  }

  /**
   * @return quantidade de milissegundos da hora corrente at o prximo cleanup
   */
  private long getWaitDelta() {
    Calendar now = Calendar.getInstance();
    Calendar cleanupTime = Calendar.getInstance();
    cleanupTime.set(Calendar.HOUR_OF_DAY, cleanupHour);
    cleanupTime.set(Calendar.MINUTE, 0);
    cleanupTime.set(Calendar.SECOND, 0);
    if (now.after(cleanupTime)) {
      int day = cleanupTime.get(Calendar.DAY_OF_MONTH);
      cleanupTime.set(Calendar.DAY_OF_MONTH, day + 1);
    }
    // DEBUG
    //    return 10 * 1000; // 10s
    return cleanupTime.getTimeInMillis() - now.getTimeInMillis();
  }

  /**
   * Mtodo para testes.
   * 
   * @param args
   */
  public static void main(String[] args) {
    File dir = new File("/tmp/sgacleanup");
    Date lastModified = new Date(dir.lastModified());
    System.out.println(lastModified.toString());
  }
}
