package csbase.server.services.diagnosticservice.monitors.openbus;

import java.util.Locale;

import scs.core.ComponentId;
import scs.core.IComponent;
import csbase.exception.OperationFailureException;
import csbase.logic.diagnosticservice.StatusCode;
import csbase.logic.diagnosticservice.ValidationStatus;
import csbase.server.services.diagnosticservice.MessageTranslator;
import csbase.server.services.diagnosticservice.monitors.Validation;
import csbase.server.services.openbusservice.OpenBusService;

/**
 * Validao da oferta no barramento procurando por seu identificador.
 */
public class OfferIdValidation implements Validation {
  /** Nome que identifica a validao */
  public static final String NAME = "OfferIdValidation";

  /**
   * Componente monitorado.
   */
  private ComponentId component;

  /** Identificador da oferta */
  private String offerId;

  /**
   * Construtor.
   * 
   * @param component o componente
   */
  public OfferIdValidation(ComponentId component) {
    this.component = component;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValidationStatus check(Locale locale) {
    this.offerId =
      OpenBusService.getInstance().getRegisteredComponentBusId(component);
    boolean isPublished = false;
    if ((offerId != null)) {
      isPublished = checkByOfferId();
    }
    if (isPublished) {
      return new ValidationStatus(StatusCode.OK, MessageTranslator.getString(
        "server.offermonitor.no.offer.with.id.error", new String[] { offerId },
        locale));
    }
    return new ValidationStatus(StatusCode.OK, MessageTranslator.getString(
      "server.offermonitor.offer.with.id.success", new String[] { offerId },
      locale));
  }

  /**
   * Verifica se existe uma oferta publicada com um determinado identificador.
   * 
   * @return {@code true} se existe uma oferta publicado com um determinado
   *         identificador ou {@code false} caso contrrio.
   */
  private boolean checkByOfferId() {
    IComponent service;
    try {
      service = OpenBusService.getInstance().findServiceById(offerId);
    }
    catch (OperationFailureException e) {
      return false;
    }

    return service != null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String id() {
    return NAME;
  }

  /**
   * {@inheritDoc}
   * 
   * Depende do servio de registro estar disponvel.
   */
  @Override
  public String[] requires() {
    return new String[] { RegistryServiceValidation.NAME };
  }
}