package csbase.server.services.schedulerservice;

import java.io.Serializable;
import java.util.Comparator;
import java.util.Hashtable;

import csbase.logic.CapacityType;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;

/**
 * Comparador dos SGAs. Calcula o melhor SGA disponvel considerando os
 * critrios de leitura/escrita em disco e taxa de transferncia na rede.
 */
public class SGASetIOComparator implements Comparator<SGASet>, Serializable {
  private long inputFilesLength;
  Hashtable<String, Long> platfBinFilesLengthList;

  /**
   * Construtor do comparador.
   * 
   * @param inputFilesLength Tamanho total dos arquivos de entrada do algoritmo
   *        a ser executado.
   * @param platfBinFilesLengthList Hashtable com os tamanhos dos arquivos
   *        binrios dos algoritmos listados por tipo de plataforma.
   */
  public SGASetIOComparator(long inputFilesLength,
    Hashtable<String, Long> platfBinFilesLengthList) {
    this.inputFilesLength = inputFilesLength;
    this.platfBinFilesLengthList = platfBinFilesLengthList;
  }

  /**
   * Compara as capacidades de dois SGAs.
   * 
   * @param sga1
   * @param sga2
   * @return -1 quando a capacidade de E/S do SGA1 for maior que a capacidade do
   *         SGA2. 0 quando a capacidade de E/S do SGA1 for igual  capacidade
   *         do SGA2. 1 quando a capacidade de E/S do SGA1 for menor que a
   *         capacidade do SGA2.
   */
  public int compare(SGASet sga1, SGASet sga2) {
    long totalFilesLength1 =
      inputFilesLength + platfBinFilesLengthList.get(sga1.getName());
    long totalFilesLength2 =
      inputFilesLength + platfBinFilesLengthList.get(sga2.getName());

    float transferTime1 = 0;
    float transferTime2 = 0;

    long sgaWriting1 = sga1.getCapacity(CapacityType.DISK_WRITE);
    long sgaWriting2 = sga2.getCapacity(CapacityType.DISK_WRITE);

    if (sgaWriting1 == SGAInfo.NO_CAPACITY)
      sgaWriting1 = 1;
    if (sgaWriting2 == SGAInfo.NO_CAPACITY)
      sgaWriting2 = 1;

    long sgaTransfer1 = sga1.getCapacity(CapacityType.NET);
    long sgaTransfer2 = sga2.getCapacity(CapacityType.NET);

    if (sgaTransfer1 == SGAInfo.NO_CAPACITY)
      transferTime1 = totalFilesLength1;
    else if (sgaTransfer1 == SGAInfo.ALL_CAPACITY)
      transferTime1 = 0;
    else
      transferTime1 = totalFilesLength1 / sgaTransfer1;

    if (sgaTransfer2 == SGAInfo.NO_CAPACITY)
      transferTime2 = totalFilesLength2;
    else if (sgaTransfer2 == SGAInfo.ALL_CAPACITY)
      transferTime2 = 0;
    else
      transferTime2 = totalFilesLength2 / sgaTransfer2;

    float execTime1 = (totalFilesLength1 / sgaWriting1) + transferTime1;
    float execTime2 = (totalFilesLength2 / sgaWriting2) + transferTime2;

    if (execTime1 < execTime2) {
      return -1;
    }
    if (execTime1 > execTime2) {
      return 1;
    }
    return 0;
  }
}