package csbase.server.services.sgaservice;

import org.omg.CORBA.IntHolder;

import sgaidl.CompletedCommandInfo;
import sgaidl.RetrievedInfo;
import sgaidl.SGACommand;
import sgaidl.SGADynamicInfo;
import sgaidl.SGAManagerPOA;
import sgaidl.SGAServer;
import sgaidl.StaticNodeInfo;
import csbase.logic.CommandInfo;
import csbase.server.Service;

/**
 * A classe <code>SGAHandler</code> implementa o <i>servant</i> associado 
 * interface idl <code>SGAManager</code>, atravs da qual os SGAs interagem
 * com o servio de gerncia de SGAs. As requisies recebidas dos SGAs so
 * repassadas pelo SGAHandler  implementao do servio de gerncia de SGAs
 * (SGAService).
 */
public class SGAHandler extends SGAManagerPOA {
  /** Referncia para a implementao do servio SGAService */
  private SGAService service = null;

  /**
   * SGA avisa ao Manager que est ativo.
   * 
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   * @param staticNodesInfo informaes estticas dos ns
   * @param updateInterval informa de quanto em quanto tempo o SGA deve dizer
   *        que est vivo.
   * 
   * @return true se o SGA foi registrado com sucesso junto ao SSI
   */
  @Override
  public boolean registerSGA(SGAServer sga, String sganame,
    StaticNodeInfo[] staticNodesInfo, IntHolder updateInterval) {
    return service.registerSGA(sga, sganame, staticNodesInfo, updateInterval);
  }

  /**
   * SGA avisa ao Manager que sair do ar.
   * 
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   */

  @Override
  public void unregisterSGA(SGAServer sga, String sganame) {
    service.unregisterSGA(sga, sganame);
  }

  /**
   * SGA avisa ao Manager que est apto a receber novos comandos.
   * 
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   */
  @Override
  public void setSGAEnabled(SGAServer sga, String sganame) {
    service.setSGAEnabled(sganame);
  }

  /**
   * SGA avisa ao Manager que no est apto a receber novos comandos, apesar de
   * continuar ativo.
   * 
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   */
  @Override
  public void setSGADisabled(SGAServer sga, String sganame) {
    service.setSGADisabled(sganame);
  }

  /**
   * SGA avisa ao Manager que um comando terminou
   * 
   * @param sgaName nome do SGA
   * @param cmd referncia para o gerenciador do comando
   * @param cmdId identificador para o comando
   * @param info informaes sobre o trmino de execuo do comando
   * @param cmdString .
   * 
   * @return <code>true</code> se a notificao foi feita, <code>false</code>
   *         se o SSI desconhece o SGA
   */
  @Override
  public boolean commandCompleted(String sgaName, SGACommand cmd, String cmdId,
    CompletedCommandInfo info, String cmdString) {
    try {
      CommandInfo command = service.getSGACommand(sgaName, cmdId);
      if (command != null) {
        Object userId = command.getUserId();
        Service.setUserId(userId);
      }
      return service.commandCompleted(sgaName, cmd, cmdId, info);
    }
    finally {
      Service.setUserId(null);
    }
  }

  /**
   * SGA avisa ao Manager que um comando no foi recuperado
   * 
   * @param sgaName nome do SGA
   * @param cmdId identificador do comando
   * @param cmdString .
   * 
   * @return <code>true</code> se a notificao foi feita, <code>false</code>
   *         se o SSI desconhece o SGA
   */
  @Override
  public boolean commandLost(String sgaName, String cmdId, String cmdString) {
    try {
      CommandInfo command = service.getSGACommand(sgaName, cmdId);
      if (command != null) {
        Object userId = command.getUserId();
        Service.setUserId(userId);
      }
      return service.commandLost(sgaName, cmdId);
    }
    finally {
      Service.setUserId(null);
    }
  }

  /**
   * SGA avisa ao Manager que comandos foram recuperados.
   * 
   * @param sgaName nome do SGA
   * @param cmds comandos recuperados
   * 
   * @return <code>true</code> se os comandos foram registrados com sucesso,
   *         <code>false</code> se o SSI desconhece o SGA
   */
  @Override
  public boolean commandRetrieved(String sgaName, RetrievedInfo[] cmds) {
    /*
     * o mtodo do servio se encarrega de definir o usurio na thread para cada
     * comando
     */
    return service.commandRetrieved(sgaName, cmds);
  }

  /**
   * SGA verifica se o Manager est ativo
   * 
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   * 
   * @return true
   */

  @Override
  public boolean isRegistered(SGAServer sga, String sganame) {
    return service.isRegistered(sga, sganame);
  }

  /**
   * Atualiza os dados de um SGA no Manager.
   * 
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   * @param dynamicInfo informaes dinmicas do SGA
   * 
   * @return <code>true</code> se os dados foram atualizados com sucesso
   */

  @Override
  public boolean updateSGAInfo(SGAServer sga, String sganame,
    SGADynamicInfo dynamicInfo) {
    return service.updateSGAInfo(sga, sganame, dynamicInfo);
  }

  /**
   * Constri um servant para a interface SGAManager
   * 
   * @param srv referncia para a implementao do servio SGAService
   */
  public SGAHandler(SGAService srv) {
    service = srv;
  }
}
