/*
 * $Id$
 */

package csbase.server.keystore;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.text.MessageFormat;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import csbase.exception.OperationFailureException;
import csbase.server.Server;

/**
 * Representa um repositrio de chaves.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class CSKeyStore {
  /**
   * O algoritmo utilizado para gerar ou verificar uma assinatura digital.
   */
  private static final String SIGNATURE_ALGORITHM = "SHA1withDSA";

  /**
   * O tipo do repositrio de chaves.
   */
  private static final String KEYSTORE_TYPE = "JKS";

  /**
   * O repositrio de chaves propriamente dito.
   */
  private KeyStore keyStore;

  /**
   * A instncia nica do repositrio de chaves.
   */
  private static CSKeyStore instance;

  /**
   * A senha do repositrio de chaves.
   */
  private char[] password;

  /**
   * A data da ltima atualizao do repositrio em relao ao arquivo do
   * repositrio de chaves.
   */
  private long lastSynchronizationDate;

  /**
   * O arquivo do repositrio de chaves.
   */
  private File keyStoreFile;

  /**
   * Representa um controle de acesso ao repositrio de chaves.
   */
  private ReadWriteLock keyStoreLock;

  /**
   * Cria um repositrio de chaves.
   * 
   * @param pathName A localizao do arquivo que contm o repositrio de
   *        chaves.
   * @param password A senha do repositrio de chaves.
   * 
   * @throws IllegalArgumentException Caso a localizao do arquivo seja nula.
   */
  private CSKeyStore(String pathName, char[] password) {
    if (pathName == null) {
      final String err = "A localizao do arquivo do rep. de chaves est null";
      throw new IllegalArgumentException(err);
    }
    this.password = password;
    this.keyStoreFile = new File(pathName);
    this.resetKeyStoreData();
    this.keyStoreLock = new ReentrantReadWriteLock();
  }

  /**
   * <p>
   * Sincroniza o repositrio de chaves/certificados em relao ao arquivo do
   * repositrio.
   * </p>
   * <p>
   * Atualiza o estado do repositrio de chaves.
   * </p>
   */
  private synchronized void synchronize() {
    this.keyStoreLock.writeLock().lock();
    try {
      this.loadKeyStore();
    }
    finally {
      this.keyStoreLock.writeLock().unlock();
    }
  }

  /**
   * Carrega o repositrio de chaves a partir do seu arquivo.
   */
  private void loadKeyStore() {
    if (this.lastSynchronizationDate == this.keyStoreFile.lastModified()) {
      return;
    }
    this.resetKeyStoreData();
    InputStream fileInputStream;
    try {
      fileInputStream = new FileInputStream(this.keyStoreFile);
    }
    catch (FileNotFoundException e) {
      final String fmt =
        "O arquivo do rep. de chaves no existe. Caminho: {0}.";
      final String keyStoreFileName = this.keyStoreFile.getName();
      final String err = MessageFormat.format(fmt, keyStoreFileName);
      Server.logSevereMessage(err, e);
      return;
    }
    KeyStore keyStoreTemp;
    try {
      try {
        keyStoreTemp = KeyStore.getInstance(KEYSTORE_TYPE);
      }
      catch (KeyStoreException e) {
        final String fmt = "Tipo para chaves/certificados {0} no existe.";
        final String err = MessageFormat.format(fmt, KEYSTORE_TYPE);
        Server.logSevereMessage(err, e);
        return;
      }
      final String err = "Falha no arquivo do rep. de chaves/certificados.";
      try {
        keyStoreTemp.load(fileInputStream, this.password);
      }
      catch (IOException e) {
        Server.logSevereMessage(err, e);
        return;
      }
      catch (GeneralSecurityException e) {
        Server.logSevereMessage(err, e);
        return;
      }
    }
    finally {
      try {
        fileInputStream.close();
      }
      catch (IOException e) {
        e.printStackTrace();
        final String err = "Falha ao fechar rep. de chaves/certificados.";
        Server.logSevereMessage(err, e);
      }
    }
    this.keyStore = keyStoreTemp;
    this.lastSynchronizationDate = this.keyStoreFile.lastModified();
  }

  /**
   * Restaura os dados sobre o repositrio de chaves/certificados aos seus
   * valores iniciais.
   */
  private void resetKeyStoreData() {
    this.keyStore = null;
    this.lastSynchronizationDate = -1;
  }

  /**
   * Assina um dado em nome de uma entidade.
   * 
   * @param alias O nome da entidade.
   * @param privateKeyPassword A senha da chave privada da entidade.
   * @param data O dado que ser assinado.
   * 
   * @return o dado assinado ou null, caso a entidade no esteja cadastrada no
   *         repositrio de chaves.
   * 
   * @throws OperationFailureException Caso ocorra alguma falha na operao.
   * @throws IllegalArgumentException Caso algum dos parmetros recebidos seja
   *         nulo.
   */
  public byte[] sign(String alias, String privateKeyPassword, String data)
    throws OperationFailureException {
    if (alias == null) {
      final String err = "O nome da entidade no pode ser nulo.";
      throw new IllegalArgumentException(err);
    }
    if (privateKeyPassword == null) {
      final String err =
        "A senha da chave privada da entidade no pode ser nula.";
      throw new IllegalArgumentException(err);
    }
    if (data == null) {
      final String err = "O dado a ser assinado no pode ser nulo.";
      throw new IllegalArgumentException(err);
    }
    PrivateKey privateKey = this.getKey(alias, privateKeyPassword);
    if (privateKey == null) {
      return null;
    }
    try {
      Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
      signature.initSign(privateKey);
      signature.update(data.getBytes());
      return signature.sign();
    }
    catch (GeneralSecurityException e) {
      throw new OperationFailureException(e);
    }
  }

  /**
   * Obtm uma chave de uma determinada entidade.
   * 
   * @param alias O nome da entidade.
   * @param privateKeyPassword A senha para acesso  chave privada.
   * 
   * @return A chave ou null, caso no exista.
   * 
   * @throws OperationFailureException Caso ocorra alguma falha na operao.
   */
  private PrivateKey getKey(String alias, String privateKeyPassword)
    throws OperationFailureException {
    this.synchronize();
    this.keyStoreLock.readLock().lock();
    try {
      if (this.keyStore == null) {
        return null;
      }
      return (PrivateKey) this.keyStore.getKey(alias,
        privateKeyPassword.toCharArray());
    }
    catch (GeneralSecurityException e) {
      throw new OperationFailureException(e);
    }
    finally {
      this.keyStoreLock.readLock().unlock();
    }
  }

  /**
   * Verifica se a assinatura de um determinado dado foi feita pela entidade.
   * 
   * @param alias O nome da entidade.
   * @param data O dado que foi assinado.
   * @param signedData A assinatura do dado.
   * 
   * @return true, caso a assinatura seja vlida, ou false, caso contrrio.
   * 
   * @throws OperationFailureException Caso ocorra alguma falha na operao.
   */
  public boolean verify(String alias, String data, byte[] signedData)
    throws OperationFailureException {
    Certificate certificate = this.getCertificate(alias);
    if (certificate == null) {
      return false;
    }
    try {
      Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
      signature.initVerify(certificate);
      signature.update(data.getBytes());
      return signature.verify(signedData);
    }
    catch (GeneralSecurityException e) {
      throw new OperationFailureException(e);
    }
  }

  /**
   * Verifica se uma determinada entidade existe no repositrio.
   * 
   * @param alias O nome da entidade.
   * 
   * @return true, caso a entidade exista, ou false,caso contrrio.
   * 
   * @throws OperationFailureException Caso ocorra alguma falha na operao.
   */
  public boolean containsAlias(String alias) throws OperationFailureException {
    this.synchronize();
    this.keyStoreLock.readLock().lock();
    try {
      if (this.keyStore == null) {
        return false;
      }
      return this.keyStore.containsAlias(alias);
    }
    catch (KeyStoreException e) {
      throw new OperationFailureException(e);
    }
    finally {
      this.keyStoreLock.readLock().unlock();
    }
  }

  /**
   * Obtm um certificado de uma determinada entidade.
   * 
   * @param alias O nome da entidade.
   * 
   * @return O certificado ou null, caso no exista.
   * 
   * @throws OperationFailureException Caso ocorra alguma falha na operao.
   */
  public Certificate getCertificate(String alias)
    throws OperationFailureException {
    this.synchronize();
    this.keyStoreLock.readLock().lock();
    try {
      if (this.keyStore == null) {
        return null;
      }
      return this.keyStore.getCertificate(alias);
    }
    catch (KeyStoreException e) {
      throw new OperationFailureException(e);
    }
    finally {
      this.keyStoreLock.readLock().unlock();
    }
  }

  /**
   * Cria a instncia nica do repositrio de chaves.
   * 
   * @param pathName A localizao do arquivo que contm o repositrio de
   *        chaves.
   * 
   * @throws IllegalStateException Caso j exista uma instncia criada.
   * 
   * @see #getInstance()
   */
  public static void createInstance(String pathName) {
    if (instance != null) {
      final String err = "J existe instncia de rep. de chaves/certificados!";
      throw new IllegalStateException(err);
    }
    instance = new CSKeyStore(pathName, null);
  }

  /**
   * Cria a instncia nica do repositrio de chaves.
   * 
   * @param pathName A localizao do arquivo que contm o repositrio de
   *        chaves.
   * @param password A senha do repositrio de chaves.
   * 
   * @throws IllegalArgumentException Caso a senha seja nula.
   * @throws IllegalStateException Caso j exista uma instncia criada.
   * 
   * @see #getInstance()
   */
  public static void createInstance(String pathName, String password) {
    if (password == null) {
      final String err = "No  permitido criar uma instncia com senha nula.";
      throw new IllegalArgumentException(err);
    }
    if (instance != null) {
      final String err = "J existe instncia de rep. de chaves/certificados.";
      throw new IllegalStateException(err);
    }
    instance = new CSKeyStore(pathName, password.toCharArray());
  }

  /**
   * Obtm a instncia nica do repositrio de chaves.
   * 
   * @return A instncia nica ou null, caso esta ainda no tenha sido criada.
   * 
   * @see #createInstance(String)
   * @see #createInstance(String, String)
   */
  public static CSKeyStore getInstance() {
    return instance;
  }

  /**
   * Adiciona um certificado. Se a entidade j possuir um certificado o mesmo
   * ser sobreescrito.
   * 
   * @param alias O nome da entidade
   * @param cert A instncia do certificado
   * @return indicativo de sucesso.
   */
  public boolean addCertificate(String alias, Certificate cert) {
    if (alias == null || cert == null) {
      throw new IllegalArgumentException("alias == null || cert == null");
    }
    try {
      this.keyStoreLock.writeLock().lock();
      if (this.keyStore == null) {
        return false;
      }
      this.keyStore.setCertificateEntry(alias, cert);
      this.writeKeyStore();
      final boolean containsFlag = this.keyStore.containsAlias(alias);
      return containsFlag;
    }
    catch (KeyStoreException e) {
      final String err = "No foi possvel adicionar o cert. para a entidade:";
      Server.logSevereMessage(err + alias, e);
    }
    finally {
      this.keyStoreLock.writeLock().unlock();
    }
    return false;
  }

  /**
   * Salva o keystore em disco
   */
  private void writeKeyStore() {
    try {
      this.keyStoreLock.writeLock().lock();
      OutputStream out = new FileOutputStream(this.keyStoreFile);
      this.keyStore.store(out, this.password);
      out.close();
    }
    catch (FileNotFoundException e) {
      final String fmt = "Arquivo do rep. de chaves no existe. Caminho: {0}.";
      final String keyStoreFileName = this.keyStoreFile.getName();
      final String err = MessageFormat.format(fmt, keyStoreFileName);
      Server.logSevereMessage(err, e);
    }
    catch (KeyStoreException e) {
      final String err = "KeyStore no inicializado.";
      Server.logSevereMessage(err, e);
    }
    catch (NoSuchAlgorithmException e) {
      final String err = "Algoritmo de verificao desconhecido";
      Server.logSevereMessage(err, e);
    }
    catch (CertificateException e) {
      final String err = "Erro salvando certificados";
      Server.logSevereMessage(err, e);
    }
    catch (IOException e) {
      final String err = "Erro lendo " + this.keyStoreFile;
      Server.logSevereMessage(err, e);
    }
    finally {
      this.keyStoreLock.writeLock().unlock();
    }
  }

  /**
   * Remove o certificado de uma entidade
   * 
   * @param alias O nome da entidade
   */
  public void removeCertificate(String alias) {
    if (alias == null) {
      throw new IllegalArgumentException("alias == null");
    }

    try {
      this.keyStoreLock.writeLock().lock();
      if (this.keyStore == null) {
        final String err = "No existe keystore para remover o certificado de:";
        Server.logSevereMessage(err + alias);
        return;
      }
      if (this.keyStore.containsAlias(alias)) {
        this.keyStore.deleteEntry(alias);
        this.writeKeyStore();
      }
    }
    catch (KeyStoreException e) {
      final String err = "No foi possvel remover o certificado para: ";
      Server.logSevereMessage(err + alias, e);
    }
    finally {
      this.keyStoreLock.writeLock().unlock();
    }
  }
}
