package csbase.server.services;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

import csbase.exception.BugException;
import csbase.exception.CSBaseRuntimeException;
import csbase.exception.InfoException;
import csbase.exception.ServiceFailureException;
import csbase.logic.User;
import csbase.server.Server;
import csbase.server.Service;

/**
 * Responsvel por tratar chamadas a mtodos dos servios.<br>
 * Suas responsabilidades so:
 * <ul>
 * <li>
 * Atribuir a chave da sesso do usurio como sesso corrente antes de cada
 * chamada a um mtodo de servio e desatribuir depois da chamada.</li>
 * <li>
 * Delega para o prximo {@link InvocationHandler} a execuo do mtodo do
 * servio.</li>
 * <li>
 * Capturar e repassar as excees lanadas apenas com a mensagem de erro,
 * evitando repassar informaes da pilha para o cliente.</li>
 * </ul>
 * 
 * @author Tecgraf
 */
public class ServiceInvocationHandler implements InvocationHandler {

  /**
   * Chave da sesso do usurio utilizada para identificar o usurio que fez as
   * chamadas aos mtodos do servio.
   */
  private final Object sessionKey;

  /**
   * InvocationHandler responsvel pela execuo do mtodo do servio.
   */
  private final InvocationHandler next;

  /**
   * Construtor.
   * 
   * @param sessionKey Chave da sesso do usurio utilizada para identificar o
   *        usurio que fez as chamadas aos mtodos do servio.
   * @param next Responsvel por tratar as chamadas aos mtodos do objeto, obj.
   */
  public ServiceInvocationHandler(Object sessionKey, InvocationHandler next) {
    this.sessionKey = sessionKey;
    this.next = next;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object invoke(Object proxy, Method method, Object[] args)
    throws Throwable {

    try {
      // Loga em verbose a chamada feita.
      Service.setKey(this.sessionKey);
      return next.invoke(proxy, method, args);
    }
    catch (ServiceFailureException fe) {

      String sid = Service.getSystemId();
      if (sid == null)
        sid = "null";
      Server.logSevereMessage(String.format(
        "Falha em pedido de mtodo %s do servio %s.", method, sid));

      logException(fe);
      throw new ServiceFailureException(fe.getClientMessage());
    }
    catch (InfoException ie) {
      throw ie;
    }
    catch (CSBaseRuntimeException re) {
      throw re;
    }
    catch (Throwable t) {
      logException(t);
      throw new BugException();
    }
    finally {
      Service.setKey(null);
    }
  }

  /**
   * Registro de falhas detetadas durante a execuo de um pedido de servio.
   * 
   * @param throwable a exceo que representa a falha
   */
  private void logException(Throwable throwable) {
    final User user = Service.getUser();
    Object uid = "null";
    if (user != null) {
      uid = user.getId();
    }
    final String msg = String.format("Falha em pedido de servio para usurio %s", uid);
    Server.logSevereMessage(msg, throwable);
  }
}
