package csbase.server.services.diagnosticservice.monitors.openbus;

import java.util.Locale;
import java.util.Properties;

import scs.core.ComponentId;
import scs.core.IComponent;
import csbase.exception.OperationFailureException;
import csbase.logic.diagnosticservice.StatusCode;
import csbase.logic.diagnosticservice.ValidationStatus;
import csbase.server.services.diagnosticservice.MessageTranslator;
import csbase.server.services.diagnosticservice.monitors.Validation;
import csbase.server.services.openbusservice.OpenBusService;

/**
 * Verifica se foram encontradas mais de uma oferta publicada com as facetas e
 * as propriedades especificadas.
 * 
 */
public class DuplicatedOfferValidation implements Validation {
  /** Nome que identifica a validao */
  public static final String NAME = "DuplicatedOfferValidation";
  /**
   * Componente monitorado.
   */
  private ComponentId component;

  /**
   * Construtor.
   * 
   * @param component o componente monitorado
   */
  public DuplicatedOfferValidation(ComponentId component) {
    this.component = component;
  }

  /**
   * {@inheritDoc}
   * 
   * Verifica se for encontrada apenas uma oferta publicada com as facetas e as
   * propriedades procuradas.
   */
  @Override
  public ValidationStatus check(Locale locale) {
    String[] facets =
      OpenBusService.getInstance().getRegisteredComponentFacets(component);
    Properties properties =
      OpenBusService.getInstance().getRegisteredComponentProperties(component);

    String entity = OpenBusService.getInstance().getEntityName();
    properties.put("openbus.offer.entity", entity);

    for (String facet : facets) {
      properties.put("openbus.component.interface", facet);
    }

    IComponent[] services;
    try {
      services = OpenBusService.getInstance().findServiceList(properties);
    }
    catch (OperationFailureException e) {
      return new ValidationStatus(StatusCode.ERROR, MessageTranslator
        .getString("server.offermonitor.no.one.error",
          new String[] { properties.toString() }, locale));
    }

    int num = services.length;
    if (num == 1) {
      return new ValidationStatus(StatusCode.OK, MessageTranslator.getString(
        "server.offermonitor.just.one.success", new String[] { properties
          .toString() }, locale));
    }
    if (num >= 1) {
      return new ValidationStatus(StatusCode.ERROR, MessageTranslator
        .getString("server.offermonitor.more.than.one.error", new String[] {
            String.valueOf(num), properties.toString() }, locale));
    }
    return new ValidationStatus(StatusCode.ERROR, MessageTranslator.getString(
      "server.offermonitor.no.one.error",
      new String[] { properties.toString() }, locale));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String id() {
    return NAME;
  }

  /**
   * {@inheritDoc}
   * 
   * Depende se o servio de registro est disponvel.
   */
  @Override
  public String[] requires() {
    return new String[] { RegistryServiceValidation.NAME };
  }
}
