/*
 * $Id: ServerSideProperties.java 150380 2014-02-26 17:03:42Z oikawa $
 */

package csbase.server;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import tecgraf.javautils.core.properties.PropertiesUtils;
import tecgraf.javautils.core.properties.PropertyException;
import csbase.exception.ServiceFailureException;

/**
 * Conjunto de propriedades de um servidor CSBASE.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ServerSideProperties {

  /**
   * Identificador usado quando as propriedades no foram associadas a um
   * arquivo (construtor vazio).
   */
  private static final String NO_PROP_FILE = "_empty_";

  /**
   * Armazenador de propriedades.
   */
  final private Properties properties = new Properties();

  /**
   * Indicativo de origem de carga de propriedades.
   */
  final private String fileName;

  /**
   * Flag de carregamento de propriedade.
   */
  private boolean isLoaded = false;

  /**
   * Strings convencionanda spara mencionar que a propriedade no  usada no
   * sistema; embora tenha que ser explicitamente ajustada.
   */
  static final private List<String> NOT_USED = Arrays.asList("NULL");

  /**
   * Verifica se uma propriedade foi definida com valor no-nulo ou sinalizada
   * como no usada no sistema.
   * 
   * @param value o valor da propriedade
   * @return indicativo
   * @see ServerSideProperties#NOT_USED
   */
  final static public boolean isPropertyValueNull(final String value) {
    return NOT_USED.contains(value.trim());
  }

  /**
   * Obtm um conjunto de propriedades com base em um arquivo definido pela
   * chave.
   * 
   * @param key a identificao da propriedade
   * @return o o conjunto da propriedade (que pode ser vazia, caso o arquivo no
   *         exista ou no esteja especificado).
   */
  protected final Properties getExternalPropertyFile(final String key) {
    final String fileName = getInternalProperty(key);
    if (isPropertyValueNull(fileName)) {
      return new Properties();
    }
    try {
      return PropertiesUtils.loadProperties(fileName);
    }
    catch (Exception e) {
      throw new ServiceFailureException("erro na carga das propriedaedes de "
        + fileName, e);
    }
  }

  /**
   * Obtm uma propriedade do tipo boolean.
   * 
   * @param key a identificao da propriedade
   * @return o valor da propriedade.
   */
  protected final boolean getBooleanProperty(final String key) {
    try {
      return PropertiesUtils.getBoolenValue(properties, key, true);
    }
    catch (PropertyException e) {
      throw new ServiceFailureException(fileName + ": " + e.getMessage());
    }
  }

  /**
   * Obtm uma propriedade do tipo double.
   * 
   * @param key a identificao da propriedade
   * @return o valor da propriedade.
   */
  protected final double getDoubleProperty(final String key) {
    try {
      return PropertiesUtils.getDoubleValue(properties, key);
    }
    catch (PropertyException e) {
      throw new ServiceFailureException(fileName + ": " + e.getMessage());
    }
  }

  /**
   * Consulta efetivamente a chave, levantando exceo se no houver.
   * 
   * @param key chave
   * @return valor
   */
  private String getInternalProperty(final String key) {
    try {
      return PropertiesUtils.getValue(properties, key);
    }
    catch (PropertyException e) {
      throw new ServiceFailureException(fileName + ": " + e.getMessage());
    }
  }

  /**
   * Obtm uma propriedade do tipo int.
   * 
   * @param key a identificao da propriedade
   * @return o valor da propriedade.
   */
  protected final int getIntProperty(final String key) {
    try {
      return PropertiesUtils.getIntValue(properties, key);
    }
    catch (PropertyException e) {
      throw new ServiceFailureException(fileName + ": " + e.getMessage());
    }
  }

  /**
   * Obtm uma propriedade do tipo long.
   * 
   * @param key a identificao da propriedade
   * @return o valor da propriedade.
   */
  protected final long getLongProperty(final String key) {
    try {
      return PropertiesUtils.getLongValue(properties, key);
    }
    catch (PropertyException e) {
      throw new ServiceFailureException(fileName + ": " + e.getMessage());
    }
  }

  /**
   * Obtm uma lista de valores de propriedades.
   * <ul>
   * <li>propriedade.1=ABCD</li>
   * <li>propriedade.2=EFGH</li>
   * </ul>
   * <ul>
   * <li>propriedade.nome.1=ABCD</li>
   * <li>propriedade.nome.2=EFGH</li>
   * </ul>
   * 
   * @param keyPrefix O prefixo-nome da propriedade (nos exemplos acima, seriam
   *        "propriedade" e "propriedade.nome" respectivamente).
   * 
   * @return a lista com todas as propriedades (caso no existam propriedades, a
   *         lista estar vazia).
   */
  final protected List<String> getStringListProperty(final String keyPrefix) {
    return PropertiesUtils.getValuesList(properties, keyPrefix);
  }

  /**
   * Obtm uma propriedade
   * 
   * @param key a identificao da propriedade
   * @return o valor da propriedade como uma string (com <b>trim</b>).
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  protected final String getStringProperty(final String key) {
    return getInternalProperty(key);
  }

  /**
   * Consulta se uma propriedade existe.
   * 
   * @param key a identificao da propriedade
   * @return indicativo
   */
  protected final boolean hasProperty(final String key) {
    final String property = properties.getProperty(key);
    if (property == null) {
      return false;
    }
    return true;
  }

  /**
   * Verifica se uma propriedade foi definida com valor no-nulo ou sinalizada
   * como no usada no sistema.
   * 
   * @param key a identificao da propriedade
   * @return indicativo
   */
  protected final boolean isPropertyNull(final String key) {
    final String value = getInternalProperty(key);
    return isPropertyValueNull(value);
  }

  /**
   * Carregamento das propriedades do servio. As propriedades so carregadas a
   * partir do path fornecido para o construtor. Backreferences so expandidas.
   * 
   * @throws ServerException se houver falha de carga de propriedades no
   *         servidor
   */
  public void load() throws ServerException {
    if (NO_PROP_FILE.equals(fileName)) {
      throw new IllegalStateException("No h arquivo associado");
    }
    if (this.isLoaded) {
      throw new ServerException("Dupla carga de arquivo de propriedade:"
        + fileName);
    }
    try {
      PropertiesUtils.loadProperties(properties, fileName);
      this.isLoaded = true;
    }
    catch (Exception e) {
      throw new ServerException("Erro na carga de propriedades de " + fileName,
        e);
    }
  }

  /**
   * Ajuste de propriedade
   * 
   * @param key chave
   * @param value valor
   */
  void overrideProperty(final String key, final String value) {
    properties.setProperty(key, value);
  }

  /**
   * Construtor
   * 
   * @param name nome.
   */
  public ServerSideProperties(final String name) {
    this.fileName = name;
  }

  /**
   * Construtor
   */
  ServerSideProperties() {
    this.fileName = NO_PROP_FILE;
  }

  /**
   * Retorna um conjunto ordenado com os nomes das propriedades.
   * 
   * @return conjunto ordenado com os nomes das propriedades
   */
  SortedSet<Object> getPropertiesKeys() {
    TreeSet<Object> keys = new TreeSet<Object>(properties.keySet());
    return keys;
  }

  /**
   * Obtm um mapa com as propriedades e seus valores.
   * 
   * @return mapa com as propriedades e seus valores
   */
  Map<String, String> getPropertiesMap() {
    HashMap<String, String> result = new HashMap<String, String>();
    Set<Entry<Object, Object>> entrySet = properties.entrySet();
    for (Entry<Object, Object> entry : entrySet) {
      result.put(entry.getKey().toString(), entry.getValue().toString());
    }
    return result;
  }
}
