package csbase.server.services.schedulerservice;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import csbase.logic.CommandInfo;
import csbase.logic.CapacityType;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;
import csbase.server.services.sgaservice.SGAService;

/**
 * Poltica de escalonamento que privilegia os SGAs com a melhor combinao 
 * entre as capacidades atuais de processamento, leitura/escrita em disco e 
 * transferncia de dados na rede. Os melhores SGAs so definidos com base em 
 * uma equao com variveis alimentadas pelos resultados dos benchmarks 
 * aplicados aos SGAs.  
 * 
 * @author valeria
 *
 */
public class CPUIOCapacityPolicy implements SchedulerPolicyInterface {
  private long minWritingCap;
  private long maxWritingCap;
  private long minSendCap;
  private long maxSendCap;
  private double minProcCap;
  private double maxProcCap;
  private SGAService sgaService;
    
  /**
   * Obtm instncia do SGAService.
   */
  public CPUIOCapacityPolicy() {
    sgaService = SGAService.getInstance();
  }
  
  /**
   * Inicializa os mnimos e mximos das taxas de processamento, leitura/
   * escrita em disco e transferncia em rede dos SGAs.
   * 
   * @param serversNames Servidores disponveis para execuo 
   */
  private void initializeExtremes(List<String> serversNames) {
    long writeCap;
    long cpuCap;
    long transferCap;
    if(serversNames.size() > 0) {
      String sgaName = serversNames.get(0);
      SGASet sga = sgaService.getSGASet(sgaName);
      writeCap = sga.getCapacity(CapacityType.DISK_WRITE);
      if(writeCap == SGAInfo.NO_CAPACITY)
        writeCap = 1; 
      minWritingCap = writeCap;
      maxWritingCap = writeCap;
      cpuCap = sga.getCapacity(CapacityType.CPU);
      if(cpuCap == SGAInfo.NO_CAPACITY)
        cpuCap = 1;
      minProcCap = (1/(sga.getCPULoad1()+1))*cpuCap;
      maxProcCap = (1/(sga.getCPULoad1()+1))*cpuCap;
      // A taxa de transferncia apresenta tratamento diferenciado j
      // que pode assumir dois valores: SGASet.NO_CAPACITY e
      // SGASet.ALL_CAPACITY. 
      transferCap = sga.getCapacity(CapacityType.NET);
      if(transferCap == SGAInfo.NO_CAPACITY) {
        transferCap = 1;
      }
      else {
        if(transferCap == SGAInfo.ALL_CAPACITY) {
          transferCap = Long.MAX_VALUE;
        } 
      }
      minSendCap = transferCap;
      maxSendCap = transferCap;
    }
    for(int i = 1; i < serversNames.size(); i++) {
      SGASet sga = sgaService.getSGASet(serversNames.get(i));
      writeCap = sga.getCapacity(CapacityType.DISK_WRITE);
      if(writeCap == SGAInfo.NO_CAPACITY)
        writeCap = 1;
      if(writeCap < minWritingCap)
        minWritingCap = writeCap;
      else
        if(writeCap > maxWritingCap)
          maxWritingCap = writeCap;
      cpuCap = sga.getCapacity(CapacityType.CPU);
      if(cpuCap == SGAInfo.NO_CAPACITY)
        cpuCap = 1;
      if(((1/(sga.getCPULoad1()+1))*cpuCap) < minProcCap)
        minProcCap = (1/(sga.getCPULoad1()+1))*cpuCap;
      else
        if(((1/(sga.getCPULoad1()+1))*cpuCap) > maxProcCap)
          maxProcCap = (1/(sga.getCPULoad1()+1))*cpuCap;
      transferCap = sga.getCapacity(CapacityType.NET);
      if(transferCap == SGAInfo.NO_CAPACITY)
        transferCap = 1;
      else
        if(transferCap == SGAInfo.ALL_CAPACITY)
          transferCap = Long.MAX_VALUE;
      if(transferCap < minSendCap)
          minSendCap = transferCap;
      else
        if(transferCap > maxSendCap)
          maxSendCap = transferCap;
    }
  }
  
  /**
   * {@inheritDoc}
   */
  public String chooseServer(CommandInfo command, 
                             List<String> serversNames) {
    List<String> sgasNames;
    
    initializeExtremes(serversNames);
    
    sgasNames = getSortedSGAByCPUAndIOCapacity(serversNames);
    
    if(sgasNames.size() != 0)
      return sgasNames.get(0);
    return null;
  }

  /**
   * {@inheritDoc}
   */
  public List<String> chooseServers(CommandInfo command, 
                                    List<String> serversNames,
                                    int numServers) {
    List<String> sgasNames;
    
    initializeExtremes(serversNames);
    
    sgasNames = getSortedSGAByCPUAndIOCapacity(serversNames);

    if(sgasNames.size() >= numServers)
      return sgasNames.subList(0, numServers);
    return null;
  }
  
  private List<String> getSortedSGAByCPUAndIOCapacity(
    List<String> serversNames) {
    List<SGASet> sgasSets = new ArrayList<SGASet>();
    List<String> sgasNames = new ArrayList<String>();
    SGAService sgaService = SGAService.getInstance();
    
    for(int i = 0; i < serversNames.size(); i++) {
      SGASet sgaSet = sgaService.getSGASet(serversNames.get(i));
      if(sgaSet != null)
        sgasSets.add(sgaSet);
    }

    Collections.sort(sgasSets, new SGASetCPUAndIOComparator(serversNames, 
      minWritingCap, maxWritingCap, minProcCap, maxProcCap, minSendCap, 
      maxSendCap));
    for (SGASet sga : sgasSets){
      sgasNames.add(sga.getName());
    }
    return sgasNames;
  }
}
