/*
 * $Id: ProjectSynchronizationService.java 176168 2016-09-22 21:12:51Z fpina $
 */
package csbase.server.services.projectsynchronizationservice;

import csbase.exception.ServiceFailureException;
import csbase.logic.*;
import csbase.logic.filters.ProjectFileNameFilter;
import csbase.remote.ProjectServiceInterface;
import csbase.remote.ProjectSynchronizationMonitor;
import csbase.remote.ProjectSynchronizationServiceInterface;
import csbase.remote.ServerEntryPoint;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.projectservice.ProjectService;
import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;

import java.rmi.Naming;
import java.rmi.RemoteException;
import java.security.PublicKey;
import java.text.MessageFormat;
import java.util.*;

/**
 * Implementao do Servio de sincronizao de projetos. Este Servio realiza a
 * sincronizao entre dois projetos quaisquer, copiando os arquivos indicados
 * de um para o outro.<br>
 * Atualmente,  necessrio fornecer o usurio e a senha a serem utilizados na
 * conexo com servidores remotos. No futuro, caso o usurio fornecido para a
 * conexo com o servidor remoto seja o mesmo que chama o Servio, o suporte de
 * chaves pblica/privada sero utilizado para que no seja preciso fornecer a
 * senha. No caso de no haver registro de chaves no servidor destino, ou do
 * usurio ser outro, a senha continuar sendo necessria.
 */
public class ProjectSynchronizationService extends Service implements
  ProjectSynchronizationServiceInterface {

  /**
   * Construtor do servio
   *
   * @throws ServerException no caso de falha na criao do servio.
   */
  protected ProjectSynchronizationService() throws ServerException {
    super(SERVICE_NAME);
  }

  /**
   * Cria o servio
   *
   * @throws ServerException no caso de falha de criao.
   */
  public static void createService() throws ServerException {
    new ProjectSynchronizationService();
  }

  /**
   * Obtm a instncia do servio de sincronizao de projetos
   *
   * @return a instncia do servio de sincronizao de projetos
   */
  public static ProjectSynchronizationService getInstance() {
    return (ProjectSynchronizationService) getInstance(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() throws ServerException {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Adiciona um ProjectSynchronizationFileInfo em map representando o node, se
   * este for um arquivo. Se for um diretrio, faz recurso nos filhos.
   *
   * @param service servio.
   * @param node n
   * @param map mapa
   * @throws RemoteException a Exceo lanada.
   */
  private void addFileInfo(ProjectServiceInterface service,
    ClientProjectFile node, Map<String, ProjectSynchronizationFileInfo> map)
    throws RemoteException {
    // Verifica se o arquivo a ser adicionado est na lista de filtros de
    // arquivos a serem ignorados na sincronizao.
    if (acceptFileFilters(node)) {
      return;
    }
    if (node.isDirectory()) {
      // Temos um problema por causa da carga sob demanda da rvore de
      // projetos. Quando precisamos de mais um n, no podemos pedir ao
      // servidor, pois o projeto em questo pode estar em um servidor
      // remoto no apontado por ele.
      ClientProjectFile[] children =
        service.getChildren(node.getProjectId(), node.getPath());
      for (int i = 0; i < children.length; i++) {
        addFileInfo(service, children[i], map);
      }
    }
    else {
      ClientProjectFileInfo cpfi =
        new ClientProjectFileInfo(node.getType(), node.size(), node
          .getModificationDate(), node.isUnderConstruction(), node.isLocked());
      ProjectSynchronizationFileInfo psfi =
        new ProjectSynchronizationFileInfo(cpfi, node.getPath(), false);
      map.put(psfi.generateAbsoluteFilename(), psfi);
    }
  }

  /**
   * Note que alguns mtodos dos objetos da lgica que representam o projeto e
   * suas partes chamam diretamente o servidor atravs do ClientRemoteLocator.
   * Nesse ponto, ns no podemos usar o ClientRemoteLocator pois o projeto em
   * questo pode estar em um servidor remoto no apontado por ele. Com isso,
   * algumas chamadas precisam ser feitas diretamente ao Servio remoto.
   *
   * @param service Referncia para o servio de projetos.
   * @param projectId Identificador do projeto.
   * @return .
   *
   * @throws ServiceFailureException
   */
  private Map<String, ProjectSynchronizationFileInfo> getAllFileInfos(
    ProjectServiceInterface service, Object projectId) {
    try {
      CommonClientProject project = service.openProject(projectId, true);
      if (project == null) {
        throw new ServiceFailureException("Projeto inexistente: " + projectId);
      }
      ClientProjectFile root = project.getRoot();
      Map<String, ProjectSynchronizationFileInfo> map =
        new HashMap<String, ProjectSynchronizationFileInfo>();
      addFileInfo(service, root, map);
      service.closeProject(project.getId(), true);
      return map;
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("ProjectSynchronizationService.error.rmi"), e);
    }
  }

  /**
   * Faz o login remoto.
   *
   * @param serverName nome do servidor. Pode ser <b>null</b>; neste caso, ser
   *        usado o servidor corrente
   * @param login identificador do usurio
   * @param password senha do usurio no servidor remoto
   * @return informaes do login remoto
   */
  private LoginInfo remoteLogin(String serverName, String login, String password) {
    LoginInfo loginInfo = new LoginInfo();
    if (serverName == null) {
      loginInfo.projectService = ProjectService.getInstance();
      return loginInfo;
    }
    try {
      String url = "rmi://" + serverName + "/" + ServerEntryPoint.LOOKUP;
      try {
        loginInfo.server = (ServerEntryPoint) Naming.lookup(url);
        // Engloba as excees NotBoundException, RemoteException e
        // MalformedURLException que podem ser lanadas por Naming.lookup(url).
      }
      catch (Exception e) {
        loginInfo.server = null;
      }
      if (loginInfo.server == null) {
        String errorMsg =
          this.getString("ProjectSynchronizationService.error.invalid.server");
        errorMsg = MessageFormat.format(errorMsg, serverName);
        throw new ServiceFailureException(errorMsg);
      }
      Object key = Service.getKey();
      Locale locale = LoginService.getInstance().getUserSessionLocale(key);
      loginInfo.serverName = serverName;
      PublicKey publicKey = LoginService.getInstance().getPublicKey();
      EncryptedPassword encryptedPassword =
        LoginPasswordCipher.encrypt(password, publicKey);
      loginInfo.session =
        loginInfo.server.login(login, encryptedPassword, locale);
      if (loginInfo.session == null) {
        final String errorMsg =
          this
            .getString("ProjectSynchronizationService.error.invalid.user.password");
        throw new ServiceFailureException(errorMsg);
      }
      loginInfo.projectService =
        (ProjectServiceInterface) loginInfo.server.fetchService(
          loginInfo.session.getKey(), ProjectServiceInterface.SERVICE_NAME);

      // System.out.println("-> Testar demora maior que
      // LoginService.cleanTime!");
      // try { Thread.sleep(5000); } catch (Exception e) { e.printStackTrace();
      // }
      return loginInfo;
    }
    catch (Exception e) {
      // Mensagem de erro mais geral.
      String errorMsg = this.getString("ProjectSynchronizationService.error");
      errorMsg = MessageFormat.format(errorMsg, serverName);
      // Concatenao com mensagem de erro mais especfica.
      if (e instanceof ServiceFailureException) {
        if (e.getMessage() != null && !e.getMessage().isEmpty()) {
          errorMsg += " " + e.getMessage();
        }
      }
      throw new ServiceFailureException(errorMsg, e);
    }
  }

  /**
   * Faz o logout no servidor remoto
   *
   * @param loginInfo informaes do login remoto
   */
  private void remoteLogout(LoginInfo loginInfo) {
    try {
      if (loginInfo.server != null) {
        loginInfo.server.logout(loginInfo.session.getKey());
      }
    }
    catch (Exception e) {
      String errorMsg =
        this.getString("ProjectSynchronizationService.error.close.connection");
      errorMsg = MessageFormat.format(errorMsg, loginInfo.serverName);
      throw new ServiceFailureException(errorMsg, e);
    }
  }

  /**
   * Ajusta o parmetro <code>transfer</code> de cada arquivo de mA para que o
   * arquivo seja transferido se s existir em mA ou se existir em mA e mB mas
   * tiver sido alterado mais recentemente em mA. Caso o arquivo deva ser
   * transferido segundo o critrio acima, mas exista e seja apenas de leitura
   * em mB, ele no  ajustado para ser transferido.
   *
   * @param mA .
   * @param mB .
   */
  private void adjustTransfer(Map<String, ProjectSynchronizationFileInfo> mA,
    Map<String, ProjectSynchronizationFileInfo> mB) {
    Iterator<String> it = mA.keySet().iterator();
    while (it.hasNext()) {
      String fileName = it.next();
      ProjectSynchronizationFileInfo i1 = mA.get(fileName);
      ProjectSynchronizationFileInfo i2 = mB.get(fileName);
      if ((i2 == null)
        || ((i1.getClientFileInfo().getModificationDate() > i2
          .getClientFileInfo().getModificationDate()) && !i2
          .getClientFileInfo().isUnderConstruction())) {
        i1.setTransfer(true);
        i1.setNewFile(true);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectSynchronizationData buildTransferMaps(
    final ProjectSynchronizationData data) {
    ProjectSynchronizationUnit unitA = data.getUnitA();
    ProjectSynchronizationUnit unitB = data.getUnitB();
    if (unitA.getProjectId() == null) {
      throw new ServiceFailureException("Projeto de origem nulo.");
    }
    if (unitB.getProjectId() == null) {
      throw new ServiceFailureException("Projeto de destino nulo.");
    }
    LoginInfo loginInfo =
      remoteLogin(unitA.getServer(), unitA.getLogin(), unitA.getPassword());
    Map<String, ProjectSynchronizationFileInfo> mapA =
      getAllFileInfos(loginInfo.projectService, unitA.getProjectId());
    remoteLogout(loginInfo);
    loginInfo =
      remoteLogin(unitB.getServer(), unitB.getLogin(), unitB.getPassword());
    Map<String, ProjectSynchronizationFileInfo> mapB =
      getAllFileInfos(loginInfo.projectService, unitB.getProjectId());
    remoteLogout(loginInfo);
    adjustTransfer(mapA, mapB);
    adjustTransfer(mapB, mapA);
    unitA.setFiles(mapA);
    unitB.setFiles(mapB);
    return data;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectSynchronizationListProjects buildListProjects(
    ProjectSynchronizationData data) {
    ProjectSynchronizationUnit unitA = data.getUnitA();
    ProjectSynchronizationUnit unitB = data.getUnitB();
    try {
      LoginInfo loginInfo =
        remoteLogin(unitA.getServer(), unitA.getLogin(), unitA.getPassword());
      List<UserProjectInfo> listA =
        loginInfo.projectService.getProjectsFromUser(unitA.getProjectOwnerId());
      Collections.sort(listA);
      remoteLogout(loginInfo);
      loginInfo =
        remoteLogin(unitB.getServer(), unitB.getLogin(), unitB.getPassword());
      List<UserProjectInfo> listB =
        loginInfo.projectService.getProjectsFromUser(unitB.getProjectOwnerId());
      Collections.sort(listB);
      remoteLogout(loginInfo);
      ProjectSynchronizationListProjects pslp =
        new ProjectSynchronizationListProjects(listA, listB);
      return pslp;
    }
    catch (RemoteException e) {
      throw new ServiceFailureException("Erro ao obter os projetos.", e);
    }
  }

  /**
   * Transfere integralmente os dados do canal de origem para o canal de
   * destino, utilizando o buffer fornecido.
   *
   * @param channelFrom canal de origem
   * @param channelTo canal de destino
   * @param monitor monitor da operao de sincronizao
   *
   * @throws CancelledException
   * @throws Exception
   */
  private void transfer(RemoteFileChannelInfo channelFrom,
    RemoteFileChannelInfo channelTo, Monitor monitor)
    throws CancelledException, Exception {
    SyncRemoteFileChannel rfcfrom = null;
    SyncRemoteFileChannel rfcto = null;
    Exception err = null;
    try {
      rfcfrom =
        new SyncRemoteFileChannel(channelFrom.getIdentifier(), false,
          channelFrom.getHost(), channelFrom.getPort(), channelFrom.getKey());
      rfcfrom.open(true);
      rfcto =
        new SyncRemoteFileChannel(channelTo.getIdentifier(), true, channelTo
          .getHost(), channelTo.getPort(), channelTo.getKey());
      rfcto.open(false);
      long size = rfcfrom.getSize();
      rfcfrom.syncTransferTo(0, size, rfcto);
      rfcto.setSize(size);
    }
    catch (Exception e) {
      err = e;
    }
    finally {
      if (rfcto != null) {
        try {
          rfcto.close();
        }
        catch (FailureException e) {
          err = (err == null ? e : err);
        }
      }
      if (rfcfrom != null) {
        try {
          rfcfrom.close();
        }
        catch (FailureException e) {
          err = (err == null ? e : err);
        }
      }
      if (err != null) {
        throw err;
      }
    }
  }

  /**
   * Copia do projeto de origem para o projeto de destino os arquivos que
   * estiverem marcados para serem transferidos.
   *
   * @param serviceFrom Servio de projetos de origem
   * @param projectFrom Projeto de origem
   * @param serviceTo Servio de projetos de destino
   * @param projectTo Projeto de destino
   * @param fileInfos Informaes sobre os arquivos
   * @param results Lista onde sero includas mensagens de resultado para cada
   *        arquivo transferido
   * @param monitor Monitor da operao de sincronizao
   */
  private synchronized void copyFiles(ProjectServiceInterface serviceFrom,
    CommonClientProject projectFrom, ProjectServiceInterface serviceTo,
    CommonClientProject projectTo,
    Map<String, ProjectSynchronizationFileInfo> fileInfos,
    List<String> results, Monitor monitor) {
    Iterator<String> it = fileInfos.keySet().iterator();
    while (it.hasNext() && (monitor.isCancelled() == false)) {
      String key = it.next();
      ProjectSynchronizationFileInfo fileInfo = fileInfos.get(key);
      if (!fileInfo.getTransfer()) {
        continue;
      }
      String fileName = fileInfo.getFilename();
      String backupName = null;
      boolean fileCreated = false;
      try {
        // Pegando a data
        long modificationDate =
          fileInfo.getClientFileInfo().getModificationDate();

        // Verifica se o arquivo existe no destino como um diretrio. Se existir,
        // anota o erro e retorna.
        ClientProjectFile toFile =
          findFile(serviceTo, projectTo.getRoot(), fileInfo
            .getAbsoluteFilename());
        if ((toFile != null) && toFile.isDirectory()) {
          String msg =
            "O projeto " + projectTo.getName()
              + " possui um diretrio com o mesmo nome que " + fileName;
          monitor.partialResult(fileInfo.getAbsoluteFilename(), msg);
          results.add(msg);
          continue;
        }

        ClientProjectFile toDir =
          findFile(serviceTo, projectTo.getRoot(), fileInfo
            .getAbsoluteDirPath());

        // Se no existe o diretorio no destino, cria ele e seus filhos
        if (toDir == null) {
          serviceTo.createDirectory(projectTo.getId(), fileInfo
            .getAbsoluteDirPath());
        }
        else {
          // Se existe, precisamos verificar se no destino, ele  um diretrio
          // Se for um arquivo, anotamos o erro e passamos para o prximos
          if (!toDir.isDirectory()) {
            String msg =
              "O projeto " + projectTo.getName()
                + " possui um arquivo com o mesmo nome que o diretrio "
                + toDir.getName();
            monitor.partialResult(fileInfo.getAbsoluteFilename(), msg);
            results.add(msg);
            continue;
          }
        }

        // verificando se existe arquivo no destino
        if (toFile != null) {
          // Fazendo backup do arquivo no destino
          backupName =
            generateBackupName(serviceTo, projectTo.getRoot(), fileInfo
              .getAbsoluteFilename());
          serviceTo.renameFile(projectTo.getId(), fileInfo
            .getAbsoluteFilename(), backupName);
        }
        monitor.initTransfer(fileInfo.getFilename());
        // Criando arquivo no destino
        serviceTo.createFile(projectTo.getId(), fileInfo.getAbsoluteDirPath(),
          fileInfo.getFilename(), fileInfo.getClientFileInfo().getType());
        fileCreated = true;
        String description =
          serviceFrom.getFileDescription(projectFrom.getId(), fileInfo
            .getAbsoluteFilename());
        // Setando description do arquivo no destino
        serviceTo.setFileDescription(projectTo.getId(), fileInfo
          .getAbsoluteFilename(), description);

        // criando informao de canal para o arquivo de origem
        RemoteFileChannelInfo channelFrom =
          serviceFrom.openFileChannel(projectFrom.getId(), fileInfo
            .getAbsoluteFilename(), true);
        // criando informao de canal para o arquivo de destino
        RemoteFileChannelInfo channelTo =
          serviceTo.openFileChannel(projectTo.getId(), fileInfo
            .getAbsoluteFilename(), false);
        // transferindo os dados entre os canais
        transfer(channelFrom, channelTo, monitor);
        serviceTo.setFileModificationDate(projectTo.getId(), fileInfo
          .getAbsoluteFilename(), modificationDate);
        // arquivo copiando com sucesso
        monitor.partialResult(fileInfo.getAbsoluteFilename(), null);
      }
      catch (Exception e) {
        String result =
          "Falha na transferncia de: " + fileName + " - " + e.getMessage();
        Server.logSevereMessage(result, e);
        monitor.partialResult(fileInfo.getAbsoluteFilename(), result);
        results.add(result);
        tryRecovery(backupName, fileCreated, serviceTo, projectTo, fileInfo,
          results, monitor);
        backupName = null;
      }
      finally {
        if (backupName != null) {
          String[] delete = fileInfo.getAbsoluteFilename().clone();
          int index = delete.length - 1;
          delete[index] = backupName;
          try {
            serviceTo.removeFile(projectTo.getId(), delete);
          }
          catch (Exception e) {
            String result =
              "Falha na remoo de arquivo de backup: " + delete[index] + " - "
                + e;
            monitor.partialResult(delete, result);
            results.add(result);
          }
        }
      }
    }
  }

  /**
   * Tenta recuperar o backup do arquivo feito antes da cpia.
   *
   * @param backupName o nome do arquivo de backup
   * @param fileCreated flag para indicar se existe um arquivo destino
   *        inconcistente
   * @param service Servio de projeto
   * @param project Projeto
   * @param fileInfo Informaes sobre o arquivo
   * @param results Lista onde sero includas mensagens de resultado para cada
   *        arquivo transferido
   * @param monitor Monitor da operao de sincronizao
   */
  private void tryRecovery(String backupName, boolean fileCreated,
    ProjectServiceInterface service, CommonClientProject project,
    ProjectSynchronizationFileInfo fileInfo, List<String> results,
    Monitor monitor) {
    try {
      if (fileCreated) {
        service.removeFile(project.getId(), fileInfo.getAbsoluteFilename());
      }
      if (backupName != null) {
        String[] actualFilename = fileInfo.getAbsoluteFilename().clone();
        actualFilename[actualFilename.length - 1] = backupName;
        service.renameFile(project.getId(), actualFilename, fileInfo
          .getFilename());
      }
    }
    catch (Exception e) {
      Server.logSevereMessage(e.getMessage() + " - " + project.getName());
      String result =
        "Falha na recuperao de: " + fileInfo.getFilename() + " - " + e;
      monitor.partialResult(fileInfo.getAbsoluteFilename(), result);
      results.add(result);
    }
  }

  /**
   * Verifica se existe o arquivo representando pelo <code>absolutePath</code>.
   *
   * @param service Servio de projetos onde deve procurar o caminho
   * @param pf ProjectFile
   * @param absolutePath path absoluto do arquivo
   *
   * @return boolean true se o arquivo existe
   * @throws RemoteException
   */
  private boolean existsFile(ProjectServiceInterface service,
    ClientProjectFile pf, String[] absolutePath) throws RemoteException {
    return service.existsFile(pf.getProjectId(), absolutePath);
  }

  /**
   * Retorna o <code>ProjectFile</code> referente ao caminho dado.
   *
   * @param service Servio de projetos onde deve procurar o caminho
   * @param root diretrio a partir de onde a busca  feita
   * @param path path do arquivo, sem o nome do projeto
   *
   * @return o arquivo se ele existir, ou null caso contrrio
   * @throws RemoteException
   */
  private ClientProjectFile findFile(ProjectServiceInterface service,
    ClientProjectFile root, String[] path) throws RemoteException {
    ClientProjectFile file = root;
    for (int i = 0; i < path.length; i++) {
      ClientProjectFile child =
        service.getChild(root.getProjectId(), file.getPath(), path[i]);
      if (child == null) {
        return null;
      }
      /*
       * FIXME estas linhas no deveriam ser necessrias, deveramos obter o
       * filho diretamente do ClientProjectFile, e no via servio
       */
      child.setParent(file);
      file = child;
    }
    return file;
  }

  /**
   * Gera um nome de backup para um arquivo com base na hora local. A operao 
   * repetida se o nome j existir.
   *
   * @param service Servio de projetos que contm o arquivo
   * @param pf
   * @param absolutePath
   *
   * @return o nome
   * @throws RemoteException
   */
  private String generateBackupName(ProjectServiceInterface service,
    ClientProjectFile pf, String[] absolutePath) throws RemoteException {
    String filename = absolutePath[absolutePath.length - 1];
    String[] path = absolutePath.clone();
    do {
      long time = System.currentTimeMillis();
      path[path.length - 1] = filename + Long.toString(time);
    } while (existsFile(service, pf, path));
    return path[path.length - 1];
  }

  /**
   * Transfere os arquivos entre os projetos A e B, seguindo as indicaes dos
   * atributos <code>transfer</code> da lista A e da lista B.
   *
   * @param serviceA Servio de projetos que hospeda o projeto A
   * @param serviceB Servio de projetos que hospeda o projeto B
   * @param unitA Dados do projeto A
   * @param unitB Dados do projeto B
   * @param resultsA Lista onde sero includos os resultados para cada
   *        transferncia de arquivo de A para B
   * @param resultsB Lista onde sero includos os resultados para cada
   *        transferncia de arquivo de B para A
   * @param monitor Monitor da operao de sincronizao
   *
   * @throws ServiceFailureException
   */
  private void transferFiles(ProjectServiceInterface serviceA,
    ProjectServiceInterface serviceB, ProjectSynchronizationUnit unitA,
    ProjectSynchronizationUnit unitB, List<String> resultsA,
    List<String> resultsB, Monitor monitor) {
    try {
      CommonClientProject projectA =
        serviceA.openProject(unitA.getProjectId(), true);
      if (projectA == null) {
        String server = (unitA.getServer() != null) ? unitA.getServer() : "";
        String msg =
          String.format(
            getString("ProjectSynchronizationService.error.project.not.found"),
            unitA.getProjectId(), server);
        throw new ServiceFailureException(msg);
      }
      CommonClientProject projectB =
        serviceB.openProject(unitB.getProjectId(), true);
      if (projectB == null) {
        String server = (unitB.getServer() != null) ? unitB.getServer() : "";
        String msg =
          String.format(
            getString("ProjectSynchronizationService.error.project.not.found"),
            unitB.getProjectId(), server);
        throw new ServiceFailureException(msg);
      }
      try {
        copyFiles(serviceA, projectA, serviceB, projectB, unitA.getFiles(),
          resultsA, monitor);
        copyFiles(serviceB, projectB, serviceA, projectA, unitB.getFiles(),
          resultsB, monitor);
      }
      catch (CancelledException e) {
        // Nada a fazer:  considerado um trmino normal.
      }
      serviceA.closeProject(projectA.getId(), true);
      serviceB.closeProject(projectB.getId(), true);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("ProjectSynchronizationService.error.rmi"), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectSynchronizationResult synchronizeProjects(
    ProjectSynchronizationData data, ProjectSynchronizationMonitor monitor) {
    String errorMsg = null;
    ProjectSynchronizationResult result = null;
    List<String> resultsA = new Vector<String>();
    List<String> resultsB = new Vector<String>();
    Monitor localMonitor = new Monitor(monitor);
    try {
      ProjectSynchronizationUnit unitA = data.getUnitA();
      LoginInfo loginInfoA =
        remoteLogin(unitA.getServer(), unitA.getLogin(), unitA.getPassword());
      ProjectSynchronizationUnit unitB = data.getUnitB();
      LoginInfo loginInfoB =
        remoteLogin(unitB.getServer(), unitB.getLogin(), unitB.getPassword());
      transferFiles(loginInfoA.projectService, loginInfoB.projectService,
        unitA, unitB, resultsA, resultsB, localMonitor);
      remoteLogout(loginInfoA);
      remoteLogout(loginInfoB);
    }
    catch (ServiceFailureException e) {
      errorMsg = e.getMessage();
      result =
        new ProjectSynchronizationResult(errorMsg, e.getCause(), resultsA,
          resultsB);
    }
    localMonitor.finalResult(errorMsg);
    if (result == null) {
      result = new ProjectSynchronizationResult(resultsA, resultsB);
    }
    return result;
  }

  /**
   * Monitor da sincronizao. Esta classe  utilizada como uma casca sobre o
   * monitor recebido para tratar as eventuais excees remotas e tambm o caso
   * do monitor ser nulo. Dessa forma, a Implementao dos mtodos no precisam
   * se preocupar com esses casos.
   */
  private class Monitor implements ProjectSynchronizationMonitor {
    /** Referncia para o observador remoto. */
    private final ProjectSynchronizationMonitor monitor;

    /**
     * Construtor do monitor local.
     *
     * @param monitor monitor remoto
     */
    Monitor(ProjectSynchronizationMonitor monitor) {
      this.monitor = monitor;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void partialResult(String[] file, String result) {
      if (monitor == null) {
        return;
      }
      try {
        monitor.partialResult(file, result);
      }
      catch (RemoteException e) {
        Server.logWarningMessage("Ignorando falha em monitor.partialResult: "
          + e);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void finalResult(String result) {
      if (monitor == null) {
        return;
      }
      try {
        monitor.finalResult(result);
      }
      catch (RemoteException e) {
        Server
          .logWarningMessage("Ignorando falha em monitor.finalResult: " + e);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isCancelled() {
      if (monitor == null) {
        return false;
      }
      try {
        return monitor.isCancelled();
      }
      catch (RemoteException e) {
        Server
          .logWarningMessage("Ignorando falha em monitor.isCancelled: " + e);
        return false;
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void initTransfer(String file) throws RemoteException {
      monitor.initTransfer(file);
    }
  }

  /**
   * Estrutura de dados que armazenas as Informaes do login remoto.
   */
  private static class LoginInfo {
    /** Nome do servidor */
    String serverName;
    /** Sesso do login */
    Session session;
    /** Ponto de entrada do servidor remoto */
    ServerEntryPoint server;
    /** Interface remota do servio de projetos */
    ProjectServiceInterface projectService;
  }

  /**
   * Exceo que acusa o cancelamento pelo cliente da operao de sincronizao.
   */
  private static class CancelledException extends RuntimeException {
  }

  /**
   * Verifica se o arquivo passado  aceito pela lista de arquivos a serem
   * ignorados na sincronizao de projetos. Se o nome arquivo casa com pelo
   * menos um filtro, o arquivo  desconsiderado.
   *
   * @param node Arquivo onde os filtros sero aplicados.
   * @return Se o arquivo casa com algums dos filtros de arquivos a serem
   *         ignorados.
   */
  private boolean acceptFileFilters(final ClientProjectFile node) {

    // FIXME deveramos ter uma lista esttica j com os filtros, no deveramos
    // recri-los a cada execuo
    List<String> filtersList =
      this.getStringListProperty("filesFilterPatternsToHide");
    if (filtersList == null || filtersList.isEmpty()) {
      return false;
    }

    // Verifica se o nome de arquivo casa com algum dos filtros de arquivos.
    for (String filterName : filtersList) {
      if (filterName.isEmpty()) {
        return true;
      }
      // FIXME deveramos usar os filtros da lista esttica
      ProjectFileFilter filter = new ProjectFileNameFilter(filterName);
      if (filter.accept(node)) {
        return true;
      }
    } // Fim do FOR dos filtros.
    return false;

  }

}
