package csbase.server.services.schedulerservice.heuristic;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import csbase.logic.CommandInfo;
import csbase.logic.SGASet;

/**
 * Classe que contm todas as informaes a respeito de uma alocao, seja ela
 * factvel ou parcial.
 * 
 * Definio de factvel baseada no artigo Simple and Integrated Heuristic
 * Algorithms for Scheduling Tasks with Time and Resource Constraints. Zhao,
 * Wei; Ramamritham, Krithi. The Journal of Systems and Software 7, 195-207
 * (1987)
 * 
 * @author Tecgraf/PUC-Rio
 */
public class Allocation implements Cloneable, Comparable<Allocation> {
	private Map<CommandInfo, SGASet> allocationMap = new HashMap<CommandInfo, SGASet>();
	private List<SGASet> simulatedServers = new LinkedList<SGASet>();
	private List<CommandInfo> remainingCommands = new LinkedList<CommandInfo>();
	private boolean isFeasible;
	private ResourceControllerInterface manager;
	private HashMap<String, SGASet> originalServers;

	public Allocation(Map<CommandInfo, SGASet> allocationMap,
			List<SGASet> simulatedServers, List<CommandInfo> remainingCommands,
			boolean isFeasible, ResourceControllerInterface manager,
			HashMap<String, SGASet> originalServers) {
		this.allocationMap.putAll(allocationMap);
		this.simulatedServers.addAll(simulatedServers);
		this.remainingCommands.addAll(remainingCommands);
		this.isFeasible = isFeasible;
		this.manager = manager;
		this.originalServers = originalServers;
	}

	private Map<CommandInfo, SGASet> createAllocationMapCopy() {
		Map<CommandInfo, SGASet> map = new HashMap<CommandInfo, SGASet>();
		map.putAll(this.allocationMap);
		return map;
	}

	/**
	 * Checa se a alocao  factvel.
	 * 
	 * @return true se for factvel ou false caso contrrio
	 */
	public boolean isFeasible() {
		return this.isFeasible;
	}

	/**
	 * Retorna mapa de alocao factvel, se houver.
	 * 
	 * @return mapa de alocao se for factvel, null caso contrrio.
	 */
	public Map<CommandInfo, SGASet> getFeasibleAllocationMap() {
		return this.isFeasible ? createAllocationMapCopy() : null;
	}

	/**
	 * Retorna mapa de alocao parcial (podendo ser tambm factvel)
	 * 
	 * @return mapa de alocao parcial
	 */
	public Map<CommandInfo, SGASet> getPartialAllocationMap() {
		return createAllocationMapCopy();
	}

	/**
	 * Retorna estado simulado de ocupao dos SGAs considerando as alocaes
	 * das quais dispe.
	 * 
	 * @return lista de SGASets referentes s simulaes de cada um dos SGAs
	 */
	public List<SGASet> getServersSimulations() {
		List<SGASet> states = new LinkedList<SGASet>();
		states.addAll(this.simulatedServers);
		return states;
	}

	/**
	 * Retorna os comandos que no foram alocados nessa alocao.
	 * 
	 * @return lista de comandos no alocados
	 */
	public List<CommandInfo> getRemainingCommands() {
		List<CommandInfo> cmds = new LinkedList<CommandInfo>();
		cmds.addAll(this.remainingCommands);
		return cmds;
	}

	/**
	 * Desfaz alocao de um comando passado como parmetro, retirando-o do mapa
	 * de alocaes e colocando-o na lista de comandos no resolvidos. O SGA que
	 * estava alocado para o comando tem sua simulao atualizada. Nesse caso,
	 * qualquer alocao deixa de ser factvel.
	 * 
	 * @param cmd
	 *            Comando a ser retirado da alocao
	 */
	public void unsetAllocation(CommandInfo cmd) {
		SGASet originalPreviousSGA = allocationMap.get(cmd);
		if (originalPreviousSGA != null) {
			this.allocationMap.remove(cmd);

			// previous SGA  um SGASet original, devemos procurar o simulado
			SGASet previousSGA = null;
			for (SGASet sga : simulatedServers)
				if (sga.getName() == originalPreviousSGA.getName()) {
					previousSGA = sga;
					break;
				}
			this.simulatedServers.remove(previousSGA);
			this.simulatedServers.add(this.manager
					.undoResourceConsumptionSimulation(cmd, previousSGA));
			this.remainingCommands.remove(cmd);
			this.isFeasible = false;
		}
	}

	/**
	 * Verifica se  possvel fazer alocao de um comando passado como
	 * parmetro em um SGA passado como parmetro. Se no for possvel faz-la,
	 * o comando continua em seu estado anterior. Se for possvel faz-la, o
	 * coloca no mapa de alocaes e o retira da lista de comandos no
	 * resolvidos (se previamente no estava resolvido). O SGA que estava
	 * alocado para o comando (se houver) tem sua simulao atualizada, bem como
	 * aquele que passou a receb-lo. A alocao passa a ser factvel se e
	 * somente se a lista de comandos restantes (no alocados) estiver vazia.
	 * 
	 * @param cmd
	 *            Comando a ser retirado da alocao
	 * @param server
	 *            SGASet relativo  simulao do SGA que se deseja alocar o
	 *            comando.
	 * @throws ImpossibleAllocationException
	 *             se no  possvel alocar esse comando neste SGA por restris
	 *             nos recursos do SGA
	 * @throws InvalidSGASetException
	 *             se o SGA passado no estiver entre as simulaes de SGAs
	 *             corrente.
	 * 
	 */
	public SGASet setAllocation(CommandInfo cmd, SGASet server)
			throws ImpossibleAllocationException, InvalidSGASetException {
		if (!simulatedServers.contains(server))
			throw new InvalidSGASetException(server, this);
		unsetAllocation(cmd);
		if (server != null) {
			List<SGASet> candidates = new LinkedList<SGASet>();
			candidates.add(server);
			if (manager.getServersThatMeetsRequirements(cmd, candidates)
					.isEmpty())
				throw new ImpossibleAllocationException(cmd, server, this);
			this.allocationMap.put(cmd, originalServers.get(server.getName()));
			this.simulatedServers.remove(server);
			SGASet updatedServerState = manager.simulateResourceConsumption(
					cmd, server);
			this.simulatedServers.add(updatedServerState);
			this.remainingCommands.remove(cmd);
			if (this.remainingCommands.isEmpty())
				this.isFeasible = true;
			return updatedServerState;
		}
		return null;
	}

	/**
	 * Verifica todas as possveis mudanas que se pode fazer com um comando
	 * passado como parmetro, ou seja, quais os SGAs que podem receb-lo que
	 * no sejam o SGA em que o comando j se encontra (se o comando j tiver um
	 * SGA).
	 * 
	 * @param cmd
	 *            Comando
	 * @return lista de SGAs que podem receber o comando, excluindo SGA que j
	 *         est associado ao comando (se houver).
	 */
	public List<SGASet> getPossibleChanges(CommandInfo cmd) {
		List<SGASet> candidates = this.manager.getServersThatMeetsRequirements(
				cmd, this.simulatedServers);
		SGASet originalCurrentSGA = this.allocationMap.get(cmd);

		if (originalCurrentSGA != null) {
			SGASet currentSGA = null;
			for (SGASet sga : candidates)
				if (sga.getName() == originalCurrentSGA.getName()) {
					currentSGA = sga;
					break;
				}
			if (currentSGA != null)
				candidates.remove(currentSGA);
		}
		return candidates;
	}

	/**
	 * Verifica todas as possveis mudanas que se pode fazer com um SGA passado
	 * como parmetro, ou seja, quais os comandos com alocao resolvida ou no
	 * que podem ser adicionados a esse SGA. So excludos os comandos que j
	 * esto direcionados a esse SGA (se houver).
	 * 
	 * @param server
	 *            SGA
	 * @return lista de SGAs que podem receber o comando, excluindo SGA que j
	 *         est associado ao comando (se houver).
	 */
	public List<CommandInfo> getPossibleChanges(SGASet server) {
		List<CommandInfo> candidates = new LinkedList<CommandInfo>();
		candidates.addAll(remainingCommands);
		candidates.addAll(allocationMap.keySet());
		for (CommandInfo cmd : allocationMap.keySet())
			if (allocationMap.get(cmd) == server)
				candidates.remove(cmd);
		return manager.getPossibleCommands(candidates, server);
	}

	public Allocation clone() {
		return new Allocation(this.allocationMap, this.simulatedServers,
				this.remainingCommands, this.isFeasible, this.manager,
				this.originalServers);
	}

	/**
	 * Retorna uma lista que contm os comandos que foram direcionados a um SGA
	 * passado como parmetro
	 * 
	 * @param server
	 *            SGA
	 * @return lista de comandos que foram direcionados a um SGA
	 */
	public List<CommandInfo> getCurrentlyAllocatedCommands(SGASet server) {
		List<CommandInfo> cmds = new LinkedList<CommandInfo>();
		for (CommandInfo cmd : allocationMap.keySet())
			if (allocationMap.get(cmd).getName() == server.getName())
				cmds.add(cmd);
		return cmds;
	}

	/**
	 * Retorna simulao de SGA cujo nome foi passado como parmetro
	 * 
	 * @param serverName
	 * @return
	 */
	public SGASet getServerState(String serverName) {
		for (SGASet server : simulatedServers)
			if (server.getName() == serverName)
				return server;
		return null;
	}

	@Override
	public final int compareTo(Allocation other) {
		if (this.isFeasible==other.isFeasible)
			return 0;
		if (this.isFeasible)
			return 1;
		return -1;
	}
}

@SuppressWarnings("serial")
class ImpossibleAllocationException extends Exception {
	ImpossibleAllocationException(CommandInfo cmd, SGASet server,
			Allocation allocation) {
		super("No  possvel alocar o comando " + cmd.getId() + " no SGA "
				+ server.getName() + " com a alocao " + allocation);
	}
}

@SuppressWarnings("serial")
class InvalidSGASetException extends Exception {
	InvalidSGASetException(SGASet server, Allocation allocation) {
		super("SGASet " + server.getName() + " no encontrado na alocao "
				+ allocation);
	}
}
