/**
 * $Id: ServiceMonitor.java 142360 2013-07-03 17:16:16Z mjulia $
 */

package csbase.server.services.diagnosticservice.monitors.openbus;

import java.util.Locale;
import java.util.Properties;

import scs.core.ComponentId;
import csbase.logic.diagnosticservice.ResourceException;
import csbase.logic.diagnosticservice.Status;
import csbase.server.services.diagnosticservice.MessageTranslator;
import csbase.server.services.diagnosticservice.monitors.AbstractMonitor;
import csbase.server.services.openbusservice.OpenBusService;

/**
 * Monitor de um servio publicado no OpenBus (v 1.5).
 * 
 * @author Tecgraf
 */
public class ServiceOfferMonitor extends AbstractMonitor {

  /**
   * Componente monitorado.
   */
  private ComponentId component;

  /** Indica se as propriedades j foram adicionadas */
  private boolean propertiesChecked = false;

  /**
   * Construtor.
   * 
   * @param component o component monitorado
   * 
   * @throws ResourceException caso algum parmetro seja invlido
   */
  public ServiceOfferMonitor(ComponentId component) throws ResourceException {
    super(component.name);
    this.component = component;

    this.addValidation(new OpenbusServiceValidation());
    this.addValidation(new ConnectionValidation());
    this.addValidation(new RegistryServiceValidation());
    this.addValidation(new OfferIdValidation(component));
    this.addValidation(new DuplicatedOfferValidation(component));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized Status checkResource(Locale locale)
    throws ResourceException {
    if (!propertiesChecked) {
      propertiesChecked = true;
      this.addProperty(MessageTranslator.getString(
        "server.offermonitor.offer.name.label", locale), component.name,
        "component.name em scs.core.ComponentId");
      this.addProperty(MessageTranslator.getString(
        "server.offermonitor.offer.platform.label", locale),
        component.platform_spec, "platform_spec em scs.core.ComponentId");
      this.addProperty(MessageTranslator.getString(
        "server.offermonitor.offer.major.version.label", locale), String
        .valueOf(component.major_version),
        "major_version em scs.core.ComponentId");
      this.addProperty(MessageTranslator.getString(
        "server.offermonitor.offer.minor.version.label", locale), String
        .valueOf(component.minor_version),
        "minor_version em scs.core.ComponentId");
      this.addProperty(MessageTranslator.getString(
        "server.offermonitor.offer.patch.version.label", locale), String
        .valueOf(component.patch_version),
        "patch_version em scs.core.ComponentId");
      // Identificador da oferta
      String offerId =
        OpenBusService.getInstance().getRegisteredComponentBusId(component);
      this.addProperty(MessageTranslator.getString(
        "server.offermonitor.offer.id.label", locale), String.valueOf(offerId),
        "identificador da oferta retornada no registro");
      // Propriedades
      Properties properties =
        OpenBusService.getInstance()
          .getRegisteredComponentProperties(component);
      int i = 0;
      for (Object key : properties.keySet()) {
        this.addProperty(MessageTranslator.getString(
          "server.offermonitor.property.label", new String[] { String
            .valueOf(++i) }, locale), properties.getProperty((String) key),
          (String) key);
      }
      // Facetas
      String[] facets =
        OpenBusService.getInstance().getRegisteredComponentFacets(component);
      i = 0;
      for (String facet : facets) {
        this.addProperty(MessageTranslator.getString(
          "server.offermonitor.facet.label",
          new String[] { String.valueOf(++i) }, locale), facet, null);
      }
    }
    return super.checkResource(locale);
  }
}
