/*
 * FileSystem.java
 * 
 * $Id: FileSystem.java 149543 2014-02-11 13:02:45Z oikawa $
 */
package csbase.server;

import java.io.File;
import java.io.IOException;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import tecgraf.javautils.core.externalcommand.ExternalCommand;

/**
 * Representa o sistema de arquivos do SSI.
 */
public class FileSystem {
  /**
   * Comando para cpia de arquivos.
   */
  private static final String CP_COMMAND = "cp";

  /**
   * Diretrio raiz dos arquivos de propriedades.
   */
  public static final String PROPERTIES_DIR = "properties";

  /**
   * Verifica o espao de um determinado filesystem.
   * 
   * @param path O caminho do filesystem.
   * @param group nmero do grupo a ser capturado no resultado do comando que
   *        obtm o espao em disco (total, livre, em uso)
   * @param timeout tempo mximo disponibilizdo para obteno (negativo ou igual
   *        a zero representa infinito).
   * @return O nmero de bytes disponveis (nmero negativo em caso de erro).
   */
  public static long getSpace(String path, int group, long timeout) {
    final Server server = Server.getInstance();

    final String command = server.getStringProperty("DiskSpace.command");
    final String[] cmdArray = createCommandArray(command, path);
    final ExternalCommand process = new ExternalCommand(cmdArray);

    // Verifica se o comando foi executado com sucesso. 
    boolean ok = false;
    try {
      final int retCode;
      if (timeout <= 0) {
        retCode = process.waitFor();
      }
      else {
        retCode = process.waitFor(timeout);
      }
      ok = (retCode == 0);
    }
    catch (InterruptedException e) {
      process.destroy();
      String fmt = "Timeout (%d ms) na obteno de espao em [%s] - %s.";
      String exMsg = e.getMessage();
      String err = String.format(fmt, timeout, path, exMsg);
      Server.logSevereMessage(err);
      return -9;
    }
    catch (IOException ioe) {
      String err = "Erro de I/O ao obter espao em disco";
      Server.logSevereMessage(err, ioe);
      return -8;
    }

    if (!ok) {
      String stderr = process.getErrorOutput();
      stderr = (stderr != null ? stderr : "---");
      String fmt = "Erro obtendo espao em [%s] (stdout = '%s').";
      Server.logSevereMessage(String.format(fmt, path, stderr));
      process.destroy();
      return -1;
    }

    final String stdout = process.getStandardOutput();
    if (stdout == null) {
      String fmt = "Stdout nulo obtendo espao em [%s].";
      Server.logSevereMessage(String.format(fmt, path));
      return -1;
    }

    final String regex = server.getStringProperty("DiskSpace.regex");
    final Pattern pattern = Pattern.compile(regex);
    final Matcher matcher = pattern.matcher(stdout);
    if (!matcher.find()) {
      String fmt =
        "Falha de match na obteno de espao em [%s] "
          + "(regex = '%s'; string = '%s').";
      String err = String.format(fmt, path, regex, stdout);
      Server.logSevereMessage(err);
      return -1;
    }

    // Valor obtido com sucesso aps todos os testes.
    final String strKbytes = matcher.group(group);
    try {
      long bytes = Long.parseLong(strKbytes) * 1024;
      return bytes;
    }
    catch (NumberFormatException nfe) {
      String fmt = "M formatao na obteno de espao em [%s] ('%s').";
      String err = String.format(fmt, path, strKbytes);
      Server.logSevereMessage(err);
      return -1;
    }
  }

  /**
   * Verifica o espao disponvel para um determinado filesystem. O mtodo
   * File.getUsableSpace no funciona para filesystems maiores de 5Tb (bug
   * cadastrado na Sun). Para sistemas nessa situao, a propriedade
   * FreeDiskSpace.group deve ser definida no arquivo Server.properties de forma
   * que o espao em disco seja obtido atravs de uma chamada direta ao sistema.
   * 
   * @param path O caminho do filesystem.
   * @param timeout tempo mximo (em ms) disponibilizdo para obteno (negativo
   *        ou igual a zero significa infinito).
   * @return O nmero de bytes disponveis (nmero negativo em caso de erro).
   */
  public static long getFreeSpace(String path, long timeout) {
    final Server server = Server.getInstance();
    try {
      final String propertyName = "FreeDiskSpace.group";
      final boolean isNull = server.isPropertyNull(propertyName);
      if (isNull) {
        final File f = new File(path);
        if (f.exists()) {
          return f.getUsableSpace();
        }
        return -1;
      }
      final int group = server.getIntProperty(propertyName);
      return getSpace(path, group, timeout);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro obtendo espao", e);
      return -1;
    }
  }

  /**
   * Verifica o espao total para um determinado filesystem. O mtodo
   * File.getTotalSpace no funciona para filesystems maiores de 5Tb (bug
   * cadastrado na Sun). Para sistemas nessa situao, a propriedade
   * TotalDiskSpace.group deve ser definida no arquivo Server.properties de
   * forma que o espao em disco seja obtido atravs de uma chamada direta ao
   * sistema.
   * 
   * @param path O caminho do filesystem.
   * @param timeout tempo mximo (em ms) disponibilizado para obteno (negativo
   *        ou igual a zero significa infinito).
   * @return O nmero de bytes disponveis (nmero negativo em caso de erro).
   */
  public static long getTotalSpace(String path, long timeout) {
    final Server server = Server.getInstance();
    try {
      final String propertyName = "TotalDiskSpace.group";
      final boolean isNull = server.isPropertyNull(propertyName);
      if (isNull) {
        final File f = new File(path);
        if (f.exists()) {
          return f.getTotalSpace();
        }
        return -1;
      }

      final int group = server.getIntProperty(propertyName);
      return getSpace(path, group, timeout);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro obtendo espao", e);
      return -1;
    }
  }

  /**
   * Move um arquivo para o outro. A implementao deste mtodo copia e depois
   * remove o original.
   * 
   * @param from O arquivo de origem.
   * @param to O arquivo destino.
   * @return Verdadeiro se foi movido com sucesso ou falso, caso contrrio.
   */
  public static boolean moveRegularFile(File from, File to) {
    Server.logInfoMessage("FileSystem.moveRegularFile:  de: "
      + from.getAbsolutePath() + "  para: " + to.getAbsolutePath());

    if (!from.exists()) {
      Server
        .logSevereMessage("FileSystem.moveRegularFile: arquivo origem no existe: "
          + from.getAbsolutePath());
      return false;
    }
    if (!copyFile(from, to)) {
      return false;
    }
    if (from.delete()) {
      return true;
    }
    Server
      .logSevereMessage("FileSystem.moveRegularFile: Erro removendo arquivo origem: "
        + from.getAbsolutePath());
    return false;
  }

  /**
   * Move arquivos ou diretrios.
   * 
   * @param from arquivo ou diretrio de origem
   * @param to arquivo ou diretrio de destino
   * @return Verdadeiro se a movimentao foi realizada com sucesso, Falso caso
   *         contrrio
   */
  public static boolean move(File from, File to) {
    if (!from.isDirectory()) {
      return moveRegularFile(from, to);
    }
    if (!to.mkdirs()) {
      Server
        .logSevereMessage("FileSystem.move: Erro na criando diretrio(s) : "
          + to.getAbsolutePath());
      return false;
    }
    File[] children = from.listFiles();
    boolean success = true;
    for (int i = 0; i < children.length; i++) {
      File newChildFile = new File(to, children[i].getName());
      success = move(children[i], newChildFile);
      if (!success) {
        break;
      }
    }
    if (success) {
      if (from.delete()) {
        return true;
      }
      Server
        .logSevereMessage("FileSystem.move: Erro removendo diretrio origem: "
          + from.getAbsolutePath());
    }
    return success;
  }

  /**
   * Copia o contedo de um arquivo para outro.
   * 
   * @param from O arquivo de origem.
   * @param to O arquivo destino.
   * 
   * @return Verdadeiro se a cpia foi feita com sucesso ou falso, caso
   *         contrrio.
   */
  public static boolean copyFile(File from, File to) {
    Server.logInfoMessage(String.format(
      "FileSystem.copyFile: de '%s' para '%s'", from.getAbsolutePath(),
      to.getAbsolutePath()));
    String[] cmdArray =
      createCommandArray(CP_COMMAND, from.getAbsolutePath(),
        to.getAbsolutePath());

    String cmd =
      String.format(
        "FileSystem:copyFile: Erro copiando arquivo (%s '%s' '%s')",
        cmdArray[0], cmdArray[1], cmdArray[2]);
    try {
      ExternalCommand command = new ExternalCommand(cmdArray);
      if (command.waitFor() != 0) {
        String errorOutput = command.getErrorOutput();
        Server.logSevereMessage(cmd + " :\n"
          + (errorOutput != null ? errorOutput : "---"));
        return false;
      }
    }
    catch (InterruptedException e) {
      // Espera no deve ser interrompida...
      Server.logSevereMessage(cmd, e);
      return false;
    }
    catch (IOException e) {
      Server.logSevereMessage(cmd, e);
      return false;
    }
    return true;
  }

  /**
   * <p>
   * Habilita permisso de execuo para o arquivo especificado.
   * </p>
   * Este mtodo  especfico para cada sistema operacional, e portanto usa a
   * constante <code>ENABLE_EXECUTION_PERMISSION_COMMAND</code> para descobrir
   * qual o comando a ser executado, a qual por sua vez faz referncia a uma
   * propriedade em arquivo.<br>
   * Caso esse mtodo no faa sentido para o sistema operacional operacional
   * corrente, deixe essa propriedade em branco.<br>
   * OBS: O mtodo  bloqueante, isto , a thread de execuo ir aguardar at
   * que o processo termine para retornar.
   * 
   * @param filePath caminho para o arquivo no NFS.
   * 
   * @return <tt>true</tt> se a operao foi bem sucedida.
   * 
   * @throws ServerException caso no seja possvel executar o comando.
   */
  public static final boolean enableExecutionPermission(String filePath)
    throws ServerException {

    if (filePath == null) {
      throw new IllegalArgumentException("filePath == null");
    }
    if (filePath.equals("")) {
      throw new IllegalArgumentException("filePath est vazia");
    }

    try {
      return new File(filePath).setExecutable(true, false);
    }
    catch (Exception e) {
      // Espera no deve ser interrompida...
      throw new ServerException("Erro ao mudar execuo do arquivo '"
        + filePath + "'", e);
    }
  }

  /**
   * Cria um array de strings para ser usado pelo comando
   * {@link Runtime#exec(String[])}.
   * <p>
   * Esta deve ser a forma prefervel de se criar linhas de comando ao invs da
   * criao da linha de comando como uma string, pois se fose feito desta forma
   * (uma nica string) o mtodo {@link Runtime#exec(String)} usaria
   * {@link StringTokenizer} para identificar os tokens, e espaos nos
   * parmetros gerariam tokens indesejados. Envolver os parmetros com aspas
   * no d o resultado esperado (no UNIX p.ex. as aspas so repassadas para o
   * comando, o que gera erro).
   * 
   * @param cmd - comando. Pode conter espaos; neste caso, o comando ser
   *        quebrado em tokens usando {@link StringTokenizer}. O primeiro token
   *        ser usado como comando, e os demais como parmetros, que sero
   *        inseridos antes dos especificados pelo parmetro <code>args</code>.
   * @param args - parmetros
   * @return array de strings ({cmd, args[0], args[1], ...})
   */
  private static String[] createCommandArray(String cmd, String... args) {
    final StringTokenizer cmdTokenizer = new StringTokenizer(cmd);
    final int numTokens = cmdTokenizer.countTokens();
    final String[] cmdArray = new String[args.length + numTokens];
    int i = 0;
    while (i < numTokens) {
      cmdArray[i++] = cmdTokenizer.nextToken();
    }
    System.arraycopy(args, 0, cmdArray, i, args.length);
    return cmdArray;
  }
}
