/**
 * $Id: ServiceManager.java 160147 2014-12-16 22:32:36Z mjulia $
 */

package csbase.server;

import java.io.Serializable;
import java.lang.reflect.InvocationHandler;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import csbase.logic.AdministrationEvent;
import csbase.logic.User;
import csbase.remote.HttpServiceInterface;
import csbase.remote.ServiceInterface;
import csbase.server.plugin.service.IService;
import csbase.server.plugin.service.IServiceManager;
import csbase.server.services.ServiceInvocationHandler;
import csbase.server.services.httpservice.HttpService;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.loginservice.LogoutEvent;
import csbase.server.services.messageservice.MessageService;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import csbase.util.messages.filters.BodyTypeFilter;
import csbase.util.proxy.ProxyUtils;

/**
 * Classe que implementa o gerente de servios do SSI, que ser representado por
 * um singleton.
 *
 * @see Service
 * @author Tecgraf/PUC-Rio
 */
public class ServiceManager implements IServiceManager {

  /**
   * Instncia do singleton para o gerente de servios.
   */
  private static ServiceManager instance;

  /**
   * Mapa de servios para execuo no servidor.
   */
  private final Map<String, Service> servicesMap;
  /**
   * Os servios ordenados de acordo com a sua inicializao.
   */
  private List<Service> sortedServices;

  /**
   * Guarda os proxies dos usurios logados. <br>
   * chave da sesso -> <code>Map</code> de proxies
   */
  private final Map<Object, Map<String, ServiceInterface>> usersProxiesMap;

  /**
   * O identificador de consumidor no MessageService
   */
  private Serializable consumerId;

  /**
   * Mtodo para carregar o gerente de servios; deve ser o primeiro mtodo a
   * ser chamado pois cria o nico objeto da classe.
   */
  protected static void init() {
    if (instance != null) {
      return;
    }
    instance = new ServiceManager();
  }

  /**
   * Mtodo para inicializar os servios cadastrados no gerente de servios.
   *
   * @throws ServerException caso ocorra falha na inicializao do servio
   */
  protected void initAllServices() throws ServerException {
    Server.logInfoMessage("Inicializando servios...");

    this.sortedServices = new LinkedList<Service>();

    List<Service> notInitializedServices = this.getEnabledServices();
    while (notInitializedServices.size() > 0) {
      tryInitServices(notInitializedServices);
      notInitializedServices.removeAll(this.sortedServices);
    }

    // Instalando service manager
    try {
      install();
    }
    catch (RemoteException re) {
      throw new ServerException(re);
    }

    Server.logInfoMessage("Servios inicializados");
  }

  /**
   * Obtm todos os servios que esto habilitados.
   *
   * @return O servios que esto habilitados.
   */
  private List<Service> getEnabledServices() {
    List<Service> enabledServices = new LinkedList<Service>();
    for (Service service : servicesMap.values()) {
      if (service.isEnabled()) {
        enabledServices.add(service);
        Server.logFineMessage(String.format("O servio %s est habilitado",
          service.getName()));
      }
      else {
        Server.logInfoMessage(String.format("O servio %s est desabilitado",
          service.getName()));
      }
    }
    return enabledServices;
  }

  /**
   * Tenta inicializar uma lista de servios.
   *
   * @param services A lista de servios.
   *
   * @throws ServerException Caso ocorra algum erro na inicializao do
   *         servios.
   */
  private void tryInitServices(List<Service> services) throws ServerException {
    boolean anyServiceInitialized = false;
    for (Service service : services) {
      if (service.tryInit()) {
        anyServiceInitialized = true;
        this.sortedServices.add(service);
      }
    }
    if (!anyServiceInitialized) {
      StringBuilder errorMessage =
        new StringBuilder(
          "Os seguintes servios no puderam ser inicializados, "
            + "provavelmente por causa de uma dependncia cclica:");
      for (Service notInitedService : services) {
        errorMessage.append("\n" + notInitedService.getName());
      }
      Server.logSevereMessage(errorMessage.toString());
      throw new ServerException(errorMessage.toString());
    }
  }

  /**
   * Mtodo para terminar todos os servios cadastrados no gerente de servios.
   *
   * @throws ServerException caso ocorra alguma falha no trmino de servios
   */
  public void shutdownAllServices() throws ServerException {
    /*
     * Este mtodo pode ser chamado duas vezes (uma pelo KillServerService e
     * outra pelo ServiceLogManager).
     */
    if (this.sortedServices == null) {
      return;
    }

    Server.logInfoMessage("Finalizando servios...");

    // Desinstalando service manager.
    try {
      uninstall();
    }
    catch (RemoteException re) {
      throw new ServerException(re);
    }

    // Shutdown de servios na ordem reversa.
    for (int i = this.sortedServices.size() - 1; i >= 0; i--) {
      Service service = this.sortedServices.get(i);
      service.shutdown();
    }

    this.sortedServices = null;

    Server.logInfoMessage("Servios finalizados");
  }

  /**
   * Mtodo que retorna a instncia do gerente de servios (singleton).
   *
   * @return a instncia de <code>ServiceManager</code>.
   */
  public static ServiceManager getInstance() {
    return instance;
  }

  /**
   * Construtor privado que faz as inicializaes do singleton.
   */
  private ServiceManager() {
    servicesMap = new LinkedHashMap<String, Service>();
    usersProxiesMap = new Hashtable<Object, Map<String, ServiceInterface>>();
  }

  /**
   * Passa a observar eventos de {@link LoginService}.
   *
   * @throws RemoteException caso ocorra algum erro de rmi no cadastro do
   *         observador
   */
  public void install() throws RemoteException {
    IMessageListener listener = new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message message : messages) {
          LogoutEvent event = (LogoutEvent) message.getBody();
          final Object sessionKey = event.getSessionKey();
          shutdownProxies(sessionKey);
        }
      }

    };

    consumerId =
      MessageService.getInstance().setServerMessageListener(listener,
        new BodyTypeFilter(AdministrationEvent.class));
  }

  /**
   * Deixa de observar eventos de {@link LoginService}.
   *
   * @throws RemoteException casos ocorra alguma falha de rmi na remoo do
   *         observador
   */
  public void uninstall() throws RemoteException {
    MessageService.getInstance().clearServerMessageListener(consumerId);
  }

  /**
   * Mtodo para adicionar (cadastrar) um novo servio ao SSI.
   *
   * @param service novo servio para o cadatro.
   *
   * @throws IllegalArgumentException caso o <code>service</code> seja null
   */
  public void addService(final Service service) {
    if (service == null) {
      throw new IllegalArgumentException("service == <<null>>");
    }
    servicesMap.put(service.getName(), service);
  }

  /**
   * Cria os proxies de todos os servios disponveis para uma sesso de um
   * usurio logado. O mapa de proxies tem como chave os nomes dos servios
   * conforme as constantes <code>XxxxServiceInterface.SERVICE_NAME</code>.
   *
   * @param sessionKey Chave da sesso de um usurio.
   * @param servicesNames Nomes dos servios que sero recuperados.
   *
   * @return Mapa com os proxies dos servios disponveis. Se nenhum servio
   *         estiver disponvel  retornado um mapa com tamanho 0. Se a sesso
   *         for invlida  retornado <code>null</code>.
   *
   * @throws IllegalArgumentException caso o <code>sessionKey</code> ou o
   *         <code>servicesNames</code> seja null
   */
  public Map<String, ServiceInterface> getRemoteServices(Object sessionKey,
    Set<String> servicesNames) {
    if (sessionKey == null || servicesNames == null) {
      throw new IllegalArgumentException(
        "(sessionKey || servicesNames) == <<null>>");
    }
    if (!LoginService.getInstance().isValidSession(sessionKey)) {
      return null;
    }
    Map<String, ServiceInterface> remoteServices =
      new HashMap<String, ServiceInterface>();
    for (final String serviceName : servicesNames) {
      final ServiceInterface srv = getRemoteService(sessionKey, serviceName);
      if (srv != null) {
        remoteServices.put(serviceName, srv);
      }
    }
    return Collections.unmodifiableMap(remoteServices);
  }

  /**
   * Cria o proxy de um determinado servio para uma sesso de um usurio
   * logado. Se j existe um proxy do servio especificado para essa chave,
   * ento essa referncia  retornada.
   *
   * @param sessionKey Chave da sesso de um usurio.
   * @param serviceName Nome do Servio. Conforme constante
   *        <code>XxxxServiceInterface.SERVICE_NAME</code>.
   *
   * @return Proxy para o servio especificado. <code>null</code> caso o servio
   *         solicitado no esteja disponvel.
   *
   * @throws IllegalArgumentException caso o <code>sessionKey</code> ou o
   *         <code>serviceName</code> seja null
   */
  public ServiceInterface getRemoteService(Object sessionKey, String serviceName) {
    if (sessionKey == null || serviceName == null) {
      throw new IllegalArgumentException(
        "(sessionKey || serviceName) == <<null>>");
    }

    if (!LoginService.getInstance().isValidSession(sessionKey)) {
      return null;
    }

    /*
     * O servio de HTTP  um caso especial, pois ainda no foi migrado para o
     * padro de proxies para servios exportados. Por isso  retornado uma
     * referncia para a implementao do mesmo. Em breve, esse servio ter um
     * proxy.
     */
    if (serviceName.equals(HttpServiceInterface.SERVICE_NAME)) {
      return HttpService.getInstance();
    }
    synchronized (usersProxiesMap) {
      Map<String, ServiceInterface> proxies = usersProxiesMap.get(sessionKey);
      if (proxies == null) {
        proxies = new HashMap<String, ServiceInterface>();
      }
      ServiceInterface proxy = proxies.get(serviceName);
      if (proxy != null) {
        return proxy;
      }
      proxy = newProxy(sessionKey, serviceName);
      if (proxy == null) {
        return null;
      }
      proxies.put(serviceName, proxy);
      usersProxiesMap.put(sessionKey, proxies);
      return proxy;
    }
  }

  /**
   * Cria um proxy remoto para o servio indicado, utilizando o
   * {@link ServiceInvocationHandler} para tratar as chamadas aos mtodos do
   * servio, do lado do servidor.
   *
   * @param sessionKey Chave da sesso do usurio.
   * @param serviceName Nome do servio.
   *
   * @return O proxy para o servio ou <code>null</code> caso no exista proxy
   *         para esse servio.
   */
  private ServiceInterface newProxy(Object sessionKey, String serviceName) {
    try {
      IService service = Service.getInstance(serviceName);
      if (service == null) {
        return null;
      }

      /* Cria o InvocationHandler especfico do servio do servio. */
      InvocationHandler serviceHandler =
        ProxyUtils.createInvocationHandler(service);
      /*
       * Decora o InvocationHandler especfico do servio com o
       * InvocationHandler genrico de servios, responsvel por atribuir a
       * chave da sesso do usurio como sesso corrente antes de cada invocao
       * de mtodo e tratar as excees lanadas pelo servio.
       */
      serviceHandler = new ServiceInvocationHandler(sessionKey, serviceHandler);

      int port = Server.getInstance().getRMIExportPort();
      return (ServiceInterface) ProxyUtils.newRemoteProxyInstance(service,
        serviceHandler, port);
    }
    catch (Exception e) {
      final LoginService loginService = LoginService.getInstance();
      final User key = loginService.getUserByKey(sessionKey);
      Server.logSevereMessage("Falha ao criar proxy de " + serviceName
        + " para " + key, e);
      return null;
    }
  }

  /**
   * Finaliza os <i>proxies</i> para os servios do usurio especificado.
   *
   * @param sessionKey Chave para a sesso do usurio.
   */
  private void shutdownProxies(Object sessionKey) {
    final Map<String, ServiceInterface> proxies =
      usersProxiesMap.remove(sessionKey);
    if (proxies == null) {
      return;
    }
    final Set<Entry<String, ServiceInterface>> entries = proxies.entrySet();
    for (Entry<String, ServiceInterface> entry : entries) {
      final ServiceInterface proxy = entry.getValue();
      try {
        ProxyUtils.unexportRemoteProxy(proxy);
      }
      catch (Exception ex) {
        final String serviceName = entry.getKey();
        final String fmt = "Falha ao remover proxy de %s para %s";
        final String message = String.format(fmt, serviceName, sessionKey);
        Server.logSevereMessage(message);
      }
    }
  }

  /**
   * Obtm a instncia do servio.
   *
   * @param serviceName o nome do servio desejado.
   *
   * @return a instncia.
   */
  @Override
  public IService getService(final String serviceName) {
    return servicesMap.get(serviceName);
  }

  /**
   * Mtodo para obteno de todos os servios.
   *
   * @return Hashtable com as referncias para todos os servios.
   */
  public Map<String, Service> getServices() {
    return servicesMap;
  }

}
