package csbase.server.services.csfsservice;

import java.util.Map;
import java.util.Properties;

import sgaidl.SGA_CSFS_HOST;
import sgaidl.SGA_CSFS_PORT;
import sgaidl.SGA_NAME;
import csbase.server.ServerException;
import csbase.server.plugin.service.sgaservice.ISGADataTransfer;
import csbase.server.plugin.service.sgaservice.SGADataTransferException;

/**
 * Mecanismo de transferncia de dados entre o servidor e SGA que usa o servio
 * CSFS.
 *
 * @author Tecgraf/PUC-Rio
 */
public class CSFSDataTransferMechanism implements ISGADataTransfer {

  /**
   * As propriedades que o SGA forneceu sobre o mecanismo a ser adotado.
   */
  private Properties sgaProperties;

  /**
   * O endereo (ou IP) da mquina do CSFS que atende ao SGA.
   */
  private String csfsHost;

  /**
   * A porta do CSFS que atende ao SGA.
   */
  private Integer csfsPort;

  /**
   * {@inheritDoc}
   *
   */
  @Override
  public void setSGAProperties(Properties sgaProperties) {
    this.sgaProperties = sgaProperties;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws Exception {
    // Nada a fazer
  }

  /**
   * Obtm o host do CSFS.
   *
   * @return o host
   * @throws SGADataTransferException
   */
  private String getHost() throws SGADataTransferException {
    if (csfsHost != null) {
      return csfsHost;
    }
    CSFSService csfsService = CSFSService.getInstance();
    if ((csfsService == null) || !csfsService.isActive()) {
      throw new SGADataTransferException(
        "O servio CSFS no foi instanciado ou no est ativo.");
    }
    if (sgaProperties == null) {
      throw new SGADataTransferException(
        "As propriedades do SGA no esto disponveis.");
    }
    csfsHost = sgaProperties.getProperty(SGA_CSFS_HOST.value);
    if (csfsHost == null || csfsHost.isEmpty()) {
      csfsHost = null;
      throw new SGADataTransferException("A propriedade " + SGA_CSFS_HOST.value
        + " do SGA " + sgaProperties.getProperty(SGA_NAME.value)
        + " no est definida.");
    }
    return csfsHost;
  }

  /**
   * Obtm a porta do CSFS.
   *
   * @return a porta
   * @throws SGADataTransferException
   */
  private Integer getPort() throws SGADataTransferException {
    if (csfsPort != null) {
      return csfsPort;
    }
    CSFSService csfsService = CSFSService.getInstance();
    if ((csfsService == null) || !csfsService.isActive()) {
      throw new SGADataTransferException(
        "O servio CSFS no foi instanciado ou no est ativo.");
    }
    if (sgaProperties == null) {
      throw new SGADataTransferException(
        "As propriedades do SGA no esto disponveis.");
    }
    csfsPort = Integer.parseInt(sgaProperties.getProperty(SGA_CSFS_PORT.value));
    if (csfsPort == null) {
      throw new SGADataTransferException("A propriedade " + SGA_CSFS_PORT.value
        + " do SGA " + sgaProperties.getProperty(SGA_NAME.value)
        + " no est definida ou possui um valor invlido.");
    }
    return csfsPort;
  }

  /**
   * {@inheritDoc}
   *
   * Usa o mtodo {@link CSFSService#copyTo(String, int, String[], String[])}.
   */
  @Override
  public void copyTo(String[] sourcePath, String[] targetPath)
    throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      csfsService.copyTo(getHost(), getPort(), sourcePath, targetPath);
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }

  /**
   * {@inheritDoc}
   *
   * Usa o mtodo {@link CSFSService#copyFrom(String, int, String[], String[])}.
   */
  @Override
  public void copyFrom(String[] remoteFilePath, String[] localFilePath)
    throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      csfsService.copyFrom(getHost(), getPort(), remoteFilePath, localFilePath);
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }

  /**
   * {@inheritDoc}
   *
   * Usa o mtodo {@link CSFSService#createDirectory(String, int, String[])}.
   */
  @Override
  public void createDirectory(String[] path) throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      csfsService.createDirectory(getHost(), getPort(), path);
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }

  /**
   * {@inheritDoc}
   *
   * Usa o mtodo {@link CSFSService#remove(String, int, String[])}.
   */
  @Override
  public void remove(String[] path) throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      csfsService.remove(getHost(), getPort(), path);
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }

  /**
   * {@inheritDoc}
   *
   * Usa o mtodo {@link CSFSService#checkExistence(String, int, String[])}.
   */
  @Override
  public boolean checkExistence(String[] binaryDir)
    throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      return csfsService.checkExistence(getHost(), getPort(), binaryDir);
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getProjectsRootPath() throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      return csfsService.getCSFSProjectsRootPath();
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getAlgorithmsRootPath() throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      return csfsService.getCSFSAlgorithmsRootPath();
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String[], Long> getRemoteTimestamps(String[] sandboxPath)
    throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      return csfsService.getTimestamps(getHost(), getPort(), sandboxPath);
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String[], Long> getLocalTimestamps(String[] sandboxPath)
    throws SGADataTransferException {
    CSFSService csfsService = CSFSService.getInstance();
    try {
      return csfsService.getTimestamps(sandboxPath);
    }
    catch (ServerException e) {
      throw new SGADataTransferException(e);
    }
  }
}
