package csbase.server.services.opendreamsservice.opendreams.v1_04;

import java.rmi.RemoteException;
import java.util.Arrays;

import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.openbus.DRMAA.JobSubmissionState;
import tecgraf.openbus.opendreams.OpenDreamsJobTemplate;
import csbase.exception.ParseException;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.CommonClientProject;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.server.services.algorithmservice.AlgorithmService;
import csbase.server.services.projectservice.ProjectService;

/**
 * Template que descreve um <code>OpenDreamsJobTemplate</code> para execuo de
 * algoritmos no opendreams.  criado a partir de um job template.
 * 
 * @author Tecgraf PUC-Rio
 * 
 */
public class AlgorithmTemplate {

  /** O nome do algoritmo em args */
  @Option(name = "-name")
  private String algoName;

  /** A verso do algoritmo em args */
  @Option(name = "-version")
  private String algoVersion;

  /** O configurador do algoritmo descrito pelo job template */
  private AlgorithmConfigurator configurator;

  /** O identificador da verso do algoritmo */
  private AlgorithmVersionId versionId;

  /** O projeto sobre o qual esse job template atua */
  private CommonClientProject project;

  /** O identificador do usurio. */
  private String userId;

  /** Comandos vlidos */
  private static String[] VALID_COMMANDS = new String[] { "execAlgo",
      "execFlow" };

  /** Categorias de jobs vlidas */
  private static String[] VALID_JOB_CATEGORIES = new String[] { "CSBase",
      "System" };

  /**
   * Constri um validador de um <code>OpenDreamsJobTemplate</code>
   * 
   * @param project o projeto sobre o qual esse job template atua
   * @param userId o identificador do usurio.
   * @param jt o job template a ser validado
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  public AlgorithmTemplate(CommonClientProject project, String userId,
    OpenDreamsJobTemplate jt) throws AlgorithmTemplateException {
    this.project = project;
    this.userId = userId;

    validateRemoteCommand(jt);
    validateArgs(jt);
    validateNumberOfProcesses(jt);
    validateJobParameters(jt);
    validateJobSubmissionState(jt);
    validateJobEnvironment(jt);
    validateWorkingDirectory(jt);
    validateJobCategory(jt);
    validateNativeSpecification(jt);
    validateEmail(jt);
    validateBlockEmail(jt);
    validateJobName(jt);
    validateInputPath(jt);
    validateOutputPath(jt);
    validateErrorPath(jt);
    validateJoinFiles(jt);
    validateTransferFiles(jt);
    validateHardWallClockTimeLimit(jt);
    validateSoftWallClockTimeLimit(jt);
    validateHardRunDurationLimit(jt);
    validateSoftRunDurationLimit(jt);
  }

  /**
   * Faz a validao do atributo <b>remoteCommand</b> do job template. Os
   * seguintes valores so vlidos:
   * <ul>
   * <li><i>execAlgo</i>: executa um algoritmo.
   * <li><i>execFlow</i>: executa um fluxo de algoritmos.
   * </ul>
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateRemoteCommand(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    for (String command : VALID_COMMANDS) {
      if (jt.remoteCommand.equals(command)) {
        return;
      }
    }
    throw new AlgorithmTemplateException("Falha na validao de remoteCommand:"
      + jt.remoteCommand);
  }

  /**
   * Faz a validao do atributo <b>args</b> do job template.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateArgs(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    CmdLineParser parser = new CmdLineParser(this);
    try {
      parser.parseArgument(jt.args);
      this.versionId = AlgorithmVersionId.create(this.algoVersion);
      this.configurator =
        AlgorithmService.getInstance().createAlgorithmConfigurator(
          getAlgoName(), getAlgoVersionId());
    }
    catch (CmdLineException e) {
      StringBuffer args = new StringBuffer();
      for (String arg : jt.args) {
        args.append(arg);
        args.append(" ");
      }
      throw new AlgorithmTemplateException(
        "Falha na validao de args:" + args, e);
    }
    catch (AlgorithmNotFoundException e) {
      throw new AlgorithmTemplateException("Algoritmo " + getAlgoName()
        + " verso " + getAlgoVersionId() + " no encontrados", e);
    }
  }

  /**
   * Valida os parmetros para execuo do algoritmo.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateJobParameters(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    StringBuffer pars = new StringBuffer();
    for (String[] parameter : jt.jobParameters) {
      try {
        String name = parameter[0];
        String value = parameter[1];
        pars.append(name + " " + value + " ");
        configurator.setParameterValue(name, value);
      }
      catch (ParseException e) {
        throw new AlgorithmTemplateException("O parmetro " + pars
          + " possui o valor com formato invlido", e);
      }
      catch (ParameterNotFoundException e) {
        throw new AlgorithmTemplateException("O parmetro " + pars
          + " no existe na configurao do algoritmo", e);
      }
    }
    try {
      ValidationContext context =
        new ValidationContext(ValidationMode.FULL, project.getId(), userId);
      Validation result = configurator.validate(context);
      if (!result.isWellSucceded()) {
        LocalizedMessage message = result.getMessage();
        throw new AlgorithmTemplateException(
          "Validao dos parmetros do algoritmo falhou: " + message);
      }
    }
    catch (RemoteException e) {
      throw new AlgorithmTemplateException(
        "No foi possvel validar os parmetros do algoritmo", e);
    }
  }

  /**
   * Valida o atributo <code>jobSubmissionState</code> do job template. Apenas o
   * valor {@link JobSubmissionState#ACTIVE_STATE}  aceito no OpenDreams.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateJobSubmissionState(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (!jt.jobSubmissionState.equals(JobSubmissionState.ACTIVE_STATE)) {
      throw new AlgorithmTemplateException(
        "O atributo jobSubmissionState deve " + "ser ACTIVE_STATE");
    }
  }

  /**
   * Valida o atributo <code>jobEnvironment</code> do job template. Esse
   * atributo no  vlido no OpenDreams e, portanto, no deve possui nenhum
   * valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateJobEnvironment(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.jobEnvironment.length != 0) {
      throw new AlgorithmTemplateException("O atributo jobEnvironment no  "
        + "implementado pelo OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>workingDirectory</code> do job template. Esse
   * atributo no  vlido no OpenDreams e, portanto, no deve possui nenhum
   * valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateWorkingDirectory(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.workingDirectory != null && !jt.workingDirectory.isEmpty()) {
      throw new AlgorithmTemplateException("O atributo workingDirectory no  "
        + "implementado pelo OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>jobCategory</code> do job template. No OpenDreams,
   * os seguintes valores sero vlidos: CSBase ou System. A categoria CSBase
   * identifica a execuo de algoritmos do repositrio. A categoria System
   * serviria para atender a execuo de comandos do sistema, de acordo com o
   * uso comum do DRMAA. Por enquanto, o OpenDreams somente reconhece a
   * categoria CSBase.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateJobCategory(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.jobCategory != null
      && !Arrays.asList(VALID_JOB_CATEGORIES).contains(jt.jobCategory)) {
      throw new AlgorithmTemplateException(
        "O atributo jobCategory deve ser CSBase ou System");
    }
    if (jt.jobCategory != null && jt.jobCategory.equals("System")) {
      throw new AlgorithmTemplateException(
        "A categoria System ainda no est implementada no OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>nativeSpecification</code> do job template. Esse
   * atributo no  vlido no OpenDreams e, portanto, no deve possui nenhum
   * valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateNativeSpecification(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.nativeSpecification != null && !jt.nativeSpecification.isEmpty()) {
      throw new AlgorithmTemplateException(
        "O atributo nativeSpecification no  "
          + "implementado pelo OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>numberOfProcesses</code> do job template. Esse
   * atributo possui o nmero de processos e  utilizado apenas nos modos de
   * execuo PARALELO ou DISTRIBUDO. Na verso atual, esse atributo no  mais
   * suportado.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateNumberOfProcesses(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.numberOfProcesses != 0) {
      throw new AlgorithmTemplateException(
        "O atributo numberOfProcesses no  "
          + "implementado na verso atual do sistema CSBase");
    }
  }

  /**
   * Valida o atributo <code>email</code> do job template. Esse atributo possui
   * a lista de emails para notificar o trmino de execuo dos jobs
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateEmail(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
  }

  /**
   * Valida o atributo <code>blockEmail</code> do job template. Esse atributo
   * indica o bloqueio ou no no envio de emails sobre o estado (trmino) dos
   * jobs.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateBlockEmail(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
  }

  /**
   * Valida o atributo <code>jobName</code> do job template. Esse atributo no 
   * vlido no OpenDreams e, portanto, no deve possui nenhum valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateJobName(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.jobName != null && !jt.jobName.isEmpty()) {
      throw new AlgorithmTemplateException("O atributo jobName no  "
        + "implementado pelo OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>inputPath</code> do job template. Esse atributo no
   *  vlido no OpenDreams e, portanto, no deve possui nenhum valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateInputPath(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.inputPath != null && !jt.inputPath.isEmpty()) {
      throw new AlgorithmTemplateException("O atributo inputPath no  "
        + "implementado pelo OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>outputPath</code> do job template.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateOutputPath(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    String dirPath = FileUtils.getFilePath(jt.outputPath);
    String[] paths = FileUtils.splitPath(dirPath);
    if (paths.length > 0) {
      if (!ProjectService.getInstance().existsFile(project.getId(), paths)) {
        throw new AlgorithmTemplateException("O diretrio " + dirPath
          + " no existe no projeto");
      }
    }
  }

  /**
   * Valida o atributo <code>errorPath</code> do job template.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateErrorPath(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    // TODO Auto-generated method stub
  }

  /**
   * Valida o atributo <code>joinFiles</code> do job template.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateJoinFiles(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    // TODO Auto-generated method stub
  }

  /**
   * Valida o atributo <code>transferFiles</code> do job template.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateTransferFiles(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    // TODO Auto-generated method stub
  }

  /**
   * Valida o atributo <code>hardWallClockTimeLimit</code> do job template. Esse
   * atributo no  vlido no OpenDreams e, portanto, no deve possui nenhum
   * valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateHardWallClockTimeLimit(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.hardWallClockTimeLimit != 0) {
      throw new AlgorithmTemplateException(
        "O atributo hardWallClockTimeLimit no  "
          + "implementado pelo OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>softWallClockTimeLimit</code> do job template. Esse
   * atributo no  vlido no OpenDreams e, portanto, no deve possui nenhum
   * valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateSoftWallClockTimeLimit(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.softWallClockTimeLimit != 0) {
      throw new AlgorithmTemplateException(
        "O atributo softWallclockTimeLimit no  "
          + "implementado pelo OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>hardRunDurationLimit</code> do job template. Esse
   * atributo no  vlido no OpenDreams e, portanto, no deve possui nenhum
   * valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateHardRunDurationLimit(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.hardRunDurationLimit != 0) {
      throw new AlgorithmTemplateException(
        "O atributo hardRunDurationLimit no  "
          + "implementado pelo OpenDreams");
    }
  }

  /**
   * Valida o atributo <code>softRunDurationLimit</code> do job template. Esse
   * atributo no  vlido no OpenDreams e, portanto, no deve possui nenhum
   * valor.
   * 
   * @param jt o job template que possui os argumentos a serem validados
   * @throws AlgorithmTemplateException se houver uma falha na validao do job
   *         template
   */
  private void validateSoftRunDurationLimit(OpenDreamsJobTemplate jt)
    throws AlgorithmTemplateException {
    if (jt.softRunDurationLimit != 0) {
      throw new AlgorithmTemplateException(
        "O atributo softRunDurationLimit no  "
          + "implementado pelo OpenDreams");
    }
  }

  /**
   * Obtm o nome do algoritmo, proveniente do argumento -name em args.
   * 
   * @return o nome do algoritmo
   */
  public String getAlgoName() {
    return this.algoName;
  }

  /**
   * Obtm a verso do algoritmo, proveniente do argumento -version em args.
   * 
   * @return a verso do algoritmo
   */
  public AlgorithmVersionId getAlgoVersionId() {
    return this.versionId;
  }

  /**
   * Obtm o configurador do algoritmo correspondente ao job template.
   * 
   * @return o configurador do algoritmo criado a partir do job template.
   */
  public AlgorithmConfigurator getAlgorithmConfigurator() {
    return this.configurator;
  }
}
