package csbase.server.services.opendreamsservice.opendreams.v2_0;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;

import csbase.exception.PermissionException;
import csbase.logic.*;
import csbase.server.Service;
import csbase.server.services.sgaservice.SGAService;
import org.omg.CORBA.AnyHolder;
import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAPackage.ServantNotActive;
import org.omg.PortableServer.POAPackage.WrongPolicy;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.openbus.DRMAA.v2_0.*;
import tecgraf.openbus.DRMAA.v2_0.UnsupportedOperationException;
import tecgraf.openbus.opendreams.v2_0.FinalizationType;
import tecgraf.openbus.opendreams.v2_0.OpenDreamsJobInfo;
import tecgraf.openbus.opendreams.v2_0.OpenDreamsJobInfoImpl;
import tecgraf.openbus.opendreams.v2_0.OpenDreamsJobTemplate;
import csbase.exception.OperationFailureException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.FileParameterValue;
import csbase.server.Server;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.schedulerservice.SchedulerService;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import csbase.util.messages.filters.BodyTypeFilter;

/**
 * Every job in the JobSession is represented by its own instance of the Job
 * interface. It allows one to instruct the DRM system of a job status change,
 * and to query the properties of the job in the DRM system.
 *
 * The job control functions allow modifying the status of a single job in the
 * DRM system, according to the DRMAA 2 state model. If the job is in an
 * inappropriate state for the particular method call, it MUST raise an
 * InvalidStateException.
 *
 * @author Tecgraf/PUC-Rio
 */
@SuppressWarnings("serial")
public class OpenDreamsJob implements JobOperations, Serializable {

	/**
	 * O Job terminou mas não tem um codigo de saída
	 */
	public final static int NO_EXIT_CODE = 256; // Acima de 255 não é um exit
												// code válido

	/**
	 * Tempo de sleep entre cada tentativa de terminar o job (em ms).
	 */
	private static final int TERMINATE_SLEEP_TRY = 50;
	/**
	 *
	 */
	private String jobId;

	/**
	 *
	 */
	private String sessionName;

	/**
	 *
	 */
	private String userId;

	/**
	 *
	 */
	private Object projectId;

	/**
	 *
	 */
	private OpenDreamsJobTemplate jt;

	/**
	 *
	 */
	private OpenDreamsJobInfo jobInfo;

	/**
	 *
	 */
	private transient Job ref;

	/**
	 * @param sessionName
	 * @param jt
	 */
	public OpenDreamsJob(String sessionName, OpenDreamsJobTemplate jt) {
		this.sessionName = sessionName;
		this.jt = jt;
		this.jobInfo = new OpenDreamsJobInfoImpl();
		// this.jobInfo.jobSubState =
		// OpenBusService.getInstance().getORB().create_any();
	}

	/**
	 * {@inheritDoc}
	 *
	 * This attribute reports the job identifier assigned by the DRM system in
	 * text form. This method is expected to be used as a fast alternative to
	 * the fetching of a complete JobInfo instance.
	 */
	@Override
	public String jobId() {
		return jobId;
	}

	/**
	 * {@inheritDoc}
	 *
	 * This attribute reports the name of the JobSession that was used to create
	 * the job.
	 */
	@Override
	public String sessionName() {
		return sessionName;
	}

	/**
	 * {@inheritDoc}
	 *
	 * This attribute provides a reference to a JobTemplate instance that has
	 * equal values to the one that was used for the job submission creating
	 * this Job instance.
	 */
	@Override
	public JobTemplate jobTemplate() {
		return jt;
	}

	/**
	 * @return a stub object that references the actual server's object
	 * @throws OperationFailureException
	 * @throws ServantNotActive
	 * @throws WrongPolicy
	 */
	public Job createCorbaObjReference() throws OperationFailureException, ServantNotActive, WrongPolicy {
		System.out.println("Job createCorbaObjReference()");
		POA poa = OpenBusService.getInstance().getRootPOA();
		if (ref == null) {
			System.out.println("ref == null");
			JobPOATie tie = new JobPOATie(this, poa);
			org.omg.CORBA.Object obj = poa.servant_to_reference(tie);
			ref = JobHelper.narrow(obj);
		}
		System.out.println("Returning Job ref");
		return ref;
	}

	/**
	 *
	 * @param userId
	 *            o identificador do usuário que fez a submissão
	 * @throws InvalidJobTemplateException
	 *             se houver um erro na descrição do job template
	 */
	public void execute(String userId) throws InvalidJobTemplateException {
		// Faz a validação dos atributos do job template
		JobTemplateValidator validator;
		validator = new JobTemplateValidator(userId, jt);
		this.projectId = validator.getProjectId();
		this.userId = userId;
		final AlgorithmConfigurator configurator = validator.getAlgorithmConfigurator();

		ProjectService.getInstance().openProject(projectId, false);

		// Cria o comando de submissão de job
		CommandSubmission submission = new CommandSubmission(configurator, projectId);
		submission.setDescription(jt.jobDescription);
		submission.setPriority(Priority.values()[jt.priority]);
		submission.setEmailList(jt.email);
		submission.setMailAtEnd(jt.emailOnTerminated);
		// Atualmente assume-se que só tenha uma máquina candidata
		submission.configureSimpleExecution(
				(jt.candidateMachines != null && jt.candidateMachines.length == 0) ? null : jt.candidateMachines[0]);
		int nProp = jt.extraProperties.length;
		int n = 0;
		for (; n < nProp; n++) {
			submission.addExtraInfo(jt.extraProperties[n][0], jt.extraProperties[n][1]);
		}

		Set<CommandInfo> commandInfos = null;
		try {
			commandInfos = SchedulerService.getInstance().submitCommand(submission);
		} catch (RemoteException e) {
		}
		String cmdId = commandInfos.toArray(new CommandInfo[0])[0].getId();
		System.out.println("Job " + cmdId + " scheduled to run.");
		this.jobId = cmdId;
		this.jobInfo.jobId = cmdId;
		this.jobInfo.jobOwner = userId;
		this.jobInfo.jobName = jt.jobName;
	}

	/**
	 * {@inheritDoc}
	 *
	 * Triggers a transition from RUNNING to SUSPENDED state.
	 */
	@Override
	public void suspend() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operação não suportada.");
	}

	/**
	 * {@inheritDoc}
	 *
	 * Triggers a transition from SUSPENDED to RUNNING state.
	 */
	@Override
	public void resume() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operação não suportada.");
	}

	/**
	 * {@inheritDoc}
	 *
	 * Triggers a transition from QUEUED to QUEUED_HELD, or from REQUEUED to
	 * REQUEUED_HELD state.
	 */
	@Override
	public void hold() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operação não suportada.");
	}

	/**
	 * {@inheritDoc}
	 *
	 * Triggers a transition from QUEUED_HELD to QUEUED, or from REQUEUED_HELD
	 * to REQUEUED state.
	 */
	@Override
	public void release() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operação não suportada.");
	}

	/**
	 * {@inheritDoc}
	 *
	 * Triggers a transition from any of the "Started" states to one of the
	 * "Terminated" states.
	 */
	@Override
	public void terminate() throws DrmCommunicationException, DeniedByDrmsException, InternalException {

		System.out.println("Job " + jobId + " being terminated.");
		Server.logInfoMessage("Job " + jobId + " being terminated.");

		try {
			String userId = OpenBusService.getInstance().initCSBaseAccess();
			SessionManager.checkUser(userId);

			CommandPersistenceService commandPersistenceService = CommandPersistenceService.getInstance();

			CommandInfo commandInfo = commandPersistenceService.getCommandInfo(projectId, jobId);

			if (commandInfo.getStatus().equals(CommandStatus.FINISHED)) {
				Server.logInfoMessage("O comando " + jobId + " já terminou e não pode mais ser cancelado");
			}
			SchedulerService schedulerService = SchedulerService.getInstance();

			// Tenta remover o comando da fila
			if (schedulerService.removeCommand(jobId)) {
				return;
			}
			while (true) {
				commandInfo = commandPersistenceService.getCommandInfo(projectId, jobId);
				if (commandInfo.getStatus().equals(CommandStatus.FINISHED)) {
					// Comando terminou e não pode mais ser cancelado
					return;
				}
				if (commandInfo.getStatus().equals(CommandStatus.EXECUTING)) {
					SGAService sgaService = SGAService.getInstance();
					if (sgaService.killCommand(jobId)) {
						return;
					}
					if (!sgaService.getSGASet(commandInfo.getSGAName()).getAlive()) {
						throw new DrmCommunicationException(
								"O servidor perdeu a conexão com o SGA " + commandInfo.getSGAName());
					}
				}
				try {
					Thread.sleep(TERMINATE_SLEEP_TRY);
				} catch (InterruptedException e) {
				}
			}
		} catch (PermissionException e) {
			throw new DeniedByDrmsException(
					"O usuário " + userId + " não possui permissão para cancelar o job " + jobId);
		} catch (DeniedByDrmsException e) {
			e.printStackTrace();
			String msg = "O usuário " + userId + " não possui permissão para cancelar o job " + jobId;
			Server.logSevereMessage(msg, e);
			throw e;
		} catch (DrmCommunicationException e) {
			String msg = "Falha na tentativa de alterar o estado de um comando pelo usuário " + userId + ": "
					+ e.message;
			Server.logWarningMessage(msg);
			throw e;
		} catch (Throwable e) {
			String msg = "Erro no cancelamento do job " + jobId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} finally {
			OpenBusService.getInstance().finishCSBaseAccess();
		}
	}

  @Override
  public void reap() throws UnsupportedOperationException {
    throw new UnsupportedOperationException("Operação não suportada.");
  }


	/**
	 * Handle required actions after the job enters one of the "Terminated"
	 * states.
	 */
	public void onFinish(String userId, CommandNotification cmdEvent) {

		this.jobInfo.jobState = JobState.DONE;
		// TODO: pegar submissionTime e finishTime
		if (cmdEvent.getFinalizationInfo().getFinalizationType() == CommandFinalizationType.SUCCESS) {
			CommandSuccessNotification endNotification = (CommandSuccessNotification) cmdEvent;
			System.out.println(endNotification);
			if (endNotification.getCPUTime() != null) {
				this.jobInfo.cpuTime = Integer.valueOf(endNotification.getCPUTime().toString());
			}
			if (endNotification.getElapsedTime() != null) {
				this.jobInfo.wallclockTime = Long.valueOf(endNotification.getElapsedTime().toString());
			}
			if (endNotification.getUserTime() != null) {
			}

			Map<String, String> execData = endNotification.getExecutionData();
			if (execData != null) {
				jobInfo.executionData = new String[execData.size()][2];
				Iterator<Map.Entry<String, String>> iterator = execData.entrySet().iterator();
				int index = 0;
				while (iterator.hasNext()) {
					Map.Entry<String, String> data = iterator.next();
					jobInfo.executionData[index] = new String[]{data.getKey(), data.getValue()};
					index++;
				}
			}
		}

		if (cmdEvent.getFinalizationInfo().getInfoType() == CommandFinalizationInfo.FinalizationInfoType.EXTENDED) {
			ExtendedCommandFinalizationInfo finalizationInfo = (ExtendedCommandFinalizationInfo) cmdEvent
					.getFinalizationInfo();
			jobInfo.guiltyNodeId = String.valueOf(finalizationInfo.getGuiltyNodeId());
		}

		CommandFinalizationInfo finalizationInfo = cmdEvent.getFinalizationInfo();
		if (finalizationInfo == null) {
			Server.logSevereMessage("A informação de finalizacao do comando " + jobId + " não existe (null)");
			jobInfo.exitStatus = NO_EXIT_CODE;
		} else {
			Integer exitCode = finalizationInfo.getExitCode();
			Server.logInfoMessage("A informação de finalizacao do comando " + jobId + " possui exitCode=" + exitCode);
			jobInfo.exitStatus = exitCode == null ? NO_EXIT_CODE : exitCode;
			jobInfo.finalizationType = getFinalizationType(finalizationInfo.getFinalizationType(),
					finalizationInfo.getFailureCause());
			jobInfo.description = finalizationInfo.getFailureCause() == null
					? finalizationInfo.getFinalizationType().getDescription()
					: finalizationInfo.getFailureCause().getDescription();
		}

		jobInfo.terminatingSignal = "";

		this.flush(userId);

	}

	/**
	 * {@inheritDoc}
	 *
	 * This method allows the application to get the current status of the job
	 * according to the DRMAA state model, together with an implementation
	 * specific sub state (see Section 8.1). It is intended as a fast
	 * alternative to the fetching of a complete JobInfo instance.
	 */
	@Override
	public JobState getState(AnyHolder jobSubState) {
		return jobInfo.jobState;
	}

	/**
	 * {@inheritDoc}
	 *
	 * This method returns a JobInfo instance for the particular job.
	 */
	@Override
	public JobInfo getInfo() {
		return jobInfo;
	}

	/**
	 * {@inheritDoc}
	 *
	 * Blocks until the job entered one of the "Started" states.
	 */
	@Override
	public void waitStarted(long timeout) throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operação não suportada.");
	}

	/**
	 * {@inheritDoc}
	 *
	 * Blocks until the job entered one of the "Terminated" states
	 */
	@Override
	public void waitTerminated(long timeout) throws TimeoutException, DeniedByDrmsException, InternalException {
		final SynchronousQueue<Boolean> onJobTerminate = new SynchronousQueue<Boolean>();

		Server.logFineMessage("OpenDreamsJob (" + jobId + "): waitTerminated");
		System.out.println("OpenDreamsJob (" + jobId + "): waitTerminated");

		try {
			userId = OpenBusService.getInstance().initCSBaseAccess();
			SessionManager.checkUser(userId);

			Serializable listenerId = setOnJobTerminateListener(onJobTerminate);
			Boolean terminated = onJobTerminate.poll(timeout, TimeUnit.SECONDS);
			MessageService.getInstance().clearServerMessageListener(listenerId);
			if (terminated != null && terminated) {
				return;
			} else {
				throw new TimeoutException("O job não terminou dentro do timeout especificado.");
			}
		} catch (DeniedByDrmsException e) {
			e.printStackTrace();
			String msg = "Erro de autenticação no CSBASE do usuário " + userId;
			Server.logSevereMessage(msg, e);
			throw e;
		} catch (TimeoutException e) {
			e.printStackTrace();
			String msg = "Timeout excedido na espera por finalização do job " + jobId;
			Server.logSevereMessage(msg, e);
			throw e;
		} catch (InterruptedException e) {
			e.printStackTrace();
			String msg = "Erro na espera por finalização do job " + jobId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} catch (Throwable e) {
			String msg = "Erro na espera por finalização do job " + jobId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} finally {
			OpenBusService.getInstance().finishCSBaseAccess();
		}
	}

	/**
	 * Registra o ouvinte de mensagens.
	 *
	 * @param terminated
	 * @return o identificador do listener setado
	 */
	private Serializable setOnJobTerminateListener(final SynchronousQueue<Boolean> terminated) {
		return MessageService.getInstance().setServerMessageListener(new IMessageListener() {
			@Override
			public void onMessagesReceived(Message... messages) throws Exception {
				for (Message message : messages) {
					CommandNotification notification = (CommandNotification) message.getBody();
					String cmdId = (String) notification.getCommandId();
					System.out.println(jobId + " onJobTerminateListener: onMessageReceived " + cmdId);

					if (jobId.equals(cmdId)) {
						boolean handled = terminated.offer(true);
						System.out.println("Job " + cmdId + " offer accepted: " + handled);
					}
				}
			}
		}, new BodyTypeFilter(CommandNotification.class));
	}

	/**
	 * Manipulação dos arquivos e dados de saída do job.
	 *
	 * @param userId
	 */
	public void flush(String userId) {
		try {
			if (jt.outputPath != null && !jt.outputPath.isEmpty()) {

				String projectOwner = jt.projectOwner.isEmpty() ? userId : jt.projectOwner;
				Object projectId = ProjectService.getInstance().getProjectId(projectOwner, jt.projectName);

				CommandInfo info = CommandPersistenceService.getInstance().getCommandInfo(projectId, jobId);

				// Cria o arquivo de saída definido em outputFilePath
				try {
					Set<FileParameterValue> files = info.getConfigurator().getStandardOutputFiles();
					dump(projectId, files, jt.outputPath);
				} catch (RemoteException e) {
					// Essa exceção não deve ser lançada já que não é uma
					// chamada remota
					Server.logSevereMessage("Erro na recuperação do configurador do comando.", e);
				}
			}
		} catch (Throwable e) {
			// Se ocorrer algum erro durante o tratamento do arquivo de saída
			Server.logSevereMessage("Erro na recuperação do log de saída do comando.", e);
		}
	}

	/**
	 * Cria os arquivos de saída.
	 *
	 * @param projectId
	 *            identificador do projeto associado ao job
	 * @param stdOutputFiles
	 *            lista de arquivos para criar
	 * @param outputFile
	 *            diretório onde criar os arquivos
	 *
	 * @return true se os arquivos foram criados e false caso contrário
	 */
	private boolean dump(Object projectId, Set<FileParameterValue> stdOutputFiles, String outputFile) {
		ProjectService projectService = ProjectService.getInstance();
		String[] outputFilePath = FileUtils.splitPath(outputFile);
		// Remove o arquivo definido em outputFilePath se já existir um com o
		// mesmo nome
		if (projectService.existsFile(projectId, outputFilePath)) {
			projectService.removeFile(projectId, outputFilePath);
		}
		// Caminho para o diretório do arquivo de saída
		String[] outputDirPath = FileUtils.splitPath(FileUtils.getFilePath(outputFile));

		// Nome do arquivo que deve ser criado no diretório do projeto
		String newFileName = FileUtils.getFileName(outputFile);

		// Cria o arquivo de output de saída
		projectService.createFile(projectId, outputDirPath, newFileName, "TEXT");
		ClientProjectFile cpf = projectService.getChild(projectId, outputDirPath, newFileName);

		// Transfere os arquivos de saída do comando para o arquivo outputFile
		OutputStream fileTo = null;
		try {
			fileTo = projectService.getOutputStream(projectId, outputFilePath);
			PrintWriter printWriter = new PrintWriter(new OutputStreamWriter(fileTo));
			for (final FileParameterValue stdOutputFile : stdOutputFiles) {
				String[] stdOutputFilePath = stdOutputFile.getPathAsArray();
				InputStream fileFrom = null;
				try {
					if (stdOutputFiles.size() > 1) {
						printWriter.printf("\n---- %s ----\n", stdOutputFile.getPath());
					}
					if (!projectService.existsFile(projectId, stdOutputFilePath)) {
						continue;
					}
					fileFrom = projectService.getInputStream(projectId, stdOutputFilePath);
					BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(fileFrom));
					String line = bufferedReader.readLine();
					while (line != null) {
						printWriter.println(line);
						line = bufferedReader.readLine();
					}
					printWriter.flush();
				} catch (IOException e) {
					Server.logSevereMessage(
							"Erro na gravação do arquivo " + stdOutputFile.getPath() + " em " + cpf.getStringPath(), e);
				} catch (Exception e) {
					Server.logSevereMessage("Erro na recuperação do inputstream do arquivo " + stdOutputFile.getPath(),
							e);
				} finally {
					try {
						if (fileFrom != null) {
							fileFrom.close();
						}
					} catch (IOException e) {
						Server.logSevereMessage(
								"Erro no fechamento do inputStream do arquivo " + stdOutputFile.getPath(), e);
					}
				}
			}
		} catch (Exception e) {
			Server.logSevereMessage("Erro na recuperação do outputsream do arquivo de saída " + cpf.getStringPath(), e);
			return false;
		} finally {
			if (fileTo != null) {
				try {
					fileTo.close();
				} catch (IOException e) {
					Server.logSevereMessage(
							"Erro no fechamento do outputsream do arquivo de saída " + cpf.getStringPath(), e);
				}
			}
		}
		return true;
	}

	/**
	 * Mapeia o tipo de finalização do comando do CSBase para o tipo de
	 * finalização de comando definido pela IDL do OpenDreams.
	 *
	 * @param type
	 *            o tipo de finalização do comando no CSBase
	 * @param cause
	 *            a causa da falha no caso do comando ou null caso o comando não
	 *            tenha falhado.
	 * @return o tipo de finalização do comando mapeado para o OpenDreams
	 */
	private FinalizationType getFinalizationType(CommandFinalizationType type, FailureFinalizationType cause) {
		switch (type) {
			case END :
				return FinalizationType.EXIT_CODE_IGNORED;
			case SUCCESS :
				return FinalizationType.EXIT_CODE_SUCCESS;
			case EXECUTION_ERROR :
				return FinalizationType.EXIT_CODE_ERROR;
			case NO_EXIT_CODE :
				return FinalizationType.EXIT_CODE_FAILED;
			case FAILED :
				switch (cause) {
					case UNKNOWN :
						return FinalizationType.UNKNOWN;
					case COMMAND_IDENTIFIER_NOT_FOUND :
						return FinalizationType.COMMAND_IDENTIFIER_NOT_FOUND;
					case SGA_EXECUTION_ERROR :
						return FinalizationType.SGA_EXECUTION_ERROR;
					case CSFS_SERVICE_UNAVAILABLE :
						return FinalizationType.CSFS_SERVICE_UNAVAILABLE;
					case FAILED_SETUP_EXECUTION_ENVIRONMENT :
						return FinalizationType.FAILED_SETUP_EXECUTION_ENVIRONMENT;
					case NO_SGA_AVAILABLE_TO_ROOT_COMMAND :
						return FinalizationType.NO_SGA_AVAILABLE_TO_ROOT_COMMAND;
					case SGA_IS_NOT_AVAILABLE :
						return FinalizationType.SGA_IS_NOT_AVAILABLE;
					case PROJECT_NOT_FOUND :
						return FinalizationType.PROJECT_NOT_FOUND;
					case USER_WITHOUT_PERMISSION_FOR_EXECUTION :
						return FinalizationType.USER_WITHOUT_PERMISSION_FOR_EXECUTION;
				}
			case KILLED :
				return FinalizationType.KILLED;
			case LOST :
				return FinalizationType.LOST;
			default :
				return FinalizationType.UNDEFINED;
		}
	}
}
