package csbase.server.services.schedulerservice.heuristic;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import csbase.logic.CommandInfo;
import csbase.logic.SGASet;

/**
 * Heurstica baseada no artigo Simple and Integrated Heuristic Algorithms
 * for Scheduling Tasks with Time and Resource Constraints. Zhao, Wei;
 * Ramamritham. The Journal of Systems and Software 7, 195-207 (1987)
 * 
 * Se houver uma alocao factvel, retorna a primeira alocao factvel que
 * a heurstica encontrar, sem qualquer garantia de que a alocao seja
 * tima.
 * 
 * Em caso de no haver alocao factvel, retorna uma alocao
 * parcial, garantidamente tima.
 * 
 * Nota: desempenho inferior a HeuristicApplyer
 */

public class FeasibleWithTreeHeuristicApplyer {
	
	

	Comparator<CommandInfo> commandComparator; 
	Comparator<Allocation> allocationComparator; // TODO: avaliar se devemos ter
													// um default
	private List<CommandInfo> commands;
	private List<SGASet> servers;
	private List<SGASet> simulatedServers;
	private List<CommandInfo> remainingCommands = new LinkedList<CommandInfo>();
	private HashMap<String, SGASet> originalServers = new HashMap<String, SGASet>();
	private ResourceControllerInterface manager;
	private Allocation allocation;
	private AllocationStorage storage = new AllocationStorage();

	public FeasibleWithTreeHeuristicApplyer(Comparator<CommandInfo> commandComparator,
			Comparator<Allocation> allocationComparator,
			List<CommandInfo> commands, List<SGASet> servers,
			ResourceControllerInterface manager) {
		this.commandComparator = commandComparator;
		this.allocationComparator = allocationComparator;
		this.commands = commands;
		this.servers = servers;
		this.simulatedServers = new ArrayList<SGASet>(servers);
		this.manager = manager;
		applyHeuristic();
	}

	public FeasibleWithTreeHeuristicApplyer(Comparator<Allocation> allocationComparator,
			List<CommandInfo> commands, List<SGASet> servers,
			ResourceControllerInterface manager) {
		this.allocationComparator = allocationComparator;
		this.commands = commands;
		this.servers = servers;
		this.simulatedServers = new ArrayList<SGASet>(servers);
		this.manager = manager;
		applyHeuristic();
	}

	private void applyHeuristic() {

		Map<CommandInfo, SGASet> currentAllocation = new HashMap<CommandInfo, SGASet>();
		remainingCommands.addAll(commands);

		for (SGASet sga : servers) {
			originalServers.put(sga.getName(), sga);
		}
		if (chooseCommand(currentAllocation)) {
			this.allocation = new Allocation(currentAllocation,
					simulatedServers, remainingCommands, true, this.manager,
					this.originalServers);
		}
	}

	private boolean chooseCommand(Map<CommandInfo, SGASet> currentAllocationMap) {
		List<CommandInfo> cmdsCopy = new ArrayList<CommandInfo>(
				remainingCommands);
		if (this.commandComparator != null)
			Collections.sort(cmdsCopy, this.commandComparator);
		for (CommandInfo cmd:cmdsCopy)
			if (tryCommand(cmd, currentAllocationMap))
				return true;
		return false;
	}
	
	private boolean tryCommand(CommandInfo cmd, Map<CommandInfo, SGASet> currentAllocationMap) {
		remainingCommands.remove(cmd);
		List<SGASet> possibleAllocations = manager
				.getServersThatMeetsRequirements(cmd, simulatedServers);
		if (possibleAllocations.isEmpty()) {
			System.out.println("No server found for command.");
			Allocation currentAllocation = new Allocation(
					currentAllocationMap, simulatedServers,
					remainingCommands, false, this.manager,
					this.originalServers);
			if (allocation == null
					|| allocation.compareTo(currentAllocation) == -1
					|| (allocation.compareTo(currentAllocation) == 0 && allocationComparator
							.compare(allocation, currentAllocation) == -1)) {
				allocation = currentAllocation;
			}
			remainingCommands.add(cmd);
			return false;
		}
		for (SGASet sga : possibleAllocations) {
			boolean successful = tryAllocation(currentAllocationMap, cmd,
					sga);
			if (successful)
				return true;
		}
		remainingCommands.add(cmd);
		return false;
	}

	private boolean tryAllocation(
			Map<CommandInfo, SGASet> currentAllocationMap, CommandInfo cmd,
			SGASet server) {
		currentAllocationMap.put(cmd, originalServers.get(server.getName()));
		
		if (!storage.hasAllocation(currentAllocationMap))
		{
			storage.addAllocation(currentAllocationMap);
			// simula uso de recursos
			SGASet consumedSGA = manager.simulateResourceConsumption(cmd, server);
			simulatedServers.remove(server);
			simulatedServers.add(consumedSGA);

			if (remainingCommands.isEmpty() || chooseCommand(currentAllocationMap))
				return true;

			// desfaz simulao de uso de recursos
			simulatedServers.remove(consumedSGA);
			simulatedServers.add(server);
			manager.undoResourceConsumptionSimulation(cmd, server);
		}	
		currentAllocationMap.remove(cmd);
		return false;
	}

	public Allocation getAllocation() {
		return this.allocation;
	}

}

class AllocationStorage {
	
	Map<Integer, List<Map<CommandInfo, SGASet>>> storage = new HashMap<Integer, List<Map<CommandInfo, SGASet>>>();
	
	public void addAllocation(Map<CommandInfo, SGASet> allocationMap)
	{
		List<Map<CommandInfo, SGASet>> allocations = storage.get(allocationMap.size());
		if (allocations == null)
		{
			allocations = new LinkedList<Map<CommandInfo,SGASet>>();
			storage.put(allocationMap.size(), allocations);
		}
		allocations.add(new HashMap<CommandInfo, SGASet>(allocationMap));
	}
	
	public boolean hasAllocation(Map<CommandInfo, SGASet> allocationMap)
	{
		List<Map<CommandInfo, SGASet>> allocations = storage.get(allocationMap.size());
		if (allocations == null)
			return false;
		for (Map<CommandInfo, SGASet> alloc:allocations)
			if (alloc.equals(allocationMap))
				return true;
		return false;
	}
}