/**
 * $Id$
 */

package csbase.servlet;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.URL;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Servlet responsvel por construir uma pgina JNLP para carregar uma aplicao
 * via Java Web Start.
 *
 * A pgina construda  uma cpia de um pgina original, especfica para cada
 * aplicao, na qual so modificados alguns parmetros e includos outros.
 * Esses parmetros so passados atravs da requisio (
 * {@code HttpServletRequest}) ou so obtidos do {@link #PROPERTIES_FILE
 * arquivo de propriedades} (que deve ser copiado para o diretrio
 * {@code WEB-INF} pelo script de inicializao do sistema).
 *
 * A pgina original deve existir na mesma URL base deste servlet. O nome da
 * pgina pode ser fornecido atravs do parmetro <i>page</i>. Se o nome no for
 * fornecido,  assumido <i>index.jnlp</i>.
 *
 * Este servlet trata os seguintes marcadores (em parnteses o nome da
 * propriedade correspondente, quando aplicvel):
 * <ul>
 * <li> {@code server_host_name} ({@code Server.hostName}) : nome do
 * servidor.
 * <li> {@code server_host_addr} ({@code Server.hostAddr}) : endereo
 * IP do servidor.
 * <li> {@code server_port_http} : porta HTTP do servidor.  obtido
 * diretamente do request feito ao servlet.
 * <li> {@code server_port_rmi} (ver mtodo {@link #getServerRMIport()}):
 * porta RMI do servidor.  o nico parmetro que pode no estar definido na URL
 * nem no arquivo de propriedades (neste caso o valor default 1099 ser usado).
 * <li> {@code client_addr} : endereo IP do cliente.  obtido diretamente
 * do request feito ao servlet.
 * </ul>
 * Os demais parmetros recebidos pelo servlet (com exceo do parmetro
 * <i>page</i>) so repassados  aplicao atravs da tag &lt;argument&gt; do
 * JNLP.
 *
 * <b>IMPORTANTE:</b>
 *
 * <ul>
 * <li>Os parmetros referentes ao nome ou endereo do servidor, assim como a
 * porta RMI para conexo, s podem ser definidos via arquivo de propriedades.
 * Isto  feito desta forma porque outros servlets fazem conexo direta com o
 * servidor usando estes valores, e o custo de passar estes parmetros via URL
 * nos casos em que isto seria necessrio supera os benefcios.
 * <li>Este mecanismo s funciona com arquivos JNLP se as seguintes condies
 * forem observadas:
 * <ul>
 * <li>a tag &lt;jnlp&gt; <b>no</b> pode possuir o parmetro {@code href}.
 * Caso contrrio o Java WebStart tentar obter o arquivo diretamente do Tomcat,
 * sem passar pelo servlet, e as tradues no sero aplicadas
 * <li>outros arquivos JNLP referenciados a partir do arquivo principal via tag
 * &lt;extension&gt; <b>no</b> podem possuir os parmetros
 * {@code codebase} nem {@code href} nas suas tags &lt;jnlp&gt;, pelo
 * mesmo motivo acima
 * </ul>
 * </ul>
 */
public class InitServlet extends AbstractServlet {
  /**
   * MIME-type para arquivos JNLP.
   */
  private static final String JNLP_MIME_TYPE =
    "application/x-java-jnlp-file;charset=iso-8859-1";

  /**
   * MIME-type para arquivos HTML.
   */
  private static final String HTML_MIME_TYPE = "text/html;charset=iso-8859-1";

  /**
   * Parmetro que permite a carga de uma pgina especfica diferente da
   * {@link #DEFAULT_PAGE default}.
   */
  private static final String PARAM_PAGE = "page";

  /**
   * Pgina modelo a ser carregada se nenhuma tiver sido especificada.
   */
  private static final String DEFAULT_PAGE = "index.jnlp";

  /**
   * Parmetro com o host do servidor a ser usado para conexes RMI.
   */
  private static final String PARAM_SERVER_HOST_NAME = "server_host_name";

  /**
   * Parmetro com a porta para conexes RMI ao servidor.
   */
  private static final String PARAM_SERVER_PORT_RMI = "server_port_rmi";

  /**
   * Parmetro com a porta onde os clientes exportam os objetos RMI.
   */
  private static final String PARAM_CLIENT_PORT_RMI = "client_port_rmi";

  /**
   * Parmetro com o IP do servidor.
   */
  private static final String PARAM_SERVER_HOST_ADDR = "server_host_addr";

  /**
   * Parmetro porta HTTP do servidor.
   */
  private static final String PARAM_SERVER_PORT_HTTP = "server_port_http";

  /**
   * Parmetro IP do cliente.
   */
  private static final String PARAM_CLIENT_ADDR = "client_addr";

  /**
   * Constri e retorna uma pgina JNLP, usando as informaes fornecidas.
   * 
   * @param request informaes sobre a requisio.
   * @param response informaes sobre a resposta.
   * 
   * @throws IOException em caso de erro de I/O
   * @throws ServletException no caso de erro no servlet.
   */
  @Override
  protected void doGet(HttpServletRequest request, HttpServletResponse response)
    throws IOException, ServletException {
    if (!hasPropertiesFile()) {
      if (!loadPropertiesFile()) {
        showErrorPage(response,
          "Arquivo de propriedades System.properties no foi encontrado.");
        return;
      }
    }

    PrintWriter out = response.getWriter();

    try {
      Map<String, String> processedParams = processParams(request);
      /*
       * lemos o arquivo original, aplicamos as "tradues" e devolvemos o
       * arquivo traduzido
       */
      String baseFile = readBaseFile(request);
      Set<String> requiredParams = collectRequiredParams(baseFile);
      Set<String> missingParams =
        checkMissingParams(requiredParams, processedParams);
      if (!missingParams.isEmpty()) {
        showErrorPage(response, getErrorPage(missingParams));
        return;
      }
      String translatedFile =
        translateFile(baseFile, request, requiredParams, processedParams);
      response.setContentType(JNLP_MIME_TYPE);
      response.setHeader("Pragma", "no-cache");
      response.setHeader("Cache-Control",
        "private, no-store, no-cache, must-revalidate");
      out.println(translatedFile);
      /*
       * acrescentamos um timestamp ao arquivo alterado, como comentrio
       */
      out.println("<!-- " + Calendar.getInstance().getTime().toString()
        + " -->");
    }
    catch (Exception ex) {
      System.out.println(ex.getMessage()); // TODO costa remover
      showErrorPage(response, getErrorPage(ex));
      throw new ServletException(ex);
    }
    finally {
      if (out != null) {
        out.flush();
        out.close();
      }
    }
  }

  /**
   * Verifica se todos os parmetros definidos no arquivo JNLP foram fornecidos.
   * 
   * @param requiredParams parmetros requeridos
   * @param processedParams parmetros processados.
   * 
   * @return conjunto com os parmetros que no foram fornecidos (ou um conjunto
   *         vazio caso todos tenham sido fornecidos)
   */
  private Set<String> checkMissingParams(Set<String> requiredParams,
    Map<String, String> processedParams) {
    Set<String> missingParams = new HashSet<String>(requiredParams);
    missingParams.removeAll(processedParams.keySet());

    for (Map.Entry<String, String> entry : processedParams.entrySet()) {
      String paramValue = entry.getValue();
      if (paramValue == null || paramValue.isEmpty()) {
        missingParams.add(entry.getKey());
      }
    }
    return missingParams;
  }

  /**
   * Obtm valores para os parmetros, dando prioridade  URL e em seguida ao
   * arquivo de propriedades.
   * 
   * @param request informaes sobre a requisio
   * @return Mapa com os parmetros processados
   */
  private Map<String, String> processParams(HttpServletRequest request) {
    Map<String, String> processedParams = new Hashtable<String, String>();
    /*
     * a porta para o servidor HTTP e o endereo IP do cliente so os nicos
     * parmetros obtidos diretamente do request
     */
    String serverPortHttp = Integer.toString(request.getServerPort());
    String clientAddress = request.getRemoteAddr();
    processedParams.put(PARAM_SERVER_PORT_HTTP, serverPortHttp);
    processedParams.put(PARAM_CLIENT_ADDR, clientAddress);
    /*
     * endereo IP do servidor (propriedade)
     */
    processedParams.put(PARAM_SERVER_HOST_ADDR, getServerAddress());
    /*
     * nome do servidor (propriedade)
     */
    processedParams.put(PARAM_SERVER_HOST_NAME, getServerName());
    /*
     * porta RMI do servidor
     */
    processedParams.put(PARAM_SERVER_PORT_RMI, getServerRMIport());
    /*
     * porta RMI onde os clientes exportam os objetos RMI
     */
    processedParams.put(PARAM_CLIENT_PORT_RMI, getClientRMIport());

    return processedParams;
  }

  /**
   * Extrai o parmetro obrigatrio especificado do mapa de parmetros.
   * 
   * @param key chave do parmetro a ser extrado.
   * @param request informaes sobre a requisio do servlet.
   * @param defaultValue valor default, caso no tenha sido informado.
   * 
   * @return valor associado ao parmetro no mapa.
   */
  private String processParam(String key, HttpServletRequest request,
    String defaultValue) {
    String value;
    if (request != null) {
      value = request.getParameter(key);
      if (value == null || value.isEmpty()) {
        value = defaultValue;
      }
    }
    else {
      value = defaultValue;
    }
    return value;
  }

  /**
   * Traduz o arquivo original, modificando os marcadores e adicionando novos
   * parmetros que tenham sido passados via URL.
   * 
   * @param baseFile contedo do arquivo a ser traduzido.
   * @param request informaes sobre a requisio do servlet.
   * @param requiredParams parmetros requeridos
   * @param processedParams parmetros processados.
   * @return contedo do arquivo personalizado para a aplicao.
   */
  private String translateFile(String baseFile, HttpServletRequest request,
    Set<String> requiredParams, Map<String, String> processedParams) {
    /*
     * os parmetros obrigatrios foram fornecidos de alguma forma, podemos
     * continuar
     */
    String translatedFile = baseFile;
    for (String param : requiredParams) {
      translatedFile = replaceParam(param, translatedFile, processedParams);
    }
    /*
     * os demais parmetros que ainda no tenham sido tratados so repassados
     * para aplicao via tag <argument> (com exceo do 'page').
     */
    for (Enumeration<?> e = request.getParameterNames(); e.hasMoreElements();) {
      String key = (String) e.nextElement();
      if (key.equals(PARAM_PAGE) || processedParams.containsKey(key)) {
        /*
         * o parmetro j foi processado ou  o 'page', que no queremos
         * repassar
         */
        continue;
      }
      String value = request.getParameter(key);
      translatedFile = appendParam(key, value, translatedFile);
    }
    return translatedFile;
  }

  /**
   * Exibe uma pgina de erro HTML.
   * 
   * @param response resposta a ser dada para o cliente
   * @param htmlContent contedo HTML da pgina
   */
  private void showErrorPage(HttpServletResponse response, String htmlContent) {
    try {
      PrintWriter writer = response.getWriter();
      response.setContentType(HTML_MIME_TYPE);
      writer.println(htmlContent);
    }
    catch (IOException e) {
      System.err.println("Erro exibindo pgina de erro do InitServlet");
      e.printStackTrace();
    }
  }

  /**
   * Obtm uma pgina de erro que exibe uma exceo.
   * 
   * @param ex exceo ocorrida.
   * 
   * @return retorna uma pgina HTML de erro para uma exceo
   */
  private String getErrorPage(Exception ex) {
    StringBuilder output = startErrorMessage();
    output.append("<p>").append(ex.toString()).append("</p>\n");
    StackTraceElement[] stackLines = ex.getStackTrace();
    for (int i = 0; i < stackLines.length; i++) {
      output.append("&nbsp;&nbsp;&nbsp;").append(stackLines[i]).append(
        "<br/>\n");
    }
    finishErrorMessage(output);
    return output.toString();
  }

  /**
   * Gera uma pgina HTML de erro a partir de uma lista de erros.
   * 
   * @param errors lista de erros
   * @return pgina HTML de erro
   */
  private String getErrorPage(Set<String> errors) {
    StringBuilder output = startErrorMessage();
    output
      .append("Parmetros obrigatrios no foram definidos via URL nem no arquivo ");
    output.append(PROPERTIES_FILE);
    output.append('\n');
    output.append("<ul>\n");
    for (String error : errors) {
      output.append("<li> ");
      String name = paramMatchesProperty(error);
      if (name != null) {
        output.append(name);
      }
      else {
        output.append(error);
      }
      output.append('\n');
    }
    output.append("</ul>\n");
    finishErrorMessage(output);
    return output.toString();
  }

  /**
   * Este mtodo recupera o nome esperado no arquivo para um parmetro.
   * 
   * @param param Nome do parmetro
   * @return Nome esperado no arquivo para o parmetro.
   */
  private String paramMatchesProperty(String param) {
    if (param.equalsIgnoreCase(PARAM_SERVER_HOST_ADDR)) {
      return PROP_SERVER_HOST_ADDR;
    }
    if (param.equalsIgnoreCase(PARAM_SERVER_HOST_NAME)) {
      return PROP_SERVER_HOST_NAME;
    }
    if (param.equalsIgnoreCase(PARAM_SERVER_PORT_RMI)) {
      return PROP_SERVER_PORT_RMI;
    }
    if (param.equalsIgnoreCase(PARAM_CLIENT_PORT_RMI)) {
      return PROP_CLIENT_PORT_RMI;
    }
    return null;
  }

  /**
   * Inicializa um {@link StringBuilder} com o prembulo de uma pgina de erro
   * HTML.
   * 
   * @return <code>StringBuilder</code> com o incio de uma tela HTML de erro
   * 
   * @see #finishErrorMessage(StringBuilder)
   */
  private StringBuilder startErrorMessage() {
    StringBuilder msg = new StringBuilder("<html>\n");
    msg.append("<head><title>Erro</title></head>\n");
    msg.append("<body>\n");
    msg.append("<h2>Erro</h2>\n");
    msg
      .append("Por favor, entre em contato com o suporte do sistema e fornea "
        + "a seguinte mensagem: <hr/>\n");
    msg.append(Calendar.getInstance().getTime().toString());
    msg.append("<br><br>\n");
    return msg;
  }

  /**
   * Finaliza uma mensagem de erro HTML.
   * 
   * @param msg mensagem
   * 
   * @see #startErrorMessage()
   */
  private void finishErrorMessage(StringBuilder msg) {
    msg.append("<hr/></body>\n");
    msg.append("</html>");
  }

  /**
   * Obtm o endereo para o arquivo comum no servidor http, extrado do
   * <code>HttpServletRequest</code> recebido por este servlet. Constri a mesma
   * URL usada para acessar este servlet, trocando o nome do arquivo (init.html)
   * pelo nome da pgina original.
   * 
   * @param request informaes sobre a requisio do servlet.
   * 
   * @return endereo para o arquivo comum no servidor http.
   */
  String getBaseFileUrl(HttpServletRequest request) {
    String serverPort = Integer.toString(request.getServerPort());
    String fileURL =
      "http://" + request.getServerName() + ':' + serverPort
        + request.getRequestURI();
    String fileName = processParam(PARAM_PAGE, request, DEFAULT_PAGE);
    int lastSlashPos = fileURL.lastIndexOf('/');
    return fileURL.substring(0, lastSlashPos + 1) + fileName;
  }

  /**
   * L o arquivo que seria devolvido como resposta ao request.
   * 
   * @param request requisio HTTP
   * @return <code>String</code> com o contedo do arquivo original
   * @throws IOException em caso de erro de I/O
   */
  private String readBaseFile(HttpServletRequest request) throws IOException {
    URL url = new URL(getBaseFileUrl(request));
    BufferedReader br = null;
    StringBuilder buffer = new StringBuilder();
    try {
      br = new BufferedReader(new InputStreamReader(url.openStream()));
      String line = null;
      while ((line = br.readLine()) != null) {
        buffer.append(line).append('\n');
      }
    }
    finally {
      if (br != null) {
        br.close();
      }
    }
    String fileContent = buffer.toString();
    return fileContent;
  }

  /**
   * Identifica os parmetros que foram especificados no arquivo JNLP.
   * 
   * @param fileContent contedo do arquivo JNLP
   * @return Conjunto de parmetros obrigatrios
   */
  private Set<String> collectRequiredParams(String fileContent) {
    Set<String> requiredParams = new HashSet<String>();
    /*
     * pattern para pegar os marcadores definidos entre $...$
     */
    Pattern pattern = Pattern.compile("\\$(\\w+)\\$");
    Matcher matcher = pattern.matcher(fileContent);
    while (matcher.find()) {
      String paramName = matcher.group(1);
      if (paramName != null && !paramName.isEmpty()) {
        requiredParams.add(paramName);
      }
    }
    return requiredParams;
  }

  /**
   * Substitui o parmetro especificado no texto especificado pelo valor
   * especificado.
   * 
   * @param key chave do parmetro a ser substitudo.
   * @param text texto no qual ser feita a substituio.
   * @param processedParams mapa dos parmetros processados
   * @return texto com a chave substituda pelo valor especificado.
   */
  private String replaceParam(String key, String text,
    Map<String, String> processedParams) {
    String value = null;
    value = processedParams.get(key);
    if (value == null || value.isEmpty()) {
      throw new IllegalStateException("parmetro " + key + " no encontrado");
    }
    String customFile = text.replaceAll("\\$" + key + "\\$", value);
    return customFile;
  }

  /**
   * Adiciona um parmetro ao texto especificado.
   * 
   * @param key chave do parmetro.
   * @param value valor do parmetro.
   * @param text texto no qual ser feita a substituio.
   * 
   * @return o texto appendado.
   */
  String appendParam(String key, String value, String text) {
    int appendPos = text.toLowerCase().indexOf("</application-desc>");
    if (appendPos == -1) {
      throw new IllegalArgumentException("text no possui </application-desc>");
    }
    String appendText =
      "  <argument>--" + key + ' ' + value + "</argument>\n  ";
    StringBuilder buffer = new StringBuilder(text.substring(0, appendPos));
    buffer.append(appendText);
    buffer.append(text.substring(appendPos, text.length()));
    return buffer.toString();
  }
}
