/**
 * $Id: CommandPersistence.java 177959 2016-12-02 11:56:05Z fpina $
 */
package csbase.sga;

import java.io.BufferedOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Map;

import csbase.sga.executor.JobData;

/**
 * Camada de persistncia de jobs.
 *
 * @author Tecgraf/PUC-Rio
 */
class CommandPersistence {
  /** Intncia nica */
  private static CommandPersistence instance;
  /** Caminho para o arquivo de persistncia de comandos */
  private String persistenceFileCmds;
  /** Caminho para o arquivo de persistncia de parmetros de comandos */
  private String persistenceFileParams;
  /**
   * Mapa com os identificadores de comandos e os dados dos job na camada de
   * execuo de jobs.
   */
  private Map<String, JobData> commands;
  /** Mapa com os identificadores de comandos e parmetros de execuo. */
  private Map<String, Map<String, String>> commandsParams;

  /**
   * Construtor
   *
   * @param persistenceFile caminho para o arquivo de persistncia
   */
  CommandPersistence(String persistenceFile) {
    this.persistenceFileCmds = persistenceFile + ".cmds";
    this.persistenceFileParams = persistenceFile + ".params";
    this.commands = new Hashtable<>();
    this.commandsParams = new Hashtable<>();

    try {
      File file = new File(this.persistenceFileCmds);
      file.createNewFile();
      file = new File(this.persistenceFileParams);
      file.createNewFile();
    }
    catch (IOException e) {
      e.printStackTrace();
    }

    read();
  }

  /**
   * Adiciona um comando e grava no arquivo
   *
   * @param commandId identificador do comando
   * @param data      dado da camanda de execuo ao associado comando
   * @param params    parmetros de execuo
   */
  protected synchronized void addCommand(
      String commandId,
      JobData data,
      Map<String, String> params) {
    commands.put(commandId, data);
    commandsParams.put(commandId, params);
    write();
  }

  /**
   * Adiciona um conjunto de comandos e grava no arquivo
   *
   * @param commands       um mapa com os identificadores de comandos os dados
   *                       associado a eles
   * @param commandsParams parmetros de execuo
   */
  protected synchronized void addCommands(
      Map<String,
          JobData> commands,
      Map<String, Map<String, String>> commandsParams) {
    this.commands = commands;
    this.commandsParams = commandsParams;
    write();
  }

  /**
   * Remove um comando.
   *
   * @param commandId identificador do comando
   */
  protected synchronized void removeCommand(String commandId) {
    this.commands.remove(commandId);
    this.commandsParams.remove(commandId);
    write();
  }

  /**
   * Obtm os comandos armazenados
   *
   * @return um mapa com os identificadores de comandos e os dados associado a
   * eles
   */
  protected Map<String, JobData> getCommands() {
    return Collections.unmodifiableMap(this.commands);
  }

  /**
   * Obtm os parmetros de execuo dos comandos armazenados
   *
   * @return um mapa com os identificadores de comandos e os parmetros de
   * execuo associado a eles
   */
  protected Map<String, Map<String, String>> getCommandsParams() {
    return Collections.unmodifiableMap(this.commandsParams);
  }

  /**
   * Grava os comandos, seus dados e seus parmetros de execuo em arquivo
   */
  private void write() {
    try (FileOutputStream fos = new FileOutputStream(persistenceFileCmds);
         ObjectOutputStream oos =
        new ObjectOutputStream(new BufferedOutputStream(fos))) {
      oos.writeObject(commands);
    }
    catch (IOException e) {
      e.printStackTrace();
    }

    try (FileOutputStream fos = new FileOutputStream(persistenceFileParams);
         ObjectOutputStream oos =
             new ObjectOutputStream(new BufferedOutputStream(fos))) {
      oos.writeObject(commandsParams);
    }
    catch (IOException e) {
      e.printStackTrace();
    }
  }

  /**
   * Le os comandos, seus dados e seus parmetros de execuo do arquivo
   */
  private void read() {
    try (FileInputStream fis = new FileInputStream(persistenceFileCmds);
         ObjectInputStream ois = new ObjectInputStream(fis);) {

      commands = (Map<String, JobData>) ois.readObject();
    }
    catch (IOException | ClassNotFoundException e) {
      e.printStackTrace();
    }

    try (FileInputStream fis = new FileInputStream(persistenceFileParams);
         ObjectInputStream ois = new ObjectInputStream(fis);) {

      commandsParams = (Map<String, Map<String, String>>) ois.readObject();
    }
    catch (IOException | ClassNotFoundException e) {
      e.printStackTrace();
    }
  }
}
