package csbase.sga;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.FileHandler;
import java.util.logging.Level;
import java.util.logging.Logger;

import csbase.sga.monitor.DefaultEnvironmentMonitor;
import org.omg.CORBA.IntHolder;

import csbase.server.plugin.service.IServiceManager;
import csbase.server.plugin.service.sgaservice.ISGADaemon;
import csbase.server.plugin.service.sgaservice.ISGAService;
import csbase.server.plugin.service.sgaservice.SGADaemonException;
import csbase.sga.executor.DefaultJobExecutor;
import csbase.sga.executor.JobData;
import csbase.sga.executor.JobExecutor;
import csbase.sga.executor.JobInfo;
import csbase.sga.executor.JobObserver;
import csbase.sga.monitor.EnvironmentMonitor;
import csbase.sga.monitor.SGAInfo;
import sgaidl.COMMAND_CPU_TIME_SEC;
import sgaidl.COMMAND_USER_TIME_SEC;
import sgaidl.COMMAND_WALL_TIME_SEC;
import sgaidl.CompletedCommandInfo;
import sgaidl.InvalidCommandException;
import sgaidl.InvalidParameterException;
import sgaidl.InvalidPathException;
import sgaidl.InvalidSGAException;
import sgaidl.MissingParameterException;
import sgaidl.NoPermissionException;
import sgaidl.Pair;
import sgaidl.PathNotFoundException;
import sgaidl.RetrievedInfo;
import sgaidl.SGAAlreadyRegisteredException;
import sgaidl.SGACommand;
import sgaidl.SGAControlAction;
import sgaidl.SGANotRegisteredException;
import sgaidl.SGAPath;
import sgaidl.SGAProperties;
import sgaidl.SystemException;

/**
 * Local SGA.
 *
 * @author Tecgraf/PUC-Rio
 */
public class SGALocal implements ISGADaemon {
  /** Nome do servio SGA. */
  public static final String SGA_SERVICE_NAME = "SGAService";
  public static final String SGA_MACHINE_TIME_KEY =
    "csbase_machine_time_seconds";
  public static final String SGA_UPDATE_TIME_KEY = "csbase_update_time_seconds";
  public static final String SGA_PROCESS_TIME_KEY =
    "csbase_process_time_seconds";
  public static final String SGA_COMPLETED_TIME_KEY =
    "csbase_completed_time_seconds";

  private static final String SGA_PERSISTENCE_PATH_KEY =
    "csbase_persistence_path";
  private static final String SGA_LOG_PATH_KEY = "csbase_log_path";
  private static final String SGA_LOG_LEVEL_KEY = "csbase_log_level";
  private static final Level DEFAULT_LOG_LEVEL = Level.INFO;

  /** As propriedades do SGALocal. */
  protected Properties pluginProperties;

  /** A interface SGAService com o qual o SGA se comunica. */
  private ISGAService sgaService;
  /** Executor de comandos */
  private JobExecutor executor;
  /** Monitor do ambiente de execuo */
  private EnvironmentMonitor monitor;

  /** Mapa com os identificadores de commando e suas referncias */
  private Map<String, SGALocalCommand> commands;

  /** Logger usado pelo SGA */
  protected Logger logger;

  /** Agendamento da atualizao das propriedades do SGA. */
  private TimerExecutor updatePropertiesTask;

  /** Agendamento da busca e notificao de comandos completados. */
  private TimerExecutor checkCompletedCommandsTask;

  /** Camada de persistncia de comandos */
  private CommandPersistence persistence;

  /** Nome do SGA */
  private String sgaName;

  /** Chaves padres de configurao do SGA */
  private String[] defaltConfigKeys =
  { sgaidl.SGA_NODE_NUM_PROCESSORS.value,
      sgaidl.SGA_NODE_MEMORY_RAM_INFO_MB.value,
      sgaidl.SGA_NODE_MEMORY_SWAP_INFO_MB.value,
      sgaidl.JOB_CONTROL_ACTIONS.value };

  /** Chaves padres de informaes do SGA */
  private String[] defaltInfoKeys =
  { sgaidl.SGA_NODE_LOAD_AVG_1MIN_PERC.value,
      sgaidl.SGA_NODE_LOAD_AVG_5MIN_PERC.value,
      sgaidl.SGA_NODE_LOAD_AVG_15MIN_PERC.value,
      sgaidl.SGA_NODE_MEMORY_RAM_FREE_PERC.value,
      sgaidl.SGA_NODE_MEMORY_SWAP_FREE_PERC.value,
      sgaidl.SGA_NODE_NUMBER_OF_JOBS.value };

  /** Chaves padres de informaes de processo */
  private String[] defaltProcessKeys =
  { sgaidl.COMMAND_PID.value, sgaidl.COMMAND_STRING.value,
      sgaidl.COMMAND_EXEC_HOST.value, sgaidl.COMMAND_STATE.value,
      sgaidl.COMMAND_MEMORY_RAM_SIZE_MB.value,
      sgaidl.COMMAND_MEMORY_SWAP_SIZE_MB.value, sgaidl.COMMAND_CPU_PERC.value,
      //    sgaidl.COMMAND_ "csbase_command_time_sec",
      sgaidl.COMMAND_WALL_TIME_SEC.value, sgaidl.COMMAND_USER_TIME_SEC.value,
      sgaidl.COMMAND_SYSTEM_TIME_SEC.value };

  /**
   * Executa uma tarefa peridica, de acordo com uma configurao de intervalo
   * de tempo.
   */
  private class TimerExecutor {
    /** Thread que faz a execuo peridica de uma tarefa. */
    final private Timer timer;
    /** Tempo (em segundos) para execuo da tarefa. */
    final private int seconds;
    /** Tarefa agendada. */
    final private TimerTask task;

    /**
     * Construtor.
     *
     * @param seconds tempo (em segundos) para renovao do registro.
     * @param task tarefa cuja execuo est agendada.
     */
    protected TimerExecutor(int seconds, TimerTask task) {
      this.seconds = seconds;
      this.task = task;
      this.timer = new Timer();
    }

    /**
     * Inicia a thread de agendamento de execuo da tarefa.
     */
    protected void start() {
      timer.schedule(task, seconds, seconds);
    }

    /**
     * Interrompe a thread de agendamento de execuo da tarefa.
     */
    protected void stop() {
      timer.cancel();
    }
  }

  /**
   * Tarefa agendada que executa a chamada ao SGAService para atualizar as
   * propriedades do SGA e seus ns de execuo.
   */
  private class SGAUpdatePropertiesTask extends TimerTask {
    @Override
    public void run() {
      logger.finest("Atualiza os dados do SGA  " + sgaName);

      SGAProperties sgaProps = loadSGAProperties();
      if (sgaProps.nodesProperties.length <= 0) {
        logger.log(Level.WARNING, "No existem ns de SGAs");
      }
      else {
        try {
          if(monitor.isAlive()) {
            if (sgaService.isRegistered(SGALocal.this, sgaName)) {
              sgaService.updateSGAInfo(SGALocal.this, sgaName, sgaProps);
            } else {
              registerSGA(sgaProps);
            }
          }
        }
        catch (InvalidParameterException e) {
          logger.log(Level.SEVERE,
            "Erro ao atualizar as informaes do SGA: {0}: {1}", new Object[] {
                e, e.message });
        }
        catch (NoPermissionException e) {
          logger.log(
            Level.SEVERE, "Erro ao atualizar as informaes do SGA: {0}: {1}",
            new Object[] { e, e.message });
        }
        catch (SGANotRegisteredException e) {
          logger.log(
            Level.SEVERE, "Erro ao atualizar as informaes do SGA: {0}: {1}",
            new Object[] { e, e.message });
        }
        catch (InvalidSGAException e) {
          logger.log(
            Level.SEVERE, "Erro ao atualizar as informaes do SGA: {0}: {1}",
            new Object[] { e, e.message });
        }
      }
    }
  }

  /**
   * Tarefa agendada que busca e notifica comandos completados.
   */
  private class CheckCompletedCommandsTask extends TimerTask {
    @Override
    public void run() {
      logger.finest("Buscando comandos finalizados ...");
      executor.searchAndNotifyFinishedJobs();
    }
  }

  /**
   * Construtor padro.
   *
   * @param serviceManager gerente dos servios
   */
  public SGALocal(IServiceManager serviceManager) {
    this.sgaService =
      ISGAService.class.cast(serviceManager.getService(SGA_SERVICE_NAME));
    commands = new ConcurrentHashMap<>();
  }

  /**
   * Mtodo de inicializao onde so definidos os {@link JobExecutor executor}
   * e {@link EnvironmentMonitor monitor}. Deve ser sobreescrito nas classes que
   * extendem {@link SGALocal} para definirem implementaes especficas de
   * {@link JobExecutor JobExecutor} e {@link EnvironmentMonitor
   * EnvironmentMonitor} usando os mtodos {@link #setExecutor(JobExecutor)} e
   * {@link #setMonitor(EnvironmentMonitor)} respectivamente.
   *
   * A implementao padro usa {@link DefaultJobExecutor} e
   * {@link EnvironmentMonitor}.
   */
  protected void init() {
    try {
      setMonitor(new DefaultEnvironmentMonitor(pluginProperties));
      setExecutor(new DefaultJobExecutor(pluginProperties));
    }
    catch (SGADaemonException e) {
      logger.log(Level.SEVERE, "Erro ao iniciar plugin do SGA", e);
      stop();
    }
  }

  /**
   * Obtm o executor de jobs.
   *
   * @return o executor
   */
  protected JobExecutor getExecutor() {
    return executor;
  }

  /**
   * Define o executor de jobs.
   *
   * @param executor o executor
   */
  protected void setExecutor(JobExecutor executor) {
    this.executor = executor;
  }

  /**
   * Obtm o monitor do ambiente de execuo.
   *
   * @return o monitor
   */
  protected EnvironmentMonitor getMonitor() {
    return monitor;
  }

  /**
   * Define o monitor do ambiente de execuo.
   *
   * @param monitor o monitor
   */
  protected void setMonitor(EnvironmentMonitor monitor) {
    this.monitor = monitor;
  }

  /**
   * Carrega as propriedades do SGA.
   *
   * @return retorna as propriedades do SGA.
   */
  private synchronized SGAProperties loadSGAProperties() {
    SGAInfo sgaInfo = monitor.getSGAInfo();

    SGAProperties sgaProperties = new SGAProperties();
    sgaProperties.properties = mapToPair(sgaInfo.properties);

    if (sgaInfo.nodesInfo.size() > 0) {
      sgaProperties.nodesProperties = new Pair[sgaInfo.nodesInfo.size()][];
      for (int j = 0; j < sgaInfo.nodesInfo.size(); j++) {
        sgaProperties.nodesProperties[j] =
          mapToPair(sgaInfo.nodesInfo.get(j).properties);
      }
    }
    else {
      // No tem n, ento coloca o prprio SGA como o nico n
      sgaProperties.nodesProperties = new Pair[1][];
      sgaProperties.nodesProperties[0] = sgaProperties.properties;
    }

    // Coloca no log as propriedades dos SGAs
    logger.finest("Propriedades do SGA: ");
    for (Pair p : sgaProperties.properties) {
      logger.finest("\t" + p.key + "=" + p.value);
    }

    logger.finest(
      "Propriedades dos ns do SGA (" + sgaProperties.nodesProperties.length
      + ")");
    int j = 0;
    for (Pair[] nodes : sgaProperties.nodesProperties) {
      logger.finest("\t" + "N " + (j++));
      for (Pair p : nodes) {
        logger.finest("\t\t" + p.key + "=" + p.value);
      }
    }

    return sgaProperties;
  }

  /**
   * Valida se as propriedades do plugin esto definidas.
   *
   * @throws SGADaemonException se as propriedades do plugin no estiverem
   *         definidas
   */
  private void validatePluginProperties() throws SGADaemonException {
    if (this.pluginProperties == null) {
      throw new SGADaemonException(
        "As propriedades do SGADaemon no foram atribudas.");
    }
    if (pluginProperties.getProperty(sgaidl.SGA_NAME.value) == null) {
      throw new SGADaemonException("A propriedade " + sgaidl.SGA_NAME.value
        + " com o nome do SGA no foi definida.");
    }

    if (pluginProperties.getProperty(SGA_UPDATE_TIME_KEY) == null) {
      throw new SGADaemonException("A propriedade " + SGA_UPDATE_TIME_KEY
        + " no foi definida.");
    }
    try {
      Integer.parseInt(pluginProperties.getProperty(SGA_UPDATE_TIME_KEY));
    }
    catch (NumberFormatException e) {
      throw new SGADaemonException("A propriedade " + SGA_UPDATE_TIME_KEY
        + " possui um valor no numrico.", e);
    }

    if (pluginProperties.getProperty(SGA_COMPLETED_TIME_KEY) == null) {
      throw new SGADaemonException("A propriedade " + SGA_COMPLETED_TIME_KEY
          + " no foi definida.");
    }
    try {
      Integer.parseInt(pluginProperties.getProperty(SGA_COMPLETED_TIME_KEY));
    }
    catch (NumberFormatException e) {
      throw new SGADaemonException("A propriedade " + SGA_COMPLETED_TIME_KEY
          + " possui um valor no numrico.", e);
    }

    // Log file directory
    if (pluginProperties.getProperty(SGA_LOG_PATH_KEY) == null) {
      throw new SGADaemonException("A propriedade " + SGA_LOG_PATH_KEY
        + " no foi definida.");
    }
    File logPath = new File(pluginProperties.getProperty(SGA_LOG_PATH_KEY));
    if (!logPath.exists() || !logPath.isDirectory()) {
      throw new SGADaemonException("O diretrio defindo na propriedade "
        + SGA_LOG_PATH_KEY + " no existe ou no  diretrio.");
    }

    // Persistence directory
    if (pluginProperties.getProperty(SGA_PERSISTENCE_PATH_KEY) == null) {
      throw new SGADaemonException("A propriedade " + SGA_PERSISTENCE_PATH_KEY
        + " no foi definida.");
    }

    // Project root directory
    if (pluginProperties.getProperty(
      sgaidl.SGA_PROJECT_ROOT_DIR.value) == null) {
      throw new SGADaemonException("A propriedade "
        + sgaidl.SGA_PROJECT_ROOT_DIR.value + " no foi definida.");
    }
    File projDirPath =
      new File(pluginProperties.getProperty(sgaidl.SGA_PROJECT_ROOT_DIR.value));
    if (!projDirPath.exists() || !projDirPath.isDirectory()) {
      throw new SGADaemonException("O diretrio defindo na propriedade "
        + sgaidl.SGA_PROJECT_ROOT_DIR.value
        + " no existe ou no  diretrio.");
    }
    pluginProperties.setProperty(
      sgaidl.SGA_PROJECT_ROOT_DIR.value, projDirPath.getAbsolutePath());

    // Algorithm root directory
    if (pluginProperties.getProperty(
      sgaidl.SGA_ALGORITHM_ROOT_DIR.value) == null) {
      throw new SGADaemonException("A propriedade "
        + sgaidl.SGA_ALGORITHM_ROOT_DIR.value + " no foi definida.");
    }
    File algoDirPath =
      new File(pluginProperties.getProperty(
        sgaidl.SGA_ALGORITHM_ROOT_DIR.value));
    if (!algoDirPath.exists() || !algoDirPath.isDirectory()) {
      throw new SGADaemonException("O diretrio defindo na propriedade "
        + sgaidl.SGA_ALGORITHM_ROOT_DIR.value
        + " no existe ou no  diretrio.");
    }
    pluginProperties.setProperty(
      sgaidl.SGA_ALGORITHM_ROOT_DIR.value, algoDirPath.getAbsolutePath());
  }

  /**
   * Registra o SGA no servidor.
   *
   * @param sgaProps propriedades do SGA
   */
  private synchronized void registerSGA(SGAProperties sgaProps) {
    IntHolder updateInterval = new IntHolder();
    try {
      sgaService.registerSGA(this, sgaName, sgaProps, updateInterval);
    }
    catch (InvalidParameterException e) {
      logger.log(
        Level.SEVERE, "Erro ao registrar o SGA: {0}: {1}", new Object[] { e,
            e.message });
      this.stop();
    }
    catch (NoPermissionException e) {
      logger.log(
        Level.SEVERE, "Erro ao registrar o SGA: {0}: {1}", new Object[] { e,
            e.message });
      this.stop();
    }
    catch (SGAAlreadyRegisteredException e) {
      logger.log(
        Level.SEVERE, "Erro ao registrar o SGA: {0}: {1}", new Object[] { e,
            e.message });
      this.stop();
    }

    Map<String, Map<String, String>> persistedCmdParams = persistence.getCommandsParams();
    Map<String, JobData> persistedCommands = persistence.getCommands();
    for (String commandId : persistedCommands.keySet()) {
      JobData jobData = persistedCommands.get(commandId);
      SGALocalCommand commandRef =
        new SGALocalCommand(executor, commandId, jobData);
      commandRef.setLogger(logger);
      logger.log(
        Level.INFO, "Recuperando comando {0}", new Object[] { commandId });
      if (persistedCmdParams.containsKey(commandId)) {
        Map<String, String> params = persistedCmdParams.get(commandId);
        commandRef.recover(jobData, params, new SynchronizedJobObserver(commandId, params));
        commands.put(commandId, commandRef);
      } else {
        String msg = "Erro ao recuperar comando {0}: no foi possvel obter os parametros de execuo.";
        logger.log(Level.SEVERE, msg, new Object[] { commandId });
      }
    }

    Map<String, JobData> jobsData = new HashMap<>();
    Map<String, Map<String, String>> jobsParams = new HashMap<>();
    List<RetrievedInfo> infos = new LinkedList<>();
    for (String commandId : commands.keySet()) {
      SGALocalCommand commandRef = commands.get(commandId);
      JobData jobData = commandRef.getJobData();
      jobsData.put(commandId, jobData);
      Map<String, String> params = commandRef.getExecParam();
      jobsParams.put(commandId, params);
      infos.add(new RetrievedInfo(commandId, commandRef));
    }
    persistence.addCommands(jobsData, jobsParams);
    try {
      sgaService.commandRetrieved(sgaName, infos.toArray(new RetrievedInfo[infos.size()]));
    }
    catch (InvalidSGAException e) {
      logger.log(
        Level.SEVERE, "Erro ao recuperar comandos: {0}: {1}", new Object[] { e,
            e.message });
    }
    catch (NoPermissionException e) {
      logger.log(
        Level.SEVERE, "Erro ao recuperar comandos: {0}: {1}", new Object[] { e,
            e.message });
    }
    catch (InvalidCommandException e) {
      logger.log(
        Level.SEVERE, "Erro ao recuperar comandos: {0}: {1}", new Object[] { e,
            e.message });
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SGACommand executeCommand(final String commandString,
    final String commandId, Pair[] extraParams) throws SystemException,
  MissingParameterException {

    logger.log(
      Level.INFO, "Iniciando execuo do comando {0}.", new Object[] {
        commandId });
    logger.fine("Parmetros extras:");
    for (Pair pair : extraParams) {
      logger.log(
        Level.FINE, "{0} = {1}.", new Object[] { pair.key, pair.value });
    }

    //TODO Verificar se todos os parmetros padres esto preenchidos (MissingParameterException)

    final Map<String, String> paramMap = Utils.convertDicToMap(extraParams);

    SGALocalCommand commandRef = new SGALocalCommand(executor, commandId);
    commandRef.setLogger(logger);
    commands.put(commandId, commandRef);
    commandRef.execute(
      commandString, paramMap, new SynchronizedJobObserver(commandId, paramMap));

    return commandRef;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void control(SGAControlAction action) {
    logger.info("Solicitado " + action.value());
    if (action.equals(SGAControlAction.SHUTDOWN)) {
      try {
        sgaService.unregisterSGA(this, sgaName);
      }
      catch (NoPermissionException e) {
        logger.log(
          Level.SEVERE, "Erro ao desregistrar o SGA: {0}: {1}", new Object[] {
              e, e.message });
      }
      catch (SGANotRegisteredException e) {
        logger.log(
          Level.SEVERE, "Erro ao desregistrar o SGA: {0}: {1}", new Object[] {
              e, e.message });
      }

      stop();
    }
  }

  /**
   * {@inheritDoc}
   *
   * Se o path passado como parmetro no for acessvel no sistema de arquivos
   * do SGA, retorna null. Seno, retorna o SGAPath com as informaes sobre o
   * caminho recebido.
   */
  @Override
  public SGAPath getPath(String path) throws InvalidPathException,
  PathNotFoundException {

    return new SGAPath(path, 0, true, false, path, true, true, false, true);

    //    struct SGAPath {
    //      string path;
    //      double sizeKB;
    //      boolean isDir;
    //      boolean isSymbolicLink;
    //      string linkPath;
    //      boolean readable;
    //      boolean writable;
    //      boolean executable;
    //      boolean exists;
    //    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SGAPath[] getPaths(String root) throws InvalidPathException,
  PathNotFoundException {
    //TODO Implementar
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void ping() {
    logger.info("Acesso ao SGA com sucesso.");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDefaultConfigKeys(String[] keys) {
    //TODO Implementar
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDefaultInfoKeys(String[] keys) {
    //TODO Implementar
  }

  /**
   *
   * {@inheritDoc}
   */
  @Override
  public void setProperties(Properties pluginProps) {
    this.pluginProperties = pluginProps;
  }

  private Pair[] mapToPair(Map<String, String> map) {
    Pair[] pairs = new Pair[map.size()];
    int i = 0;
    for (Object key : map.keySet()) {
      pairs[i] = new Pair();
      pairs[i].key = (String) key;
      pairs[i].value = map.get(key);
      i++;
    }
    return pairs;
  }

  /**
   *
   * {@inheritDoc}
   *
   */
  @Override
  public boolean start() throws SGADaemonException {
    if (this.sgaService == null) {
      throw new SGADaemonException("O servio SGAService est nulo.");
    }
    // Valida as propriedades do plugin
    validatePluginProperties();

    sgaName = pluginProperties.getProperty(sgaidl.SGA_NAME.value);
    String logFile = "";
    try {
      if (pluginProperties.getProperty(SGA_LOG_PATH_KEY) != null) {
        logFile =
          pluginProperties.getProperty(SGA_LOG_PATH_KEY) + "/" + sgaName + ".%g.log";
      }

      logger = Logger.getLogger(this.getClass().getName() + "." + sgaName);
      FileHandler fh = new FileHandler(logFile, 10485760, 10);
      fh.setLevel(DEFAULT_LOG_LEVEL);
      logger.addHandler(fh);

      String logLevel;
      if (pluginProperties.getProperty(SGA_LOG_LEVEL_KEY) != null) {
        logLevel = pluginProperties.getProperty(SGA_LOG_LEVEL_KEY);
        logger.setLevel(Level.parse(logLevel));
        fh.setLevel(Level.parse(logLevel));
      }
      else {
        logger.setLevel(DEFAULT_LOG_LEVEL);
        fh.setLevel(DEFAULT_LOG_LEVEL);
      }
    }
    catch (NullPointerException | IllegalArgumentException e) {
      logger.setLevel(DEFAULT_LOG_LEVEL);
      logger.log(
        Level.INFO, "Erro ao identificar a propriedade " + SGA_LOG_LEVEL_KEY
        + ". Usando nvel de log padro; " + DEFAULT_LOG_LEVEL.getName(), e);
    }
    catch (SecurityException | IOException e) {
      throw new SGADaemonException("Erro na criao do log " + logFile, e);
    }

    commands = new Hashtable<>();
    persistence =
      new CommandPersistence(pluginProperties.getProperty(
        SGA_PERSISTENCE_PATH_KEY) + "/" + sgaName);
    init();

    SGAProperties sgaProps = loadSGAProperties();

    if (sgaProps != null && sgaProps.nodesProperties.length > 0) {
      registerSGA(sgaProps);
    }
    int updateTime =
      Integer.parseInt(pluginProperties.getProperty(SGA_UPDATE_TIME_KEY));
    this.updatePropertiesTask =
      new TimerExecutor(updateTime * 1000, new SGAUpdatePropertiesTask());
    this.updatePropertiesTask.start();

    int checkCompletedCommandsInterval =
        Integer.parseInt(pluginProperties.getProperty(SGA_COMPLETED_TIME_KEY));
    this.checkCompletedCommandsTask =
        new TimerExecutor(checkCompletedCommandsInterval * 1000, new CheckCompletedCommandsTask());
    this.checkCompletedCommandsTask.start();

    logger.info("SGADaemon " + sgaName + " iniciado");

    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void stop() {
    if (checkCompletedCommandsTask != null) {
      checkCompletedCommandsTask.stop();
    }

    if (updatePropertiesTask != null) {
      updatePropertiesTask.stop();
    }

    logger.info("Finaliza o plugin do SGA.");
  }

  /**
   * Observador que  chamado somente aps o lock de sincronizao ser liberado.
   *
   * @author Tecgraf/PUC-Rio
   */
  class SynchronizedJobObserver implements JobObserver {
    /**
     * Identificador do comando
     */
    private String commandId;
    private Map<String, String> params;

    /**
     * Construtor
     *
     * @param commandId identificador do comando
     */
    SynchronizedJobObserver(String commandId, Map<String, String> params) {
      this.commandId = commandId;
      this.params = params;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    synchronized public void onJobCompleted(JobInfo jobInfo) {
      long startTime = System.currentTimeMillis();

      int elapsedTimeSec =
          Integer.parseInt(jobInfo.jobParam.get(COMMAND_WALL_TIME_SEC.value));
      int userTimeSec =
          Integer.parseInt(jobInfo.jobParam.get(COMMAND_USER_TIME_SEC.value));
      int cpuTimeSec =
          Integer.parseInt(jobInfo.jobParam.get(COMMAND_CPU_TIME_SEC.value));

      CompletedCommandInfo completedInfo =
          new CompletedCommandInfo(elapsedTimeSec, userTimeSec, cpuTimeSec);

      logger.log(
          Level.INFO, "Notificando comado terminado: {0}. [{1}; {2}; {3}]",
          new Object[]{commandId, elapsedTimeSec, userTimeSec, cpuTimeSec});

      try {
        if (commands.containsKey(commandId)) {
          SGALocalCommand command = commands.remove(commandId);
          sgaService.commandCompleted(sgaName, command, commandId, completedInfo);
        }
      } catch (Exception e) {
        logger.log(
            Level.SEVERE, "Erro ao notificar fim do comando {0}. [{1}:{2}]",
            new Object[]{commandId, e, e.getMessage()});
        logger.log(Level.FINE, "Exceo:", e);
      } finally {
        persistence.removeCommand(commandId);
      }

      logger.fine("Tempo para notificar trmino do comando: " + (System.currentTimeMillis() - startTime + " ms"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    synchronized public void onJobLost() {
      try {
        logger.log(
          Level.INFO, "Notificando comando perdido: {0}.", new Object[] {
              commandId });

        if (commands.containsKey(commandId)) {
          commands.remove(commandId);
          sgaService.commandLost(sgaName, commandId);
        }
      }
      catch (Exception e) {
        logger.log(
          Level.SEVERE, "Erro ao notificar o comando perdido: {0}. [{1}:{2}]",
          new Object[] { commandId, e, e.getMessage() });
        logger.log(Level.FINE, "Exceo:", e);
      }
      finally {
        persistence.removeCommand(commandId);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    synchronized public void onJobStarted(JobData jobData) {
      logger.log(
        Level.INFO, "Notificando inicio de execuo de comado: {0}.",
        new Object[] { commandId });

      if (commands.containsKey(commandId)) {
        persistence.addCommand(commandId, jobData, params);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onJobKilled() {
      logger.log(
        Level.INFO, "Notificando interrupo de execuo de comado: {0}.",
        new Object[] { commandId });
      if (commands.containsKey(commandId)) {
        persistence.removeCommand(commandId);
        commands.remove(commandId);
      }
    }
  }
}
