/**
 * $Id: SSHMonitor.java 168372 2015-09-18 20:10:22Z fpina $
 */
package csbase.sga.ssh;

import java.util.Properties;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.logging.Level;
import java.util.logging.Logger;

import csbase.sga.monitor.EnvironmentMonitor;
import csbase.sga.monitor.SGAInfo;
import csbase.sshclient.CommandResult;
import csbase.sshclient.SSHClient;

/**
 * @author Tecgraf/PUC-Rio
 */
public class SSHMonitor implements EnvironmentMonitor {
  /** Executor that periodically runs the task to check the environment */
  private ScheduledExecutorService checkEnvironmentExecutor =
    Executors.newScheduledThreadPool(1);

  private final Properties pluginProperties;
  private final SGADriver driver;
  private SSHClientFactory sshClientFactory;
  private SGAInfo sgaInfo;
  /** Logger */
  private Logger logger;

  private long lastJobInfoUpdate = 0;
  private long checkEnvironmentInterval = 20 * 1000;

  public SSHMonitor(Properties pluginProperties, SGADriver driver,
    SSHClientFactory sshClientFactory) {
    this.pluginProperties = pluginProperties;

    if (pluginProperties.containsKey(SGASSH.SGA_MACHINE_TIME_KEY)) {
      checkEnvironmentInterval =
        Long.parseLong(
          pluginProperties.getProperty(SGASSH.SGA_MACHINE_TIME_KEY)) * 1000;
    }

    this.driver = driver;
    this.sshClientFactory = sshClientFactory;
    this.sgaInfo = new SGAInfo(this.pluginProperties);

    String sgaName = pluginProperties.getProperty(sgaidl.SGA_NAME.value);
    logger = Logger.getLogger(SGASSH.class.getName() + "." + sgaName);

    //    checkEnvironmentExecutor.scheduleWithFixedDelay(
    //      new CheckEnvironmentTask(), checkEnvironmentInterval,
    //      checkEnvironmentInterval, TimeUnit.MILLISECONDS);
  }

  @Override
  public synchronized SGAInfo getSGAInfo() {
    updateSGAInfo();
    return sgaInfo;
  }

  private synchronized void setSGAInfo(SGAInfo sgaInfo) {
    this.sgaInfo = sgaInfo;
  }

  private void updateSGAInfo() {
    if (System.currentTimeMillis()
      - lastJobInfoUpdate <= checkEnvironmentInterval) {
      return;
    }
    SSHClient client = null;
    try {
      client = sshClientFactory.getSSHClient();
      String command = driver.buildCheckEnvironmentCommand();
      logger.fine("Check enviroment command line: " + command);
      CommandResult result = client.execute(command);
      if (result.getStatus() > 0) {
        logger.log(
          Level.WARNING, "Check environment return code: " + result.getStatus()
            + "\nOutput: " + result.getOutput() + "\nError: "
            + result.getError());
        return;
      }
      logger.fine(
        "Check environment return code: " + result.getStatus() + "\nOutput: "
          + result.getOutput() + "\nError: " + result.getError());
      this.sgaInfo = driver.parseCheckEnvironmentOutput(result.getOutput());
      //TODO Record in the log file the info
    }
    catch (Exception e) {
      logger.log(Level.SEVERE, "Erro checking the environment", e);
    }
    finally {
      if (client != null) {
        client.disconnect();
      }
    }
  }

  /**
   * Task to check the environment.
   */
  private class CheckEnvironmentTask implements Runnable {
    @Override
    public void run() {
      SSHClient client = null;
      try {
        client = sshClientFactory.getSSHClient();
        String command = driver.buildCheckEnvironmentCommand();
        logger.fine("Check enviroment command line: " + command);
        CommandResult result = client.execute(command);
        if (result.getStatus() > 0) {
          logger.log(
            Level.WARNING, "Check environment return code: "
              + result.getStatus() + "\nOutput: " + result.getOutput()
              + "\nError: " + result.getError());
          return;
        }
        logger.fine(
          "Check environment return code: " + result.getStatus() + "\nOutput: "
            + result.getOutput() + "\nError: " + result.getError());
        setSGAInfo(driver.parseCheckEnvironmentOutput(result.getOutput()));
        //TODO Record in the log file the info
      }
      catch (Exception e) {
        logger.log(Level.SEVERE, "Erro checking the environment", e);
      }
      finally {
        if (client != null) {
          client.disconnect();
        }
      }
    }
  }
}
