/**
 * $Id: PBSDriver.java 172930 2016-04-29 17:45:41Z fpina $
 */

package csbase.sga.ssh.pbs;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import csbase.sga.executor.JobData;
import csbase.sga.executor.JobInfo;
import csbase.sga.monitor.SGAInfo;
import csbase.sga.ssh.SGADriver;
import sgaidl.COMMAND_CPU_TIME_SEC;
import sgaidl.COMMAND_EXEC_HOST;
import sgaidl.COMMAND_MEMORY_RAM_SIZE_MB;
import sgaidl.COMMAND_MEMORY_SWAP_SIZE_MB;
import sgaidl.COMMAND_PID;
import sgaidl.COMMAND_STATE;
import sgaidl.COMMAND_WALL_TIME_SEC;
import sgaidl.ProcessState;

/**
 * @author Tecgraf
 */
public class PBSDriver implements SGADriver {
	private static final String SUBMIT_JOB = "echo \"{0}\" | qsub";
	private static final String CHECK_JOB = "qstat -x {0}";
	private static final String KILL_JOB = "qdel {0}";
	private static final String CHECK_ALL_NODES_XML = "pbsnodes -x";
	private static final String CHECK_ALL_JOBS_XML = "qstat -x";

	private static Pattern JOB_ID_PATTERN = Pattern
			.compile("([^\\.]+)\\.(.*)\n");
	private static Pattern NODES_ATTRIBUTES_PATTERN = Pattern
			.compile("([^,=]+)=([^,]+)");
	private static Pattern JOBID_PATTERN = Pattern.compile("([^.]+).(\\S+)");
	private static Pattern TIME_INFO_PATTERN = Pattern
			.compile("(\\d+):(\\d+):(\\d+)");
	private static Pattern JOB_IDS_PATTERN = Pattern.compile("([^,]+)");
	private static Pattern BYTE_INFO_PATTERN = Pattern.compile("(\\d+)(\\S+)");

	private Properties properties;

	@Override
	public void init(Properties properties) {
		this.properties = properties;
	}

	@Override
	public String buildSubmitJobCommand(String script,
			Map<String, String> extraParam) {
		return MessageFormat.format(SUBMIT_JOB, script);
	}

	@Override
	public String buildKillJobCommand(JobData jobData) {
		PBSJobData data = (PBSJobData) jobData;
		return MessageFormat.format(KILL_JOB, data.getJobId());
	}

	@Override
	public String buildCheckJobCommand(JobData jobData) {
		PBSJobData data = (PBSJobData) jobData;
		return MessageFormat.format(CHECK_JOB, data.getJobId());
	}

	@Override
	public String buildCheckJobListCommand(JobData[] jobDataList) {
		return CHECK_ALL_JOBS_XML;
	}

	@Override
	public JobData parseJobSubmissionOutput(String output) {
		Matcher matcher = JOB_ID_PATTERN.matcher(output);
		if (matcher.matches()) {
			return new PBSJobData(matcher.group(1));
		} else {
			return null;
		}
	}

	@Override
	public JobInfo parseCheckJobOutput(JobData jobData, String output) {
		return parseXMLJobOutput(output).get(jobData);
	}

	private Map<JobData, JobInfo> parseXMLJobOutput(String output) {
		Map<JobData, JobInfo> jobsInfo = new HashMap<JobData, JobInfo>();
		if (output.isEmpty()) // no h jobs
			return jobsInfo;
		byte[] xmlbytes = output.getBytes();
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			ByteArrayInputStream input = new ByteArrayInputStream(xmlbytes);
			Document doc = builder.parse(input);
			doc.getDocumentElement().normalize();
			NodeList nList = doc.getElementsByTagName("Job");

			for (int temp = 0; temp < nList.getLength(); temp++) {
				Map<String, String> attrsMap = new HashMap<String, String>();
				Node nNode = nList.item(temp);
				if (nNode.getNodeType() == Node.ELEMENT_NODE) {
					Element jobElem = (Element) nNode;
					splitJobId(jobElem.getTextContent(), attrsMap);
					fillJobProperties(jobElem, attrsMap);
					JobInfo job = convertToJobInfo(attrsMap);
					jobsInfo.put(new PBSJobData(attrsMap.get("pid")), job);
					/*
					 * System.out.println(">>>"); for (Entry<String, String>
					 * entry : job.jobParam.entrySet())
					 * System.out.println(entry.getKey() + " = " +
					 * entry.getValue()); System.out.println("<<<");
					 */
				}
			}
			return jobsInfo;

		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		// TODO: definir como fica o retorno em caso de erro.
		return null;
	}

	private void fillJobProperties(Element jobElem, Map<String, String> attrsMap) {
		int resources_usedCounter = 0;
		NodeList atts = jobElem.getElementsByTagName("*");
		for (int i = 0; i < atts.getLength(); i++) {
			// No caso do Torque temos a chave "resources_used"
			// sobrescrita; temos que reconhecer os recursos
			// utilizados uma a um.
			if (atts.item(i).getNodeName().equals("resources_used")) {
				resources_usedCounter++;
				attrsMap.put(
						atts.item(i).getNodeName() + resources_usedCounter,
						atts.item(i).getTextContent());
			} else
				attrsMap.put(atts.item(i).getNodeName(), atts.item(i)
						.getTextContent());
		}
	}

	private void splitJobId(String jobId, Map<String, String> attrsMap) {

		Matcher attrsMatcher = JOBID_PATTERN.matcher(jobId);
		attrsMatcher.find();
		attrsMap.put("pid", attrsMatcher.group(1));
	}

	private JobInfo convertToJobInfo(Map<String, String> jobMap) {
		JobInfo jobInfo = new JobInfo();
		jobInfo.jobParam.put(COMMAND_PID.value, jobMap.get("pid"));
		jobInfo.jobParam.put(COMMAND_STATE.value,
				convertJobState(jobMap.get("job_state")).toString());
		String host = jobMap.get("exec_host");
		jobInfo.jobParam.put(COMMAND_EXEC_HOST.value, (host == null) ? ""
				: host);
		retrieveResourcesUsed(jobInfo, jobMap);
		return jobInfo;
	}

	private ProcessState convertJobState(String state) {
		switch (state) {

		case "C": // Job is completed after having run
			return ProcessState.FINISHED;
		case "E": // Job is exiting after having run
			return ProcessState.FINISHED;
		case "H": // Job is held
			return ProcessState.SLEEPING;
		case "Q": // job is queued, eligible to run or routed
			return ProcessState.WAITING;
		case "R": // job is running
			return ProcessState.RUNNING;
		case "T": // job is being moved to new location
			return ProcessState.RUNNING;
		case "W": // job is waiting for its execution time (-a option) to be
					// reached.
			return ProcessState.SLEEPING;
		case "S": // (Unicos only) job is suspend
			return ProcessState.FINISHED;
		default:
			return ProcessState.WAITING;
		}
	}

	private void retrieveResourcesUsed(JobInfo jobInfo,
			Map<String, String> jobMap) {
		// 1 -> cpu time
		// 2 -> RAM used
		// 3 -> Swap used
		// 4 -> walltime
		String ctimeS = jobMap.get("resources_used1");
		String ramS = jobMap.get("resources_used2");
		String swapS = jobMap.get("resources_used3");
		String walltimeS = jobMap.get("resources_used4");

		long ram = (ramS != null) ? getInfoInMb(ramS) : -1;
		long swap = (swapS != null) ? getInfoInMb(swapS) : -1;
		long ctime = (ctimeS != null) ? getInfoInSec(ctimeS) : -1;
		long walltime = (walltimeS != null) ? getInfoInSec(walltimeS) : -1;

		jobInfo.jobParam.put(COMMAND_MEMORY_RAM_SIZE_MB.value,
				Long.toString(ram));
		jobInfo.jobParam.put(COMMAND_MEMORY_SWAP_SIZE_MB.value,
				Long.toString(swap));
		jobInfo.jobParam.put(COMMAND_CPU_TIME_SEC.value, Long.toString(ctime));
		jobInfo.jobParam.put(COMMAND_WALL_TIME_SEC.value,
				Long.toString(walltime));
	}

	@Override
	public String buildCheckEnvironmentCommand() {
		return CHECK_ALL_NODES_XML;
	}

	@Override
	public SGAInfo parseCheckEnvironmentOutput(String output) {
		SGAInfo sga = new SGAInfo(properties);
		byte[] xmlbytes = output.getBytes();
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();

			ByteArrayInputStream input = new ByteArrayInputStream(xmlbytes);
			Document doc = builder.parse(input);
			doc.getDocumentElement().normalize();
			NodeList nList = doc.getElementsByTagName("Node");

			for (int temp = 0; temp < nList.getLength(); temp++) {
				Node nNode = nList.item(temp);
				PBSNodeInfo pbsnode = new PBSNodeInfo();
				if (nNode.getNodeType() == Node.ELEMENT_NODE) {
					Element nodeElem = (Element) nNode;
					NodeList atts = nodeElem.getElementsByTagName("*");
					for (int i = 0; i < atts.getLength(); i++)
						pbsnode.attMap.put(atts.item(i).getNodeName(), atts
								.item(i).getTextContent());
					if (pbsnode.attMap.get("state").equals("down")
							|| pbsnode.attMap.get("status") == null) {

						continue;
					}
					splitInfo(pbsnode);
				}

				long swapMem = pbsnode.totmemmb - pbsnode.physmemmb;
				long availableRAM = pbsnode.availmemmb >= pbsnode.physmemmb ? pbsnode.availmemmb
						- pbsnode.physmemmb
						: 0;
				long availableSwap = pbsnode.availmemmb - availableRAM;
				sga.addNode(pbsnode.attMap.get("uname"),
						//pbsnode.attMap.get("opsys"),
						null,
						pbsnode.attMap.get("ncpus"), "0",
						Long.toString(pbsnode.physmemmb),
						Long.toString(swapMem), Long.toString(availableRAM),
						Long.toString(availableSwap),
						pbsnode.attMap.get("loadave"),
						pbsnode.attMap.get("loadave"),
						pbsnode.attMap.get("loadave"),
						Integer.toString(pbsnode.njobs));
				/*
				 * System.out.println(">>>");
				 * System.out.println(pbsnode.attMap.get("uname") + "\n" +
				 * pbsnode.attMap.get("opsys") + "\n" +
				 * pbsnode.attMap.get("ncpus") + "\n" + pbsnode.physmemkb + "\n"
				 * + swapMem + "\n" + availableRAM + "\n" + availableSwap + "\n"
				 * + pbsnode.attMap.get("loadave") + "\n" + pbsnode.njobs);
				 * System.out.println("<<<");
				 */

			}
			return sga;

		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		// TODO: definir como fica o retorno em caso de erro.
		return null;
	}

	private void splitInfo(PBSNodeInfo pbsnode) {

		// nmero de jobs tem que ser contado antes de separar os atributos de
		// status, se no o valor ser sobrescrito.
		pbsnode.njobs = countJobs(pbsnode);

		Matcher attrsMatcher = NODES_ATTRIBUTES_PATTERN.matcher(pbsnode.attMap
				.get("status"));
		while (attrsMatcher.find()) {
			pbsnode.attMap.put(attrsMatcher.group(1), attrsMatcher.group(2));
		}

		pbsnode.totmemmb = getInfoInMb(pbsnode.attMap.get("totmem"));
		pbsnode.availmemmb = getInfoInMb(pbsnode.attMap.get("availmem"));
		pbsnode.physmemmb = getInfoInMb(pbsnode.attMap.get("physmem"));
	}

	private int countJobs(PBSNodeInfo pbsnode) {

		if (pbsnode.attMap.containsKey("jobs")) {
			Matcher jobsMatcher = JOB_IDS_PATTERN.matcher(pbsnode.attMap
					.get("jobs"));
			int jobCounter = 0;
			while (jobsMatcher.find())
				jobCounter++;
			return jobCounter;
		}
		return 0;
	}

	private long getInfoInMb(String string) {

		Matcher attrsMatcher = BYTE_INFO_PATTERN.matcher(string);
		attrsMatcher.find();
		long value = Long.parseLong(attrsMatcher.group(1));
		String measure = attrsMatcher.group(2);
		switch (measure) {
		case "b":
			value /= 1000000;
			break;
		case "kb":
			value /= 1000;
			break;
		case "mb":
			break;
		case "gb":
			value *= 1000;
			break;
		case "tb":
			value *= 1000000;
			break;
		default:
			value = 0;
			break;

		}

		return value;
	}

	private long getInfoInSec(String string) {
		// TODO: verificar como o Torque exibe jobs que rodam h pelo menos 1
		// dia
		Matcher attrsMatcher = TIME_INFO_PATTERN.matcher(string);
		attrsMatcher.find();
		return Long.parseLong(attrsMatcher.group(1)) * 3600
				+ Long.parseLong(attrsMatcher.group(2)) * 60
				+ Long.parseLong(attrsMatcher.group(3));
	}

	private class PBSNodeInfo {
		public int njobs;
		public long totmemmb;
		public long availmemmb;
		public long physmemmb;
		public Map<String, String> attMap = new HashMap<String, String>();

	}

	@Override
	public Map<JobData, JobInfo> parseCheckJobListOutput(JobData[] jobDataList, String output) {
		return parseXMLJobOutput(output);
	}
}
