/**
 * @file Arquivo principal de implementao do controle.
 * $Id: iupfilechooser.c 75808 2008-05-08 21:03:37Z clinio $
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <assert.h>

#ifdef Linux
#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>
#else
#include <direct.h>
#include <sys/stat.h>
#include "dirent.h"
#include "compat.h"
#endif

#include <iup.h>
#include <iupcpi.h>
#include <iuptree.h>

#include "iupfilechooser.h"

// ======================================================================= 

/** Nmero mximo de aninhamento de diretrios */
#define MAX_DEPTH 40

/** Nmero para controle de tamanho de vetores para
 *  a criao da rvore */
#define MAX_SIZE 20

/** Nmero mximo de extenses/filtros */
#define NUM_EXTENSIONS 10

/** Macro para liberao de memria */
#define FREE(obj){if (obj!= NULL){free(obj);obj=NULL;}}

/** Nome do filtro default */
#define DEFAULT_FILTER "Todos os arquivos (*.*)"

/** Tamanho mximo de arquivo */
#define MAX_FILE_PATH 512

// ======================================================================= 
// Estruturas para deixar o programa thread-safe
 
/** 
 * Estrutura de variveis de controle da aplicao  
 */
typedef struct{
  /** Mantm a hierarquia de diretrios pela qual se est entrando.
   Varivel depth indica o fim da lista em um determinado instante.
   Esta lista s  vlida no instante em que est sendo usada */
  //char* paths[MAX_DEPTH];

  /** profundidade do prximo n a ser inserido */
  int depth;

  /** id do ltimo n inserido */
  int id;

  /** vetor de descries de filtros permitidos */
  char* filters[NUM_EXTENSIONS];  
  
  /** vetor de extenses de cada um dos filtros
   *  descritos em filters */
  char* filters_extensions[NUM_EXTENSIONS];
  
  /** extenses selecionadas pelo usurio a partir do filtro */
  char* extensions[NUM_EXTENSIONS]; 

  /** define se foi requisitada uma atualizao da rvore. */
  int refreshTree;
} Globals;

// -----------------------------------------------------------------------
/** 
 * Estrutura de elementos IUP  
 */
typedef struct{

  /** dilogo principal de manipulao de arquivos */
  Ihandle* dialog;

  /** caixa de texto onde  apresentado o nome do diretrio a partir do qual 
   *  a caixa de dilogo est sendo aberta */
  Ihandle* directoryField;

  /** tree que detm a rvore de diretrios do projeto corrente */
  Ihandle* tree;

  /** campo que apresenta o nome do arquivo selecionado pelo usurio */
  Ihandle* selectionField;
  
  /** elemento IUP onde so inseridas as entradas do filtro do dilogo */
  Ihandle* filter;

  /** Boto de cancelar operao */
  Ihandle *cancelButton;

  /** Boto de confirmar operao */
  Ihandle *actionButton;
} Widgets;

// ======================================================================= 
// Prottipos das funes internas do mdulo
// ======================================================================= 

// -----------------------------------------------------------------------
// Mtodos IUP                                              
char *iupGetEnv(Ihandle *n, char *attr);
void iupSetEnv(Ihandle *n, char *attr, char *attr_value);

// -----------------------------------------------------------------------
// Callbacks 
static int cancel_cb(Ihandle * self);
static int action_cb(Ihandle * self);
static int selection_cb(Ihandle * self, int c, char* after);
static int selectionNode_cb(Ihandle *self, int id, int status);
static int branchopen_cb(Ihandle *self, int id);
static int branchclose_cb(Ihandle *self, int id);
static int filter_cb (Ihandle *self, char *t, int i, int v);

// -----------------------------------------------------------------------
// Mtodos auxiliares na construo do dilogo

static char* getCurrentDir();
static Ihandle* createDirectoryBox(Widgets *wStruct, char* directory);
static void createTree(char* initDirectory, Widgets *wStruct, Globals *gStruct);
static void updateTree(char* initDirectory, Widgets *wStruct, Globals *gStruct);
static void buildSubTree(Widgets *wStruct, Globals *gStruct, int id);
static Ihandle* createSelectionBox(Widgets *wStruct, Globals *gStruct);
static Ihandle* createFilterBox(Widgets *wStruct, Globals *gStruct);
static void updateFilterBox(Widgets *wStruct, Globals *gStruct, 
    char* newFilter, char* att_name);
static Ihandle* createButtonsBox(Widgets *wStruct, Globals *gStruct);
static void updateButton(Widgets *wStruct);
static int browseDirectory(
  Widgets *wStruct, Globals *gStruct, DIR* dirOpened, char* dirName, int dirId, 
  int depth, int maxDepth);
static void findFilterAcceptedExtensions(Widgets *wStruct, Globals *gStruct);
static int isAcceptedExtension(char* name, char* exts[NUM_EXTENSIONS]);
static void verifyOpendirError(char* directory);
static void removeChildren(Widgets *wStruct, Globals *gStruct, int dirId);

static Ihandle* IupFC(void);

static Ihandle *iupFileChooserCreate(Iclass *ic);
static void iupFileChooserRefresh(Ihandle* handle);
static void iupFileChooserDestroy(Ihandle* handle);
static void iupFileChooserSetAttribute(Ihandle* handle, char* att_name, char* att_value);
static char* iupFileChooserGetAttribute(Ihandle* handle, char* att_name);

// =========================================================================
// Variveis globais ao mdulo
// =========================================================================

//* Varivel que representa a classe do controle CPI 
static Iclass *ICFileChooser = NULL;

//* para manipulao de erros
extern int errno; 

//* Vetor de extenses que no devem ser permitidas no dilogo
static char* notAcceptedExt[NUM_EXTENSIONS]={".csbase", ".csbase_description"};

// ======================================================================
// Mtodos visveis da biblioteca
// ======================================================================

// .....................................................................
/**
 * Mtodo de inicializao do controle filechooser. 
 */
void IupFileChooserOpen(void) 
{
}

// .....................................................................
/**
 * Funo que cria uma instncia de filechoooser.
 * @return um handle do dilogo de arquivos
 */
Ihandle* IupFileChooser(void) 
{
  return IupFC();
}

void IupFileChooserDestroy(Ihandle* h) 
{
  iupFileChooserDestroy(h);
}

// .....................................................................
/**
 * Mtodo de finalizao do filechoooser.
 */
void IupFileChooserClose(void) 
{
}

void IupFileChooserSetAttribute(Ihandle* handle, char* att_name, char* att_value)
{
  iupFileChooserSetAttribute(handle, att_name, att_value);
}

char* IupFileChooserGetAttribute(Ihandle* handle, char* att_name) 
{
  return iupFileChooserGetAttribute(handle, att_name);
}

void IupFileChooserPopup(Ihandle* handle, int x, int y)
{
  iupFileChooserRefresh(handle);
  IupPopup(handle, x, y);
}

// ======================================================================
// Mtodos auxiliares 
// ======================================================================

static char* stripFileName(const char* dir)
{
  char* name = dir;
  while (*dir != '\0') {
    if (*dir == '/' || *dir == '\\')
      name = dir + 1;
    dir++;
  }
  return name;
}

static void stripFilePath(const char* dir, char* path)
{
  char* idir = dir;
  char* name = dir;
  while (*idir != '\0') {
    if (*idir == '/' || *idir == '\\')
      name = idir + 1;
    idir++;
  }
  idir = dir;
  while (idir != name) {
    *path = *idir;
    path++; idir++;
  }
}

// .....................................................................
/**
 * Criao de um controle.
 * @param ic a classe IUP
 * @return o objeto criado.
 */
static Ihandle* iupFileChooserCreate(Iclass *ic) {
  if (ic == NULL) return NULL;
  return IupFC();
}

static void iupFileChooserRefresh(Ihandle* handle)
{
  Globals* gStruct = NULL;
  Widgets* wStruct = NULL;
  char* directory;

  gStruct = (Globals*)IupGetAttribute(handle, "FC_GLOBALS");
  wStruct = (Widgets*)IupGetAttribute(handle, "FC_WIDGETS");
  directory = IupGetAttribute(wStruct->dialog, IUP_DIRECTORY);
  if (directory == NULL)
    directory = getCurrentDir();
  updateTree(directory, wStruct, gStruct);
}

// .....................................................................
/**
 * Criao de um fill com tamanho especificado.
 * @param rsize o tamanho como uma string.
 * @return um elemento IUP (fill).
 */
static Ihandle* createFill(int rsize) {
   char* buff = (char*) malloc(sizeof(char)*10);
   Ihandle* f = IupFill();
   sprintf(buff, "%d", rsize);
   IupStoreAttribute(f, IUP_RASTERSIZE, buff);
   free(buff);
   return f;
}

// .....................................................................
/**
 * Obtm o nome completo do diretrio corrente.
 * @return directory o nome do diretrio corrente
 */
static char* getCurrentDir(){
  char directory[MAX_FILE_PATH];
  // diretrio corrente  o diretrio default
  //#ifdef WIN32
  //    _getcwd(directory, MAX_FILE_PATH);
  //#else
  getcwd(directory, MAX_FILE_PATH);
  //#endif
  return strdup(directory);
}

/**
 * Obtm o caminho completo de um n da rvore.
 * handle
 */
static void getNodePath(Widgets *wStruct, Globals *gStruct, int id, char* path)
{
  int nodes[MAX_DEPTH];
  int nnodes = 0;
  int i;

  while (id != 0) {
    nodes[nnodes++] = id;
    id = IupTreeGetInt(wStruct->tree, "PARENT", id);
  }
//TODO: otimizar
  strcpy(path, IupGetAttribute(wStruct->dialog, IUP_DIRECTORY));
  for (i = nnodes-1; i >= 0; i--) {
    strcat(path, "/");
    strcat(path, IupTreeGetAttribute(wStruct->tree, "NAME", nodes[i]));
  }
}


// .....................................................................
/**
 * Libera todo o espaco de memria alocado para este elemento IUP (handle)
 * @param handle handle do elemento IUP a ser destrudo e ao 
 *        qual est vinculada a execuo corrente.
 */
static void iupFileChooserDestroy(Ihandle* handle) {
  Globals* gStruct = NULL;
  Widgets* wStruct = NULL;

  gStruct = (Globals*)IupGetAttribute(handle, "FC_GLOBALS");
  wStruct = (Widgets*)IupGetAttribute(handle, "FC_WIDGETS");

  if (gStruct != NULL) {
/*
     int i = 0;
     while(gStruct->paths[i] != NULL){
        FREE(gStruct->paths[i]);
        assert(i<MAX_DEPTH);
        i++;
     }
*/
     int i = 0;
     while(i < NUM_EXTENSIONS){
        FREE(gStruct->filters[i]);
        FREE(gStruct->filters_extensions[i]);
        FREE(gStruct->extensions[i]);
        i++;
     }
  }
  FREE(handle);
}

// .....................................................................
/**
 * Funo principal que cria a janela de dilogo.
 * @return Ihandle* handle do dilogo
 */
static Ihandle* IupFC(void) {

  int e = 0;

  Ihandle* dialog = NULL;
  Globals *gStruct = NULL;
  Widgets *wStruct = NULL;

  /** armazena temporariamente o diretrio default */
  char* directory;
  directory = getCurrentDir();
  
  gStruct = (Globals*) malloc(sizeof(Globals));
  wStruct = (Widgets*) malloc(sizeof(Widgets));

  // inicializa campos da estrutura de widgets com NULL
  wStruct->dialog = NULL;
  wStruct->directoryField = NULL;
  wStruct->tree = NULL;
  wStruct->selectionField = NULL;
  wStruct->filter = NULL;
  wStruct->actionButton = NULL;
  wStruct->cancelButton = NULL;

  // inicializa vetor de extenses permitidas com NULL
  for (e = 0; e < NUM_EXTENSIONS; e++) {
     gStruct->filters[e] = NULL;
     gStruct->filters_extensions[e] = NULL;
     gStruct->extensions[e] = NULL;
  }
  gStruct->refreshTree = 1;
  
  // para evitar que novas rvores sejam criadas quando diretrio
  // ao qual se refere for atualizado
  wStruct->tree = IupTree();
  IupSetAttribute(wStruct->tree, IUP_FONT, IUP_HELVETICA_NORMAL_12);

  // criao do dilogo principal
  dialog = IupDialog(
      IupVbox(
         createDirectoryBox(wStruct, directory),
         createFill(5),
         IupFrame(wStruct->tree),
         createSelectionBox(wStruct, gStruct),
         createFill(5),
         createFilterBox(wStruct, gStruct),
         createFill(10),
         createButtonsBox(wStruct, gStruct),
         NULL
      )
  );

  IupSetAttribute(dialog, IUP_MARGIN, "7x7");
  IupSetAttribute(dialog, IUP_FONT, IUP_HELVETICA_BOLD_12);
  IupSetAttribute(dialog, IUP_TITLE, "");
  IupSetAttribute(dialog, IUP_RESIZE, IUP_YES);
  IupSetAttribute(dialog, IUP_MAXBOX, IUP_NO);
  IupSetAttribute(dialog, IUP_MINBOX, IUP_NO);

  IupMap(dialog);

  IupSetAttribute(dialog, IUP_DEFAULTESC, "Cancelar");
  // TODO: Como fazer para no deixar o usurio acionar boto
  // se ele estiver desabilitado aqui?
  IupSetAttribute(dialog, IUP_DEFAULTENTER, "actionButton");
  // Fechar a janela deve ser o mesmo que pressionar o boto de cancelar
  IupSetAttribute(dialog, IUP_CLOSE_CB, "cancel_cb");

  wStruct->dialog = dialog;

  IupSetAttribute(dialog, "FC_WIDGETS", (char*)wStruct);
  IupSetAttribute(dialog, "FC_GLOBALS", (char*)gStruct);

  // imitando o IUP_VALUE de IupFileDlg
  IupSetAttribute(dialog, IUP_VALUE, NULL);

  // atributo SELECTED_NODE armazena o id do n selecionado
  // No incio da execuo, n selecionado  o n raiz
  IupStoreAttribute(dialog, "SELECTED_NODE", "0");

  // imitando o IUP_DIALOGTYPE de IupFileDlg
  if (!strcmp(IupGetAttribute(wStruct->actionButton, IUP_TITLE), "Abrir")) {
     IupSetAttribute(dialog, IUP_DIALOGTYPE, IUP_OPEN);
  } else {
     IupSetAttribute(dialog, IUP_DIALOGTYPE, IUP_SAVE);
  }

  IupSetAttribute(wStruct->dialog, IUP_STATUS, "-1");
  createTree(directory, wStruct, gStruct);
  FREE(directory);  
  return dialog;
}

// .....................................................................
/**
 * Mtodo auxiliar para permitir que o usurio possa alterar valores
 * de determinados parmetros, sem ter conhecimento de como o cdigo est
 * implementado, mas apenas seguindo o padro IUP
 * @param handle handle do elemento IUP a partir do qual se quer atribuir
 *        valor a um atributo
 * @param att_name nome do atributo para o qual se deseja definir novo valor
 * @param att_value novo valor para o atributo
 */
static void iupFileChooserSetAttribute(Ihandle* handle, char* att_name, 
char* att_value) {

  Widgets* wStruct = NULL;
  Globals* gStruct = NULL;

  gStruct = (Globals*)IupGetAttribute(handle, "FC_GLOBALS");
  wStruct = (Widgets*)IupGetAttribute(handle, "FC_WIDGETS");
  
  if (strcmp(att_name, IUP_TITLE) == 0) {
    IupStoreAttribute(wStruct->dialog, IUP_TITLE, att_value);
  } else if (strcmp(att_name, IUP_DIRECTORY) == 0){
    if (strcmp(IupGetAttribute(wStruct->dialog, IUP_VISIBLE), IUP_YES) == 0) {
      updateTree(att_value, wStruct, gStruct);
    } else {
      IupStoreAttribute(wStruct->dialog, IUP_DIRECTORY, att_value);
      gStruct->refreshTree = 1;      
    }
  } else if (strcmp(att_name, IUP_DIALOGTYPE) == 0){

     if (strcmp(att_value, IUP_SAVE) == 0){ 
       IupSetAttribute(wStruct->dialog, IUP_DIALOGTYPE, IUP_SAVE);
       IupSetAttribute(wStruct->selectionField, IUP_READONLY, IUP_NO);
     } else {
       IupSetAttribute(wStruct->dialog, IUP_DIALOGTYPE, IUP_OPEN);
       IupSetAttribute(wStruct->selectionField, IUP_READONLY, IUP_YES);
     }
     updateButton(wStruct);
        
  } else if ((strcmp(att_name, IUP_EXTFILTER) == 0) ||
        (strcmp(att_name, IUP_FILTER) == 0)){
     // EXTFILTER tem sempre prioridade sobre FILTER e FILTERINFO
     if (IupGetAttribute(handle, IUP_EXTFILTER) == NULL || 
         strcmp(att_name, IUP_EXTFILTER) == 0){
        IupStoreAttribute(wStruct->dialog, att_name, att_value);
        updateFilterBox(wStruct, gStruct, att_value, att_name);
        // se usurio definiu IUP_EXTFILTER, 
        // ento no  vlido IUP_FILTERINFO e nem FILTER
        if (strcmp(att_name, IUP_EXTFILTER) == 0){
           IupSetAttribute(wStruct->dialog, IUP_FILTERINFO, NULL);
           IupSetAttribute(wStruct->dialog, IUP_FILTER, NULL);
        } else if (gStruct->filters[0] == NULL){
//               IupStoreAttribute(wStruct->filter, "1", gStruct->filters[1]);
//               IupStoreAttribute(handle, att_name, att_value);
        }
     } else {
         IupSetAttribute(wStruct->dialog, IUP_FILTER, NULL);
     }
  } else if (strcmp(att_name, IUP_FILTERINFO) == 0){
       // IUP_FILTERINFO s pode ser setada se for definido um nico filtro. 
       // Se IUP_EXTFILTER, ento FILTERINFO no se aplica
       int size=0;
       while(gStruct->filters_extensions[size] != NULL){
         size++;
       }
       // se existe algum filtro definido, que no seja o default
       if (size > 1){
          IupSetAttribute(handle, att_name, NULL);
       } else {
 	        // se j existe a extenso do filtro definida,
	        // ento descrio j pode aparecer no combo de filtros
	        if (gStruct->filters_extensions[0] != NULL){
	           IupStoreAttribute(handle, att_name, att_value);
	           IupStoreAttribute(wStruct->filter, "1", att_value);
	        } else{
              IupSetAttribute(handle, att_name, NULL);
	        } 
	       // de qualquer forma, descrio fica armazenada na estrutura.
          // Se depois usurio setar filtro, esta descrio  colocada no
          // combo atravs da chamada a updateFilterBox
          gStruct->filters[0] = strdup(att_value);
       }
   } else {
        // se no for atributo que eu mesma trate, IUP deve tratar
        iupCpiDefaultSetAttr(handle, att_name, att_value);
   }
}

// .....................................................................
/**
 * Funo auxiliar para permitir que o usurio possa buscar alguns valores
 * de parmetros, sem ter conhecimento de como o cdigo est
 * implementado, mas apenas seguindo o padro IUP.
 * @param handle handle do elemento IUP ao qual o usurio est se referindo
 * @param att_name nome do parmetro do qual deseja obter o valor
 * @return char* valor do parmetro
 */
static char* iupFileChooserGetAttribute(Ihandle* handle, char* att_name) {
  Globals* gStruct = NULL;
  Widgets* wStruct = NULL;

  gStruct = (Globals*)IupGetAttribute(handle, "FC_GLOBALS");
  wStruct = (Widgets*)IupGetAttribute(handle, "FC_WIDGETS");
  
  if (strcmp(att_name, IUP_VALUE) == 0 && wStruct->dialog != NULL){
    if((IupGetAttribute(wStruct->dialog, IUP_VALUE) != NULL) &&
       (IupGetAttribute(wStruct->dialog, "SELECTED_NODE") != NULL)){
      if (strcmp(IupGetAttribute(wStruct->dialog, "SELECTED_NODE"), "-1") == 0){
         return NULL;
      }
      return strdup(IupGetAttribute(wStruct->dialog, IUP_VALUE));
    }
    return NULL;
  } else if (strcmp(att_name, IUP_TITLE) == 0){
       return IupGetAttribute(handle, att_name);
  } else if (strcmp(att_name, IUP_STATUS) == 0){
      return strdup(IupGetAttribute(wStruct->dialog, IUP_STATUS));
  } else if (strcmp(att_name, IUP_FILTERINFO) == 0 || strcmp(att_name, IUP_FILTER) == 0){
      return IupGetAttribute(handle, att_name);
  }
  // se atributo no  tratado neste mtodo, ento o drive IUP  consultado
  // e o valor retornado corresponde ao valor default do elemento IUP handle
  return iupCpiDefaultGetAttr(handle, att_name);
}

// .....................................................................
/**
 * Cria a caixa de texto que indica o diretrio corrente sobre o
 * qual o usurio deve realizar a operao de abrir/salvar arquivo
 * @param directory diretrio corrente
 * @return Ihandle* handle do box que representa a interface com o nome do projeto
 */
static Ihandle* createDirectoryBox(Widgets *wStruct, char* directory){

  Ihandle* dirBox = NULL;
  Ihandle* dir = NULL;
  char* dirname;

  dir = IupText("do_nothing");
  dirname = stripFileName(directory);
  IupStoreAttribute(dir, IUP_VALUE, dirname);
  IupSetAttribute(dir, IUP_EXPAND, IUP_HORIZONTAL);
  IupSetAttribute(dir, IUP_READONLY, IUP_YES);
  IupSetAttribute(dir, IUP_FONT, IUP_HELVETICA_NORMAL_12);

  dirBox = IupHbox(
    IupLabel("Diretrio:"), IupFill(), dir,
    NULL
  );

  wStruct->directoryField = dir;
  return dirBox;
}

// .....................................................................
/**
 * Cria a rvore que representa a poro do sistema de arquivos
 * relativa ao projeto escolhido (atravs do caminho indicado)
 * @param initDirectory diretrio a partir do qual o dilogo ser construdo
 * @param wStruct estrutura que representa os widgets da interface
 * @param gStruct estrutura que apresenta as variveis de controle da aplicao
 */
static void createTree(char* initDirectory, Widgets *wStruct, Globals *gStruct)
{
  IupSetHandle("tree",wStruct->tree);
  IupSetAttribute(wStruct->tree, IUP_REDRAW, IUP_YES);
  // para que a rvore seja criada com os ns colapsados
  IupSetAttribute(wStruct->tree, IUP_ADDEXPANDED, IUP_NO);
  IupSetAttribute(wStruct->tree, IUP_VALUE, IUP_ROOT);

  IupSetFunction("selectionNode_cb", (Icallback)selectionNode_cb);
  IupSetAttribute(wStruct->tree, IUP_SELECTION_CB, "selectionNode_cb");

  IupSetFunction("branchopen_cb", (Icallback)branchopen_cb);
  IupSetAttribute(wStruct->tree, IUP_BRANCHOPEN_CB, "branchopen_cb");

  IupSetFunction("branchclose_cb", (Icallback)branchclose_cb);
  IupSetAttribute(wStruct->tree, IUP_BRANCHCLOSE_CB, "branchclose_cb");

  IupSetAttribute(wStruct->tree, "FC_WIDGETS", (char *)wStruct);
  IupSetAttribute(wStruct->tree, "FC_GLOBALS", (char *)gStruct);

  IupStoreAttribute(wStruct->dialog, IUP_DIRECTORY, initDirectory);
  gStruct->refreshTree = 1;
  //updateTree(initDirectory, wStruct, gStruct);
}

// .....................................................................
/**
 * Atualiza a rvore que representa a poro do sistema de arquivos
 * relativa ao projeto escolhido (atravs do caminho indicado)
 * @param initDirectory diretrio a partir do qual o dilogo ser construdo
 * @param wStruct estrutura que representa os widgets da interface
 * @param gStruct estrutura que apresenta as variveis de controle da aplicao
 */
static void updateTree(char* initDirectory, Widgets *wStruct, Globals *gStruct)
{
  char* currentDir;
  char* oldcursor;
  DIR* dirOpened=NULL;
  int p;

  currentDir = getCurrentDir();

  // a criao da rvore que representa o projeto deve iniciar
  // do diretrio-raiz desse projeto
  dirOpened = opendir(initDirectory);
  if (dirOpened == NULL){
    printf("Error -- ");
    if (errno == EACCES)       printf("Permission denied: [%s].\n", initDirectory);
    else verifyOpendirError(initDirectory);
    if (wStruct->dialog != NULL){
      // quando ocorre algum erro na atribuio do novo diretrio,
      // ento diretrio corrente  o padro. No entanto,de modo
      // diferente do que ocorre no IupFileDlg, o diretrio corrente
      // fica setado no IUP_DIRECTORY, e no o diretrio que o usurio
      // informou (e que no pode ser setado). Se quiser o mesmo
      // comportamento, ser preciso acrescentar um novo campo na
      // estrutura de variveis globais
      IupStoreAttribute(wStruct->dialog, IUP_DIRECTORY, currentDir);
      FREE(currentDir);
      return;
    }
    exit (1);
  }

  gStruct->depth = 1;
  gStruct->id = 0;

  // dialog est NULL se esta chamada foi feita a partir do construtor
  // de IupFileChooser. Nos outros casos, dialog no  nulo.
  if (wStruct->dialog != NULL){
    char* currentDirName = currentDir + (int)(stripFileName(initDirectory) - initDirectory);
    // imitando o IUP_DIRECTORY de IupFileDlg
    if (strcmp(initDirectory, ".") == 0 || strcmp(initDirectory, "pwd") == 0){
      IupStoreAttribute(wStruct->directoryField, IUP_VALUE, currentDirName);
      IupStoreAttribute(wStruct->dialog, IUP_DIRECTORY, currentDir);
    } else {
      IupStoreAttribute(wStruct->directoryField, IUP_VALUE, stripFileName(initDirectory));
      IupStoreAttribute(wStruct->dialog, IUP_DIRECTORY, initDirectory);
    }
  }
  IupStoreAttribute(wStruct->selectionField, IUP_VALUE, NULL);
  // primeiro n da rvore (a raiz), tem como path o diretrio
  // inicial e a profundidade igual a 0
  IupStoreAttribute(wStruct->tree, "PATH0", initDirectory);
  IupSetAttribute(wStruct->tree, "DEPTH0", "0");
  removeChildren(wStruct, gStruct, 0);
  oldcursor = IupGetAttribute(wStruct->dialog, "CURSOR");
  IupSetAttribute(wStruct->dialog, "CURSOR", IUP_BUSY);
  browseDirectory(wStruct, gStruct, dirOpened, initDirectory, 0, 0, 1);
  IupSetAttribute(wStruct->dialog, "CURSOR", oldcursor);
  IupSetAttribute(wStruct->tree, "VALUE", "0");
  IupSetAttribute(wStruct->tree, "REDRAW", IUP_YES);
  gStruct->refreshTree = 0;
  //closedir(dirOpened);
  FREE(currentDir);
}


static void buildSubTree(Widgets *wStruct, Globals *gStruct, int id)
{
  char initDirectory[MAX_FILE_PATH];
  char buffer[64];
  DIR* dirOpened = NULL;
  char* oldcursor;
  char* currentDir;
  int depth;

  currentDir = getCurrentDir();

  getNodePath(wStruct, gStruct, id, initDirectory);
  depth = IupTreeGetInt(wStruct->tree, "DEPTH", id);
  dirOpened = opendir(initDirectory);
  if (dirOpened == NULL) {
    printf("Error -- ");
    if (errno == EACCES)       printf("Permission denied: [%s].\n", initDirectory);
    else verifyOpendirError(initDirectory);
    if (id == 0 && wStruct->dialog != NULL){
      // quando ocorre algum erro na atribuio do novo diretrio,
      // ento diretrio corrente  o padro. No entanto,de modo
      // diferente do que ocorre no IupFileDlg, o diretrio corrente
      // fica setado no IUP_DIRECTORY, e no o diretrio que o usurio
      // informou (e que no pode ser setado). Se quiser o mesmo
      // comportamento, ser preciso acrescentar um novo campo na
      // estrutura de variveis globais
      IupStoreAttribute(wStruct->dialog, IUP_DIRECTORY, currentDir);
      FREE(currentDir);
      return;
    }
    exit (1);
  }

  // a criao da rvore que representa o projeto deve iniciar
  // do diretrio-raiz desse projeto
  if (id == 0) {
    gStruct->depth = depth;
    gStruct->id = id;
    
    // dialog est NULL se esta chamada foi feita a partir do construtor
    // de IupFileChooser. Nos outros casos, dialog no  nulo.
    if (wStruct->dialog != NULL){
      char* currentDirName = currentDir + (int)(stripFileName(initDirectory) - initDirectory);
      // imitando o IUP_DIRECTORY de IupFileDlg
      if (strcmp(initDirectory, ".") == 0 || strcmp(initDirectory, "pwd") == 0){
        IupStoreAttribute(wStruct->directoryField, IUP_VALUE, currentDirName);
        IupStoreAttribute(wStruct->dialog, IUP_DIRECTORY, currentDir);
      } else {
        IupStoreAttribute(wStruct->directoryField, IUP_VALUE, stripFileName(initDirectory));
        IupStoreAttribute(wStruct->dialog, IUP_DIRECTORY, initDirectory);
      }
    }
    // primeiro n da rvore (a raiz), tem como path o diretrio
    // inicial e a profundidade igual a 0
    IupStoreAttribute(wStruct->tree, "PATH0", initDirectory);
  }

  sprintf(buffer, "%d", depth);  
  oldcursor = IupGetAttribute(wStruct->dialog, "CURSOR");
  IupSetAttribute(wStruct->dialog, "CURSOR", IUP_BUSY);
  IupTreeStoreAttribute(wStruct->tree, "DEPTH", id, buffer);
  removeChildren(wStruct, gStruct, id);
  browseDirectory(wStruct, gStruct, dirOpened, initDirectory, id, depth, depth + 1);
  IupSetAttribute(wStruct->dialog, "CURSOR", oldcursor);
  IupSetAttribute(wStruct->tree, IUP_REDRAW, IUP_YES);
  //closedir(dirOpened);
  FREE(currentDir);
}



// .....................................................................
/**
 * Cria o box com o nome do arquivo selecionado.
 * @param wStruct estrutura que representa os widgets da interface
 * @param gStruct estrutura que apresenta as variveis de controle da aplicao
 * @return Ihandle* handle do box que apresenta o nome do arquivo selecionado
*/
static Ihandle* createSelectionBox(Widgets *wStruct, Globals *gStruct){

  Ihandle* selectionField = NULL;
  Ihandle* selectionBox = NULL;

  selectionField = IupText("selection_cb");
  IupSetFunction("selection_cb", (Icallback) selection_cb);
  IupSetAttribute(selectionField, IUP_EXPAND, IUP_HORIZONTAL);
  IupSetAttribute(selectionField, IUP_READONLY, IUP_YES);
  IupSetAttribute(selectionField, IUP_FONT, IUP_HELVETICA_NORMAL_12);

  selectionBox = IupHbox(
    IupLabel("Arquivo corrente:"),
    IupFill(),
    selectionField,
    NULL
  );

  IupSetAttribute(selectionBox, IUP_ALIGNMENT, "ACENTER");

  wStruct->selectionField = selectionField;

  IupSetAttribute(selectionField, "FC_WIDGETS", (char *)wStruct);
  IupSetAttribute(selectionField, "FC_GLOBALS", (char *)gStruct);

  return selectionBox;
}

// .....................................................................
/**
 * Cria o box com os possveis filtros de arquivo.
 * @param wStruct estrutura que representa os widgets da interface
 * @param gStruct estrutura que apresenta as variveis de controle da aplicao
 * @return Ihandle* handle do box de filtros de arquivo
 */
static Ihandle* createFilterBox(Widgets *wStruct, Globals *gStruct){

  Ihandle* filterList_dropdown = NULL;
  Ihandle* frm_filter = NULL;

  // na inicializao, filtro comea aceitando todos os arquivos
  filterList_dropdown = IupList(NULL);
  IupSetAttribute(filterList_dropdown, IUP_FONT, IUP_HELVETICA_NORMAL_12);
  IupSetAttribute(filterList_dropdown, "1", DEFAULT_FILTER);
  IupSetAttributes(filterList_dropdown, 
      "DROPDOWN=YES, VISIBLE_ITEMS=1, EXPAND=HORIZONTAL, ACTION=filter_act");
  IupSetFunction("filter_act", (Icallback) filter_cb);
  
  frm_filter = IupFrame (filterList_dropdown);
  IupSetAttribute (frm_filter, IUP_TITLE, "Filtro");

  IupSetAttribute(filterList_dropdown, "FC_WIDGETS", (char *)wStruct);
  IupSetAttribute(filterList_dropdown, "FC_GLOBALS", (char *)gStruct);

  wStruct->filter = filterList_dropdown;
  return frm_filter;
}

// .....................................................................
/**
 * Atualiza filtro de arquivos.
 * @param wStruct estrutura que representa os widgets da interface
 * @param gStruct estrutura que apresenta as variveis de controle da aplicao
 * @param newFilter nova string de filtro definida
 * @param att_name nome do atributo atravs do qual est sendo setado o filtro
 */
static void updateFilterBox(Widgets *wStruct, Globals *gStruct, 
    char* newFilter, char* att_name) 
{
  char* tmp;
  char* filter = strdup(newFilter);
  char* aux;
  char* iDescription;
  char att_num[MAX_SIZE];
  // se filtro  informado, ento no  usado DEFAULT_FILTER
  int n = 1;
  int e;
  
  // limpa lista de extenses 
  gStruct->extensions[0] = NULL;
  
  if (strcmp(att_name, IUP_EXTFILTER) == 0) {
    char* ptmp;
    char* paux;

    ptmp = (char *) malloc (strlen(newFilter) + 1);
    iDescription = (char *) malloc (strlen(newFilter) + 1);
    paux = (char *) malloc (strlen(newFilter) + 1);

    tmp = ptmp;
    aux = paux;

    iDescription[0]='\0';
    aux = strchr(filter, '|');
    while (aux != NULL){
      tmp = aux+1;
      aux[0] = '\0';
      gStruct->filters[n-1] = strdup(filter);
      filter = tmp;
      aux = strchr(filter, '|');
      sprintf(att_num, "%d", n);
      // par descrio-extenso
      if (aux != NULL){
        // no filtro, s aparece descrio informada
        IupStoreAttribute(wStruct->filter, att_num, gStruct->filters[n-1]);
        tmp = aux+1;
        aux[0] = '\0';
        gStruct->filters_extensions[n-1] = strdup(filter);
        filter = tmp;
        aux = strchr(filter, '|');
        n++;
        // para garantir que nmero de extenses no  maior do que NUM_EXTENSIONS
        assert(n<NUM_EXTENSIONS);
      } else {
        if (filter != NULL){
          gStruct->filters_extensions[n-1] = strdup(filter);
          IupStoreAttribute(wStruct->filter, att_num, gStruct->filters[n-1]);
          n++;
          assert(n<NUM_EXTENSIONS);	   
        } else{
          // usurio informou nmero de descries de filtro
          // diferente do nmero de extenses permitidas por
          // filtro. Desconsidera tentativa de setar filtro
          IupSetAttribute(wStruct->filter, "0", NULL);
          gStruct->filters[0]=NULL;
          gStruct->filters_extensions[0]=NULL;
          gStruct->extensions[0]=NULL;
          return;
        }
      }
    }
    sprintf(att_num, "%d", n);
    IupStoreAttribute(wStruct->filter, att_num, NULL);
    FREE(iDescription);
    FREE(paux);
    FREE(ptmp);
  } else {
    char* filter_tmp;
    // se descrio j foi setada, ento a coloca no registro do filtro. 
    // Seno, isso  feito posteriormente, quando usurio seta info
    filter = gStruct->filters[0];
    if (filter != NULL && strlen(filter)>0){
      filter_tmp = (char *) malloc (strlen(newFilter));
      strcpy(filter_tmp, filter);
    } else{
      filter_tmp = strdup(newFilter);	    
    }
    gStruct->filters_extensions[n-1] = strdup(newFilter);
    sprintf(att_num, "%d", n);
    IupStoreAttribute(wStruct->filter, att_num, filter_tmp);
    sprintf(att_num, "%d", n + 1);
    IupStoreAttribute(wStruct->filter, att_num, NULL);
    FREE(filter_tmp);
//    IupSetAttribute(wStruct->filter, IUP_DROPDOWN, IUP_NO);
    IupStoreAttribute(wStruct->dialog, IUP_FILTERINFO, filter);
  }
 
  e = n;
  for (e = n; e < NUM_EXTENSIONS; e++) {
     gStruct->filters[e] = NULL;
     gStruct->filters_extensions[e]=NULL;
  }
  
  sprintf(att_num, "%d", n-1);
  // seta tamanho da lista que representa os filtros
  IupStoreAttribute(wStruct->filter, IUP_SIZE, att_num);
  // para que no comboBox de filtros apaream tantas linhas quantas forem
  // as descries de filtros
  IupStoreAttribute(wStruct->filter, IUP_VISIBLE_ITEMS, att_num);

  if (n >= 1) IupSetAttribute(wStruct->filter, IUP_VALUE, "1");
  
  // fim da lista de filtros no comboBox
  n++;
  sprintf(att_num, "%d", n);
  IupSetAttribute(wStruct->filter, att_num, NULL);
   
  // apesar deste atributo ser Windows-only, ele est sendo utilizado
  // em qualquer um dos casos, porque filtro default pode ser usado junto 
  // com filtro definido pelo usurio 
  IupStoreAttribute(wStruct->dialog, IUP_FILTERUSED, "1");
  findFilterAcceptedExtensions(wStruct, gStruct);
  if (strcmp(IupGetAttribute(wStruct->dialog, IUP_VISIBLE), IUP_YES) == 0)
    updateTree(IupGetAttribute(wStruct->dialog, IUP_DIRECTORY), wStruct, gStruct);
  else 
    gStruct->refreshTree = 1;       
} 

// .....................................................................
/**
 * Cria os botes de cancelar e de abrir/salvar arquivos.
 * @param wStruct estrutura que representa os widgets da interface
 * @param gStruct estrutura que apresenta as variveis de controle da aplicao
 * @return Ihandle* handle do box que contm os botes de controle do dilogo
 */
static Ihandle* createButtonsBox(Widgets *wStruct, Globals *gStruct)
{
  Ihandle* cancelButton = NULL;
  Ihandle* actionButton = NULL;
  Ihandle *buttonsHbox = NULL;

  cancelButton = IupButton("Cancelar", "cancel_cb");
  IupSetFunction("cancel_cb", (Icallback) cancel_cb);
  IupSetHandle("cancelButton", cancelButton);
  IupSetAttribute(cancelButton, IUP_SIZE, "60x");
  IupSetHandle("Cancelar", cancelButton);

  wStruct->cancelButton = cancelButton;

  IupSetAttribute(cancelButton, "FC_WIDGETS", (char *)wStruct);
  IupSetAttribute(cancelButton, "FC_GLOBALS", (char *)gStruct);

  IupSetFunction("action_cb", (Icallback)action_cb);

  // Operao default  abrir
  actionButton = IupButton("Abrir", "action_cb");
  IupSetHandle("actionButton", actionButton);
  IupSetAttribute(actionButton, IUP_SIZE, "60x");
  IupSetAttribute(actionButton, IUP_ACTIVE, IUP_NO);

  IupSetAttribute(actionButton, "FC_WIDGETS", (char *)wStruct);
  IupSetAttribute(actionButton, "FC_GLOBALS", (char *)gStruct);

  wStruct->actionButton = actionButton;

  buttonsHbox = IupHbox(
     IupFill(), actionButton, IupFill(), cancelButton, IupFill(), NULL
  );

  return buttonsHbox;
}

// .....................................................................
/**
 * Atualiza o label do boto principal, conforme a operao
 * a ser realizada (abrir ou salvar arquivo)
 * @param wStruct estrutura que representa os widgets da interface
 */
static void updateButton(Widgets *wStruct)
{
  char bLabel[MAX_SIZE];

  if ((wStruct->dialog == NULL) ||
      (strcmp(IupGetAttribute(wStruct->dialog, IUP_DIALOGTYPE), IUP_OPEN) == 0)){
      sprintf(bLabel, "%s", "Abrir");
  } else{
      sprintf(bLabel, "%s", "Salvar");
  }
  IupStoreAttribute(wStruct->actionButton, IUP_TITLE, bLabel);
}

// .....................................................................
/**
 * Mtodo para preenchimento da rvore que representa o diretrio do projeto.
 * @param dirOpened estrutura C DIR que representa o diretrio aberto
 * @param wStruct estrutura que representa os widgets da interface
 * @param gStruct estrutura que apresenta as variveis de controle da aplicao
 * @param depth profundidade relativa a um determinado diretorio inicial
 * @para maxDepth profundidade mxima (sem limite, se -1)
 */
#if 0
static int browseDirectory(
  Widgets *wStruct, Globals *gStruct, DIR* dirOpened, char* dirName, int dirId, 
  int depth, int maxDepth)
{
  int id;
  struct dirent* dp;
  int dirNameLen;
  char buffer[64];

  dirNameLen = strlen(dirName);
  id = dirId;
  dp = readdir(dirOpened);  
  //if (maxDepth != -1 && depth >= maxDepth) 
  //  return dirId;

  while (dp != NULL) {
    // como usurio deve ficar limitado a rvore de diretrios do projeto,
    // ento "." e ".." no devem aparecer. 
    if (strcmp(dp->d_name, ".") != 0 && strcmp(dp->d_name, "..") != 0) {   
      char* name;
      int nameLen;

      // armazena informaes do arquivo corrente
      static struct stat buf;
      // tmp mantm caminho completo de arquivo/diretrio
      char path[MAX_FILE_PATH];
      int i = 0;

      name = dp->d_name;
      nameLen = strlen(name);

      // constri o caminho completo do arquivo/diretrio. Se for diretrio, ele mesmo
      //  armazenado em paths na prxima iterao deste mtodo
      assert(dirNameLen + nameLen < MAX_FILE_PATH);
      strcpy(path, dirName);
      strcat(path, "/");
      strncat(path, name, MAX_FILE_PATH - dirNameLen - 1);     
      if (stat(path, &buf) != 0) {
        printf(
          "Erro em stat: %d (arquivo %s). Este arquivo no foi acrescentado na rvore de diretrios.\n", errno, path);
        return id;
      }

      // se  diretrio, ou se todos os arquivos so permitidos ou se 
      // um arquivo de extenso permitida, ento inclui na rvore de diretrio
      if((S_ISDIR(buf.st_mode) != 0) || 
          ((gStruct->extensions[0] == NULL) || 
            ((S_ISDIR(buf.st_mode) == 0) && 
            (isAcceptedExtension(dp->d_name, gStruct->extensions) == 1)))) {

        if ((maxDepth != -1 && depth >= maxDepth) || S_ISREG(buf.st_mode) != 0) { // arquivo  do tipo regular
          IupTreeStoreAttribute(wStruct->tree, "ADDLEAF", id, name);
          sprintf(buffer, "%d", dirId);
          IupTreeStoreAttribute(wStruct->tree, "PARENT", id + 1, buffer);
          sprintf(buffer, "%d", depth + 1);
          IupTreeStoreAttribute(wStruct->tree, "DEPTH", id + 1, buffer);
          id++;
          if (maxDepth != -1 && depth >= maxDepth) {
            dp = NULL;
            break;
          }
        } else if(S_ISDIR(buf.st_mode) != 0) { // arquivo  do tipo diretrio            
          DIR* aux_dir;
          // checa se diretrio pode ser aberto
          aux_dir = opendir(path);
          if (aux_dir == NULL) {
            printf("Error -- ");
            if (errno == EACCES) {
              printf("Permission denied: [%s]. Directory not browsed.\n", path);
              return id;    
            }
            verifyOpendirError(path);
            exit (1);
          }        
          assert(depth + 1 < MAX_DEPTH);
          IupTreeStoreAttribute(wStruct->tree, "ADDBRANCH", id, name);
          sprintf(buffer, "%d", dirId);
          IupTreeStoreAttribute(wStruct->tree, "PARENT", id + 1, buffer);
          sprintf(buffer, "%d", depth + 1);
          IupTreeStoreAttribute(wStruct->tree, "DEPTH", id + 1, buffer);
          id = browseDirectory(wStruct, gStruct, aux_dir, path, id + 1, depth + 1, maxDepth);
          //closedir(aux_dir);
        }        
      }
    }
    dp = readdir(dirOpened);
  }
  closedir(dirOpened);
  free(dp);
  return id;
}
#else

static int compareNames(const char* n1, const char* n2)
{
  if (n1 == n2) {
    return 0;
  } else {
    while (*n1 != '\0') {
      char c1, c2;
      c1 = tolower(*n1);
      c2 = tolower(*n2);
      if (c1 < c2)
        return -1;
      else if (c1 > c2 || *n2 == '\0')
        return 1;
      else {
        n1++; n2++;
      }
    }
    if (*n2 != '\0')
      return -1;
    else
      return 0;
  }
}

static int findInsertNodePos(Widgets *wStruct, Globals *gStruct, char* path, char* name,
  int dirId, int isBranch, int depth, int maxDepth)
{
  int isBranchPos;
  int i, pos, lastId;
  lastId = dirId + ((int) IupTreeGetUserId(wStruct->tree, dirId));
  for (i = dirId; i < lastId; i++) {
    int isBranchI;
    char* nameI;
    if (IupTreeGetInt(wStruct->tree, IUP_PARENT, i + 1) != dirId)
      continue;
    isBranchI = (strcmp(IupTreeGetAttribute(wStruct->tree, IUP_KIND, i + 1), IUP_BRANCH) == 0) ? 1 : 0;
    nameI = IupTreeGetAttribute(wStruct->tree, IUP_NAME, i + 1);
    if ((isBranch && !isBranchI) || (!(!isBranch && isBranchI) && compareNames(name, nameI) < 0)) {
      break;
    }
  }
  pos = i;
  isBranchPos = (strcmp(IupTreeGetAttribute(wStruct->tree, IUP_KIND, pos), IUP_BRANCH) == 0) ? 1 : 0;
  if (isBranchPos && pos != dirId) {
    int nchildren = (int) IupTreeGetUserId(wStruct->tree, pos);
    return pos + nchildren;    
  } else {
    return pos;
  }
}

static void insertNode(Widgets *wStruct, Globals *gStruct, char* path, char* name,
  int isBranch, int pos, int parentId, int parentDepth)
{
  int nchildren;
  int pid;
  char buffer[64];
  if (isBranch) {
    IupTreeStoreAttribute(wStruct->tree, "ADDBRANCH", pos, name);
    IupTreeSetUserId(wStruct->tree, pos + 1, (void*) 0);
  } else {
    IupTreeStoreAttribute(wStruct->tree, "ADDLEAF", pos, name);
  }
  sprintf(buffer, "%d", parentId);
  IupTreeStoreAttribute(wStruct->tree, "PARENT", pos + 1, buffer);
  sprintf(buffer, "%d", parentDepth + 1);
  IupTreeStoreAttribute(wStruct->tree, "DEPTH", pos + 1, buffer);

  pid = parentId;
  while (pid != 0) {
    nchildren = (int) IupTreeGetUserId(wStruct->tree, pid);
    IupTreeSetUserId(wStruct->tree, pid, (void*) (nchildren + 1));
    pid = IupTreeGetInt(wStruct->tree, "PARENT", pid);
  }
  nchildren = (int) IupTreeGetUserId(wStruct->tree, pid);
  IupTreeSetUserId(wStruct->tree, pid, (void*) (nchildren + 1));
}

static void removeChildren(Widgets *wStruct, Globals *gStruct, int dirId)
{
  int nchildren0, nchildren;
  int pid;
  char buffer[64];

  IupTreeSetAttribute(wStruct->tree, "DELNODE", dirId, IUP_CHILDREN);
  //IupTreeSetUserId(wStruct->tree, dirId, (void*) 0);
  nchildren0 = (int) IupTreeGetUserId(wStruct->tree, dirId);
  pid = dirId;
  while (pid != 0) {
    nchildren = (int) IupTreeGetUserId(wStruct->tree, pid);
    IupTreeSetUserId(wStruct->tree, pid, (void*) (nchildren - nchildren0));
    pid = IupTreeGetInt(wStruct->tree, "PARENT", pid);
  }
  nchildren = (int) IupTreeGetUserId(wStruct->tree, pid);
  IupTreeSetUserId(wStruct->tree, pid, (void*) (nchildren - nchildren0));
}


static int browseDirectory(
  Widgets *wStruct, Globals *gStruct, DIR* dirOpened, char* dirName, int dirId, 
  int depth, int maxDepth)
{
  int id;
  struct dirent* dp;
  int dirNameLen;
  char buffer[64];

  dirNameLen = strlen(dirName);
  id = dirId;
  dp = readdir(dirOpened);  
  //if (maxDepth != -1 && depth >= maxDepth)
  //  return dirId;

  while (dp != NULL) {
    // como usurio deve ficar limitado a rvore de diretrios do projeto,
    // ento "." e ".." no devem aparecer. 
    if (strcmp(dp->d_name, ".") != 0 && strcmp(dp->d_name, "..") != 0) {   
      char* name;
      int nameLen;

      // armazena informaes do arquivo corrente
      static struct stat buf;
      // tmp mantm caminho completo de arquivo/diretrio
      char path[MAX_FILE_PATH];
      int i = 0;

      name = dp->d_name;
      nameLen = strlen(name);

      // constri o caminho completo do arquivo/diretrio. Se for diretrio, ele mesmo
      //  armazenado em paths na prxima iterao deste mtodo
      assert(dirNameLen + nameLen < MAX_FILE_PATH);
      strcpy(path, dirName);
      strcat(path, "/");
      strncat(path, name, MAX_FILE_PATH - dirNameLen - 1);     
      if (stat(path, &buf) != 0) {
        printf(
          "Erro em stat: %d (arquivo %s). Este arquivo no foi acrescentado na rvore de diretrios.\n", errno, path);
        return id;
      }

      // se  diretrio, ou se todos os arquivos so permitidos ou se 
      // um arquivo de extenso permitida, ento inclui na rvore de diretrio
      if((S_ISDIR(buf.st_mode) != 0) || 
          ((gStruct->extensions[0] == NULL) || 
            ((S_ISDIR(buf.st_mode) == 0) && 
            (isAcceptedExtension(dp->d_name, gStruct->extensions) == 1)))) {

        if (maxDepth != -1 && depth >= maxDepth) {
          insertNode(wStruct, gStruct, path, name, 0, dirId, dirId, depth);
          //id++;
          dp = NULL;
          break;
        } else if (S_ISREG(buf.st_mode) != 0) { // arquivo  do tipo regular
          int pos = findInsertNodePos(wStruct, gStruct, path, name, dirId, 0, depth, maxDepth);
          //int pos = id;
          insertNode(wStruct, gStruct, path, name, 0, pos, dirId, depth);
          //id++;
        } else if(S_ISDIR(buf.st_mode) != 0) { // arquivo  do tipo diretrio            
          DIR* aux_dir;
          int pos, pos2;
          // checa se diretrio pode ser aberto
          aux_dir = opendir(path);
          if (aux_dir == NULL) {
            printf("Error -- ");
            if (errno == EACCES) {
              printf("Permission denied: [%s]. Directory not browsed.\n", path);
              return id;    
            }
            verifyOpendirError(path);
            exit (1);
          }        
          assert(depth + 1 < MAX_DEPTH);
          pos = findInsertNodePos(wStruct, gStruct, path, name, dirId, 1, depth, maxDepth);
          //pos = id;
          insertNode(wStruct, gStruct, path, name, 1, pos, dirId, depth);
          pos2 = browseDirectory(wStruct, gStruct, aux_dir, path, pos + 1, depth + 1, maxDepth);
          //id += pos2 - pos;
          //closedir(aux_dir);
        }        
      }
    }
    dp = readdir(dirOpened);
  }
  closedir(dirOpened);
  free(dp);
  return id;
}
#endif

// .....................................................................
/**
 * Encontra todas as extenses aceitas pelo filtro correntemente
 * selecionado pelo usurio.
 * @param wStruct estrutura que representa os widgets da interface
 * @param gStruct estrutura que apresenta as variveis de controle da aplicao
 */
static void findFilterAcceptedExtensions(Widgets *wStruct, Globals *gStruct) {

  char* selected_filter=NULL;
  // mantm referncia para o filtro selecionado
  char* iFilter=NULL;
  // cpia do filtro selecionado
  char* filter=NULL;
  char* aux=NULL;
  char* tmp=NULL;
  char* aux_ext=NULL;
  char* ext=NULL;

  int filter_num;
  int i=0;

  gStruct->extensions[0] = NULL;
 
  if (wStruct-> dialog != NULL){ 
    selected_filter = IupGetAttribute(wStruct->dialog, IUP_FILTERUSED);
    filter_num = atoi(selected_filter);
    if (filter_num < 0)  return;
    iFilter = gStruct->filters_extensions[filter_num-1];
    if (iFilter == NULL)  return;
    filter = strdup(iFilter);
    
    if (strcmp(filter, DEFAULT_FILTER) != 0){
       
       aux = strchr(filter, '(');
       // como parnteses adicionais no incio e no final
       // do filtro so tratados em updateFilterBox, se aqui
       // filtro conter parentese, ento ser um no incio
       // e outro no final
       if (aux != NULL){
            aux[strlen(aux)-1] = '\0';
	    tmp=strdup(aux+1);
       } else{
	    tmp=strdup(filter);
       }
       ext = strchr(tmp, ';');
       // se existem mais do que uma extenso ou se usurio colocou vrios
       // ';' no final da extenso
       while (ext != NULL){
	  aux= strdup(ext+1);
          if (tmp == ext){
             tmp=strdup(ext+1);
          }
          ext[0] = '\0';
          aux_ext = strchr(tmp, '.');
	  gStruct->extensions[i]= strdup(aux_ext);
	  i++;
	  tmp=strdup(aux);
          while(tmp[0] == ';'){
             tmp=strdup(tmp+1);
          }
          if (tmp != NULL){
             ext = strchr(tmp, ';');
          }else{
             ext=NULL;
          }
       }
       aux_ext = strchr(tmp, '.');
       if (aux_ext != NULL){
          gStruct->extensions[i] = strdup(aux_ext);
          i++;
       }
       // para finalizar a lista de extenses
       gStruct->extensions[i] = NULL;
    } else{
       // se todos os arquivos so permitidos, ento no 
       // preciso setar vetor de extenses permitidas pelo filtro
       gStruct->extensions[0] = NULL;
    }
    selected_filter = NULL;
    iFilter = NULL;
    filter = NULL;
    aux = NULL;
    aux_ext = NULL;
    tmp = NULL;
  }
  
}

// .....................................................................
/**
 * Verifica se a extenso do arquivo  permitida, de acordo com 
 * o filtro escolhido, e de acordo com as extenses no permitidas
 * pelo prprio programa.
 * @param name nome do arquivo que deve ser verificado com relao 
 *        a extenso 
 * @param exts vetor de extenses permitidas (.extenso)
 * @return 1, se arquivo possui extenso aceita pelo filtro. 0, caso contrrio
 */
static int isAcceptedExtension(char* filename, char* exts[NUM_EXTENSIONS]) {
  int i = 0;
  char* aux=NULL;

#ifdef _UNIX_
  aux = rindex(filename, '.');
#else
  aux = strrchr(filename, '.');
#endif
  // se nome de arquivo contm extenso
  //if (aux != NULL)   tmp = strdup(aux);
  //else return 0; // se no tem extenso, ento no aceita
  if (aux == NULL)
    aux = "";

  // primeiro compara com as extenses que no so permitidas pelo 
  // prprio programa
  while(notAcceptedExt[i] != NULL && strcmp(aux, notAcceptedExt[i]) != 0){
     i++;
  } 

  if (i < NUM_EXTENSIONS && notAcceptedExt[i] != NULL)
     return 0;
  
  i = 0;
  // compara com as extenses permitidas no filtro
  while (i < NUM_EXTENSIONS && exts[i] != NULL && 
      strcmp(exts[i], "*") != 0 && strcmp(exts[i], ".*") != 0 && 
      strcmp(aux, exts[i]) != 0)
    i++;
  
  aux=NULL;
  
  if (i == NUM_EXTENSIONS || (exts[i] == NULL && i < NUM_EXTENSIONS))
     return 0; // no aceito
  
  return 1; // aceito
}

// .....................................................................

/**
 * Verifica a ocorrncia de erros em opendir e imprime o erro.
 * @param directory diretrio que est sendo pesquisado no rvore de diretrios
 */
static void verifyOpendirError(char* directory){

    if (errno == EMFILE)  printf("Too many file descriptors in use by process: [%s].\n", directory);
    else if (errno == ENFILE)  printf("Too many files are currently open in the system: [%s].\n", directory);
    else if (errno == ENOENT)  printf("Directory does not exist, or name is an empty string: [%s].\n", directory);
    else if (errno == ENOMEM)  printf("Insufficient memory to complete the operation: [%s]\n", directory);
    else if (errno == ENOTDIR) printf("Name is not a directory: [%s]\n", directory);
}

// ======================================================================
// Definio de callbacks 
// ======================================================================

// .....................................................................
/**
 * Callback que fecha a janela principal, cancelando a seleo que 
 * possivelmente o usurio poderia ter feito.
 * @param self o prprio elemento IUP (boto de cancelar)
 * @return IUP_CLOSE
 */
static int cancel_cb(Ihandle * self) {
   Globals *gStruct = NULL;
   Widgets* wStruct = NULL;

   gStruct = (Globals*)IupGetAttribute(self, "FC_GLOBALS");
   wStruct = (Widgets*)IupGetAttribute(self, "FC_WIDGETS");

   // selectedId setado para -1 indica que usurio cancelou operao e que,
   // mesmo que haja arquivo selecionado, ele no deve ser retornado
   // valor  colocado em branco
   IupStoreAttribute(wStruct->dialog, "SELECTED_NODE", "-1");
   IupStoreAttribute(wStruct->dialog, IUP_VALUE, NULL);
   
   // status = -1: operation cancelled
   IupSetAttribute(wStruct->dialog, IUP_STATUS, "-1");

   return IUP_CLOSE;
}

// .....................................................................
/**
 * Callback utilizada para que quando o usurio digite algo na caixa
 * de arquivo selecionado, o boto de salvar seja ativado, caso a operao
 * seja realmente a de salvar arquivo.
 * @param self o prprio elemento IUP (caixa de texto)
 * @return IUP_DEFAULT
 */
static int selection_cb (Ihandle* self, int c, char* after) {
  Globals* gStruct = NULL;
  Widgets* wStruct = NULL;

  gStruct = (Globals*)IupGetAttribute(self, "FC_GLOBALS");
  wStruct = (Widgets*)IupGetAttribute(self, "FC_WIDGETS");

  if (after && strlen(after) > 0 &&
      strcmp(IupGetAttribute(wStruct->dialog, IUP_DIALOGTYPE), IUP_SAVE) == 0) {
    IupSetAttribute(wStruct->actionButton, IUP_ACTIVE, IUP_YES);
  } else {
    IupSetAttribute(wStruct->actionButton, IUP_ACTIVE, IUP_NO);
  }
  return IUP_DEFAULT;
}

// .....................................................................
/**
 * Callback que determina a ao do boto com relao  ao escolhida:
 * abrir ou salvar arquivo.
 * @param self o prprio elemento IUP (boto de abrir/salvar arquivo)
 * @return IUP_CLOSE, se seleo de arquivo est correta.
 *         IUP_DEFAULT, caso contrrio
 */
static int action_cb(Ihandle * self) {
   Globals* gStruct = NULL;
   Widgets* wStruct = NULL;

   char* file_tmp = NULL;
   char* file;
   // armazena o caminho absoluto do n selecionado
   char tmp[MAX_FILE_PATH];
   char attr_id[MAX_SIZE];
   char* aux = NULL;
   
   gStruct = (Globals*)IupGetAttribute(self, "FC_GLOBALS");
   wStruct = (Widgets*)IupGetAttribute(self, "FC_WIDGETS");

   file_tmp = IupGetAttribute(wStruct->selectionField, IUP_VALUE);
   if (file_tmp == NULL || strcmp(file_tmp, "")==0){
      IupMessage("Ateno", "Primeiro voc precisa escolher um arquivo!");
      return IUP_DEFAULT;
   }

   file = strdup(file_tmp);
   if (IupGetAttribute(wStruct->dialog, "SELECTED_NODE") != NULL){
      char path[MAX_FILE_PATH];

      strcpy(attr_id, IupGetAttribute(wStruct->dialog, "SELECTED_NODE"));
      getNodePath(wStruct, gStruct, atoi(attr_id), path);
      strcpy(tmp, path);

      // usurio entrou texto como nome de arquivo
      if (strcmp(IupGetAttribute(wStruct->selectionField, IUP_READONLY), IUP_YES)){
         strncat(tmp, "/", MAX_FILE_PATH);
         strncat(tmp, strdup(file), MAX_FILE_PATH);
         // arquivo novo
         IupSetAttribute(wStruct->dialog, IUP_STATUS, "1");
         aux = strrchr(tmp, '.');
         if (aux == NULL){
            // se usurio informou extenso de arquivo, independentemente do filtro o
            // arquivo ser salvo com esta extenso. Se usurio no informou extenso,
            // ento  acrescentada ao nome do arquivo a primeira extenso do filtro
            // (ou a nica, caso o filtro seja de apenas uma extenso)
            // Ou seja, se o filtro selecionado indica mais do que uma extenso,
            // ento o default est sendo usar a primeira delas. Se h
            // apenas uma extenso, ento ela  adicionada ao nome do
            // arquivo informado pelo usurio, CASO a operao seja de SALVAR
            if(gStruct->extensions[0] != NULL){
                 strncat(tmp, gStruct->extensions[0], MAX_FILE_PATH);
            }
          }
      } else {
         IupSetAttribute(wStruct->dialog, IUP_STATUS, "0");
      }
   }
   else{
      strcpy(tmp, "");
   }
   IupStoreAttribute(wStruct->dialog, IUP_VALUE, tmp);
   FREE(file);

   // checa se arquivo j existe
   if (strcmp(IupGetAttribute(wStruct->dialog, IUP_DIALOGTYPE), IUP_SAVE) == 0){
       FILE* fp = fopen(tmp, "rb");
       if (fp) {
          int r;
          fclose(fp);
          r = IupAlarm("Aviso", "Arquivo j existe. Deseja sobrescrev-lo?", 
            "Sim", "No", NULL);
          if (r == 1) 
            return IUP_CLOSE;
          else
            return IUP_IGNORE;
       }
   } 
   return IUP_CLOSE;
}

// .....................................................................
/**
 * Callback chamada quando o usurio efetua a seleo de qualquer no 
 * da rvore.
 * @param self o prprio elemento IUP (rvore que representa a 
 *        hierarquia de diretrios)
 * @param id id do n
 * @param status status do n (selecionado ou no)
 * @return IUP_DEFAULT
 */
static int selectionNode_cb(Ihandle *self, int id, int status) 
{
  Globals *gStruct = NULL;
  Widgets *wStruct = NULL;
  char attr_id[MAX_SIZE];

  gStruct = (Globals*)IupGetAttribute(self, "FC_GLOBALS");
  wStruct = (Widgets*)IupGetAttribute(self, "FC_WIDGETS");

  // se n est selecionado e  um n-folha (ou seja, se no  um
  // diretrio), ento seu nome deve aparecer na caixa que apresenta
  // o nome do arquivo selecionado pelo usurio
  if (status){
    char currentDir[MAX_FILE_PATH];
    char* currentDirName;
    char* initDir;
    initDir = IupGetAttribute(wStruct->dialog, IUP_DIRECTORY);
    if (strcmp(IupGetAttribute(self,IUP_KIND),IUP_LEAF) == 0){
      IupStoreAttribute(wStruct->selectionField, IUP_VALUE, IupGetAttribute(self, IUP_NAME));
      IupSetAttribute(wStruct->actionButton, IUP_ACTIVE, IUP_YES);
      IupSetAttribute(wStruct->selectionField, IUP_READONLY, IUP_YES);
      getNodePath(wStruct, gStruct, IupTreeGetInt(wStruct->tree, "PARENT", id), currentDir);
    } else {
      IupSetAttribute(wStruct->selectionField, IUP_VALUE, NULL);
      if (strcmp(IupGetAttribute(wStruct->dialog, IUP_DIALOGTYPE), IUP_SAVE) == 0){
        char* name;
        IupSetAttribute(wStruct->selectionField, IUP_READONLY, IUP_NO);
        name = IupGetAttribute(wStruct->selectionField, IUP_VALUE);
        if (name && strlen(name) > 0)
          IupSetAttribute(wStruct->actionButton, IUP_ACTIVE, IUP_YES);
        else
          IupSetAttribute(wStruct->actionButton, IUP_ACTIVE, IUP_NO);
      } else{
        IupSetAttribute(wStruct->actionButton, IUP_ACTIVE, IUP_NO);
      }     
      getNodePath(wStruct, gStruct, id, currentDir);
    }
    currentDirName = currentDir + (int)(stripFileName(initDir) - initDir);
    IupStoreAttribute(wStruct->directoryField, IUP_VALUE, currentDirName);
  }

  sprintf(attr_id, "%d", id);
  IupStoreAttribute(wStruct->dialog, "SELECTED_NODE", attr_id);

  return IUP_DEFAULT;
}

// .....................................................................
/**
 * Callback chamada quando o usurio abre um n no-folha da rvore.
 * @param self o prprio elemento IUP (rvore que representa a hierarquia 
 *        de diretrios)
 * @param id id do n
 * @return IUP_DEFAULT
 */
static int branchopen_cb(Ihandle *self, int id) 
{
  Globals *gStruct = NULL;
  Widgets *wStruct = NULL;
  gStruct = (Globals*)IupGetAttribute(self, "FC_GLOBALS");
  wStruct = (Widgets*)IupGetAttribute(self, "FC_WIDGETS");
  buildSubTree(wStruct, gStruct, id);
  return IUP_DEFAULT;
}

// .....................................................................
/**
 * Callback chamada quando o usurio fecha um n no-folha da rvore.
 * @param self o prprio elemento IUP (rvore que representa a hierarquia 
 *        de diretrios)
 * @param id id do n
 * @return IUP_DEFAULT
 */
static int branchclose_cb(Ihandle *self, int id) 
{
//TODO: remover ns.
   return IUP_DEFAULT;
}

// .....................................................................
/**
 * Callback do filtro. 
 * @param self: a prpria lista das entradas do filtro
 * @param t
 * @param i
 * @param v
 */
static int filter_cb (Ihandle *self, char *t, int i, int v) {
  
  Globals *gStruct = NULL;
  Widgets *wStruct = NULL;
  char* num_filter = NULL;

  gStruct = (Globals*)IupGetAttribute(self, "FC_GLOBALS");
  wStruct = (Widgets*)IupGetAttribute(self, "FC_WIDGETS");
  
  if (v == 1) {
    num_filter = (char *) malloc (sizeof(int));
    sprintf(num_filter, "%d", i);
    IupStoreAttribute(wStruct->dialog, IUP_FILTERUSED, num_filter);
    // para que a rvore seja reconstruda de acordo com o filtro
    // escolhido pelo usurio
    findFilterAcceptedExtensions(wStruct, gStruct);
    updateTree(strdup(IupGetAttribute(wStruct->dialog, IUP_DIRECTORY)), wStruct, gStruct);
    // limpa o campo de arquivo selecionado
    IupSetAttribute(wStruct->selectionField, IUP_VALUE, NULL);
    // desativa boto de abrir/salvar quando filtro  alterado
    IupSetAttribute(wStruct->actionButton, IUP_ACTIVE, IUP_NO);
    FREE(num_filter);
  }

  return IUP_DEFAULT;
}

// ======================================================================
// ======================================================================
