/**
 * @file util.h
 * Arquivo de definio da interface exportada de funes utilitrias
 * para transferncias de arquivos.
 * $Id: util.h 59819 2007-01-10 19:28:26Z clinio $
 * @author Aldo de Araujo Nogueira
 * @author Tatiana Almeida Souza Coelho
 * @author Andr Luiz Clinio
 */

#ifndef __UTIL_H__
#define __UTIL_H__

#include <stdio.h>
#include "wioapi.h"

namespace wioapi {

  /**
   * Envia um arquivo do sistema de arquivo local para a rea de projetos.
   * throw() indica que este mtodo no levanta exceo.
   * @param localFile o caminho do arquivo local
   * @param remoteFile o Caminho do arquivo na rea de projetos
   * @param server o nome do servidor
   * @param port a porta do servio de WIO
   * @param user o identificador do usurio
   * @param pwd a senha do usurio
   * @param owner o identificador do usurio dono do arquivo
   * @param projectId o identificador do projeto
   */
  bool WIO_API put(const char *localFile, const char *remoteFile, const char *server,
       const char *port, const char *user, const char *pwd, 
       const char* owner, const char *projectId) throw();

  /**
   * Envia um arquivo do sistema de arquivo local para a rea de projetos.
   * throw() indica que este mtodo no levanta exceo.
   * @param localFile o caminho do arquivo local
   * @param remoteFile o Caminho do arquivo na rea de projetos
   * @param server o nome do servidor
   * @param port a porta do servio de WIO
   * @param user o identificador do usurio
   * @param pwd a senha do usurio
   * @param owner o identificador do usurio dono do arquivo
   * @param projectId o identificador do projeto
   * @param desc apenda a descrio ao arquivo
   */
  bool WIO_API put(const char *localFile, const char *remoteFile, const char *server,
       const char *port, const char *user, const char *pwd, 
       const char* owner, const char *projectId, const char *desc) throw();

  /**
   * Traz um arquivo da rea de projetos para o sistema de arquivos local.
   * throw() indica que esse mtodo no levanta exceo.
   * @param localFile o caminho do arquivo local.
   * @param remoteFile o caminho do arquivo na rea de projeto.
   * @param server o nome do servidor
   * @param port a porta do servio de WIO
   * @param user o identificador do usurio
   * @param pwd a senha do usurio
   * @param projectId o identificador do projeto
   */
  bool WIO_API get(const char* localFile, const char* remoteFile, const char* server,
       const char* port, const char *user, const char *pwd, 
       const char* owner, const char* projectId) throw();

  /**
   * Envia um arquivo do sistema de arquivo local para a rea de projetos.
   * @param localFile o caminho do arquivo local.
   * @param remoteFile o caminho do arquivo na rea de projeto.
   * @param proj o projeto remoto.
   * @param packSize tamanho mximo do pacote de transmisso.
   */
  bool WIO_API put(const char* localFile, const char* remoteFile,
       wioapi::WIOProject* proj, const char* owner, long packSize) throw();

  /**
   * Envia um arquivo do sistema de arquivo local para a rea de projetos.
   * @param localFile o caminho do arquivo local.
   * @param remoteFile o caminho do arquivo na rea de projeto.
   * @param proj o projeto remoto.
   * @param packSize tamanho mximo do pacote de transmisso.
   * @param desc apenda a descrio ao arquivo
   */
  bool WIO_API put(const char* localFile, const char* remoteFile,
       wioapi::WIOProject* proj, const char* owner, long packSize,
       const char *desc) throw();

  /**
   * Traz um arquivo da rea de projetos para o sistema de arquivos local.
   * @param localFile o caminho do arquivo local.
   * @param remoteFile o caminho do arquivo na rea de projeto.
   * @param proj o projeto remoto.
   * @param packSize tamanho mximo do pacote de transmisso.
   */
  bool WIO_API get(const char* localFile, const char* remoteFile,
       wioapi::WIOProject* proj, const char* owner, long packSize) throw();

  /**
   *
   */
  void WIO_API appendDescription(const char* remoteFile, wioapi::WIOProject* proj, const char *desc) throw();

  /**
   *
   */
 bool WIO_API fileExists(char *remoteFile, char *server, char *port, char *user, char *pwd,
                 char* proj, char *owner) throw();

  /**
   * Define o modo que a classe ir trabalhar: enviando do sistema de arquivo
   * local para a rea de projetos (PUT) ou da rea de projetos para o sistema
   * de arquivos local (GET).
   */
  enum WIO_API Mode { 
     /** download */ GET = 0, 
     /** upload   */ PUT      
  };

  /**
   * Transfere arquivos entre o sistema de arquivos local e a rea de projetos
   * prevendo a utilizao em uma interface grfica sem threads, usando a
   * callback IUP_IDLE_ACTION por exemplo.
   */
  class WIO_API IdleTransfer {

    /** Modo de transmisso (GET ou PUT)*/
    Mode mode;

    /** Projeto da transferncia */
    wioapi::WIOProject* project;

    /** Arquivo remoto */
    wioapi::WIOFile* remoteFile;

    /** Arquivo local */
    FILE* localFile;

    /** Tamanho do pacote de dados que  enviado de cada vez. */
    int packSize;

    /** Tamanho total do arquivo a ser transferido. */
    long long size;

    /** Quantidade de bytes transferidos no momento. */
    long long transferred;

    /** Espao de dados temporrio utilizado durante a transferncia. */
    char *buff;

    /** Dono dos arquivos de transferncia */
    const char* ownerId;

    /** Dono dos arquivos de transferncia */
    const char* userId;

    /**
     * Abre os arquivos local e remoto.
     * @param localFile O caminho do arquivo local.
     * @param remoteFile O caminho do arquivo na rea de projeto.
     */
    void openFiles(const char *localPath, const char *remotePath) 
    throw(wioapi::Exception *);

    /**
     * Fecha os arquivos utilizados na transmisso.
     * Esse mtodo no pode lanar nenhuma exceo por causa de "throw ()".
     */
    void closeFiles() throw(wioapi::Exception *);

    /**
     * Inicializa a transferncia do arquivo.
     */
    void initTransfer() throw(wioapi::Exception *);

  public:

    /**
     * @param mode o modo de transferncia.
     * @param localPath o caminho do arquivo local.
     * @param remotePath o caminho do arquivo na rea de projeto.
     * @param project o projeto.
     * @param ownerId identificador do usurio owner da transferncia.
     * @param packSize o tamanho do pacote enviado/buscado por vez (opcional).
     * @see IdleTransfer::Mode
     */
    IdleTransfer(Mode mode, const char *localPath, const char *remotePath,
      wioapi::WIOProject* project, const char* ownerId, long packSize = -1)
      throw(wioapi::Exception *);

    /**
     * Destrutor da classe
     */
     ~IdleTransfer();

    /**
     * Consulta ao projeto conectado.
     * @return uma referncia ao projeto.
     */
    wioapi::WIOProject* getProject();

    /**
     * Este mtodo deve ser chamado pela funo idle aps a construo 
     * do objeto para efetivar a transferncia do arquivo 
     * (um pacote de cada vez).
     */
    void transfer(void) throw(wioapi::Exception *);

    /**
     * Obtm o total transferido do arquivo para checar se a 
     * transferncia foi finalizada.
     * @return o tamanho (em bytes) j transferido do arquivo.
     * @see getSize()
     */
    long long getTransferred(void);

    /**
     * Obtm o total da transferncia do arquivo. 
     * @return o tamanho (em bytes) da transferncia do arquivo.
     * @see getTransferred()
     */
    long long getSize(void);

    /**
     * Acrscimo de descrio.
     */
    void appendDescription(const char *description);
  };

} //namespace wioapi

#endif // __UTIL_H__
