#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <string>
#include "TransferDialog.h"
#include "wiofiledlg.h"


using namespace wioapi;
using namespace std;


/** Varivel transferDialog para que seja alocado um espao de memria */
TransferDialog* TransferDialog::transferDialog;


/* ....................................................................... */
/**
 *
 */
TransferDialog::TransferDialog(WIOFileSystem *filesystem, WIOProject *project,
  const char* ownerId, const char* defaultPath, bool getOnly) 
  throw (wioapi::Exception*) {

  this->filesystem = filesystem;
  this->createdFileSystem = (bool)false;

  this->project = project;
  this->createdProject = (bool)false;
  
  init(defaultPath, getOnly, ownerId);
}

/* ....................................................................... */
/**
 *
 */
TransferDialog::TransferDialog(const char* server, const char* port,
const char* user, const char* pwd, const char* projectId,
const char* ownerId, const char* defaultPath, 
bool getOnly) throw (wioapi::Exception*) {
  const char* ior = WIOLocator::createIOR(server, port);
  this->filesystem = WIOLocator::buildFileSystem(ior);
  this->createdFileSystem = (bool)true;

  this->project = filesystem->openProject(user, pwd, projectId);
  this->createdProject = (bool)true;
  
  init(defaultPath, getOnly, ownerId);
}

/* ....................................................................... */
/**
 *
 */
void TransferDialog::init(const char* defaultPath, bool getOnly, 
const char* ownerId) throw (wioapi::Exception*) {
  this->transferDialog = this;
  this->localFileName = NULL;
  this->remoteFileName = NULL;
  this->idleTransfer = NULL;
  this->separatorChar = NULL;
  this->defaultPath = defaultPath;
  this->getOnly = getOnly;
  if (ownerId == NULL) {
     throw new wioapi::Exception("TransferDialog::init",
         "OwnerId nulo para transferncias", "");
  }

  this->ownerId = strdup(ownerId);
  this->separatorChar = filesystem->getSeparatorChar();
}

/* ....................................................................... */
/**
 *
 */
TransferDialog::~TransferDialog() {
  if (remoteFileName != NULL) free(remoteFileName);
  remoteFileName = NULL;
  
  if (localFileName != NULL) free(localFileName);
  localFileName = NULL;

  if (separatorChar != NULL) delete separatorChar;
  separatorChar = NULL;
 
  if (createdProject) delete project;
  project = NULL;

  if (createdFileSystem) delete filesystem;
  filesystem = NULL;

  if (ownerId != NULL) free(ownerId);
  ownerId = NULL;
}

/* ....................................................................... */

TransferDialog* TransferDialog::getDialog() {
  return transferDialog;
}

/* ....................................................................... */
/**
 * Cria dilogo que apresenta a barra de progresso da transferncia 
 * @param dialog Dialogo pai, que fica bloqueado enquanto este dilogo 
 * no for fechado.
 */
void TransferDialog::createProgressDialog(Ihandle* dialog) throw () {
  IupSetHandle("dialog", dialog);

  gauge = IupSetAttributes(IupGauge(), "SIZE=80x16, EXPAND=HORIZONTAL");
  IupSetFunction("cancelTransfer_cb", (Icallback) cancelTransfer_cb);
  cancelTransferButton = IupButton("Cancel", "cancelTransfer_cb");
  IupSetAttribute(cancelTransferButton, "TRANSFER_DIALOG", (char *)this);

  progressPopup = IupSetAttributes(IupDialog(
    IupSetAttributes(IupVbox(
      gauge,
      IupHbox(IupFill(), cancelTransferButton, IupFill(), NULL),
      NULL
    ), 
    "GAP=5, MARGIN=5x5")), 
    "RESIZE=NO, TITLE = Transfering...");

  IupSetAttribute(progressPopup, IUP_PARENTDIALOG, "dialog");
  IupSetAttribute(progressPopup, IUP_RESIZE, IUP_NO);
}

/* ....................................................................... */
/**
 * Cria dilogo de transferncia de arquivos
 * @return dialog
 */
Ihandle* TransferDialog::createDialog() throw () {
  localPath = IupSetAttributes(
       IupText("do_nothing"), 
       "READONLY=YES, EXPAND=HORIZONTAL");
  remotePath = IupSetAttributes(
       IupText("do_nothing"), 
       "READONLY=YES, EXPAND=HORIZONTAL");

  IupSetFunction("transfer_cb", (Icallback) transfer_cb);
  transferButton = IupButton("Transfer", "transfer_cb");
  IupSetHandle("transfer_button", transferButton);
  IupSetAttribute(transferButton, IUP_SIZE, "40");
  IupSetAttribute(transferButton, IUP_ACTIVE, "NO");

  IupSetFunction("close_cb", (Icallback) close_cb);
  closeButton = IupButton("Close", "close_cb");
  IupSetHandle("close_button", closeButton);
  IupSetAttribute(closeButton, IUP_SIZE, "40");

  useDirToggle = IupToggle("Use remote dir", "");

  IupSetFunction("useDefault_cb", (Icallback) useDefault_cb);
  useDefaultToggle = IupToggle("Use default local path", "useDefault_cb");

  IupSetFunction("localBrowse_cb", (Icallback) localBrowse_cb);
  localBrowseButton = IupButton("Browse...", "localBrowse_cb");
  IupSetAttribute(localBrowseButton, IUP_SIZE, "40");

  IupSetFunction("remoteBrowse_cb", (Icallback) remoteBrowse_cb);
  remoteBrowseButton = IupButton("Browse...", "remoteBrowse_cb");
  IupSetAttribute(remoteBrowseButton, IUP_SIZE, "40");


  Ihandle *getToggle = IupToggle("Get", "do_nothing");
  IupSetHandle("get", getToggle);
  Ihandle *sendToggle = IupToggle("Send", "do_nothing");
  IupSetHandle("send", sendToggle);

  Ihandle* frame;
  if (!getOnly) {
     operationRadio = IupRadio(
        IupVbox(
          getToggle,
          sendToggle,
          NULL
        )
     );
    frame = IupSetAttributes(
         IupFrame(operationRadio), 
        "TITLE=Operation, EXPAND=HORIZONTAL");
    IupSetFunction("operationRadio_cb", (Icallback) operationRadio_cb);
    IupSetAttribute(getToggle, IUP_ACTION, "operationRadio_cb");
    IupSetAttribute(sendToggle, IUP_ACTION, "operationRadio_cb");
    IupSetAttribute(getToggle, "TRANSFER_DIALOG", (char *)this);
    IupSetAttribute(sendToggle, "TRANSFER_DIALOG", (char *)this);
  }
  else {
    frame = IupLabel("");
  }

  /* Dilogo */
  Ihandle* dialog = IupSetAttributes(IupDialog(
    IupSetAttributes(IupVbox(
      frame,
      IupVbox(
        IupVbox(
          useDirToggle,
          IupLabel("Remote path"),
          IupHbox(
            remotePath,
            remoteBrowseButton,
            NULL
          ),
          NULL
        ),
        IupVbox(
          useDefaultToggle,
          IupLabel("Local path"),
          IupHbox(
            localPath,
            localBrowseButton,
            NULL
          ),
          NULL
        ),
        NULL
      ),
      IupHbox(
        IupFill(),
        transferButton,
        closeButton,
        IupFill(),
        NULL
      ),
      NULL
    ), "GAP=5, MARGIN=5x0")
  ), "TITLE = Test, SIZE=200x100");
  /* *INDENT-ON* */

  IupSetAttribute(dialog, IUP_RESIZE, IUP_NO);
  IupSetAttribute(dialog, IUP_MAXBOX, IUP_NO);
  IupSetAttribute(dialog, IUP_MINBOX, IUP_NO);

  IupSetAttribute(dialog, IUP_DEFAULTENTER, "transfer_button");
  IupSetAttribute(dialog, IUP_DEFAULTESC, "close_button");

  createProgressDialog(dialog);

  // para evitar o uso de variveis globais,  inserida a prpria 
  // classe como um atributo do widget
  IupSetAttribute(localBrowseButton, "TRANSFER_DIALOG", (char *)this);
  IupSetAttribute(remoteBrowseButton, "TRANSFER_DIALOG", (char *)this);
  IupSetAttribute(transferButton, "TRANSFER_DIALOG", (char *)this);
  IupSetAttribute(closeButton, "TRANSFER_DIALOG", (char *)this);
  IupSetAttribute(useDefaultToggle, "TRANSFER_DIALOG", (char *)this);
  IupSetAttribute(useDirToggle, "TRANSFER_DIALOG", (char *)this);
  
  if (!getOnly) {
     IupSetAttribute(dialog, IUP_SIZE, "200x140");
  }

  IupSetAttribute(useDirToggle, IUP_VALUE, IUP_OFF);
  IupSetAttribute(useDirToggle, IUP_ACTIVE, IUP_NO);

  return dialog;
}

/* ....................................................................... */
/**
 * Habilita o boto de transferncia quando tanto o caminho do arquivo
 * local quanto o caminho do arquivo remoto tiverem sido informados
 */
void TransferDialog::activateTransferButton() throw () {
  bool hasRemote = (remoteFileName != NULL && strcmp(remoteFileName, "") != 0);
  bool hasLocal = (localFileName != NULL && strcmp(localFileName, "") != 0);
  bool useDefaultToggleIsOn = strcmp(
        IupGetAttribute(useDefaultToggle, IUP_VALUE), IUP_ON) == 0;
  
  if (hasRemote && (hasLocal || useDefaultToggleIsOn)) {
    IupSetAttribute(transferButton, IUP_ACTIVE, "YES");
  }
  else {
    IupSetAttribute(transferButton, IUP_ACTIVE, "NO");
  }
}

/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */

/**
 * Inicia transferncia de arquivo e abre a janela de progresso
 * @param self
 */
int TransferDialog::transfer_cb(Ihandle * self) {
  try {
    TransferDialog* tdlg = (TransferDialog*) IupGetAttribute(
        self, "TRANSFER_DIALOG");

    enum Mode op = GET;
    if (!tdlg->getOnly) {
       char *operation = IupGetAttribute(tdlg->operationRadio, "VALUE");
       op = (strcmp(operation, "get") == 0) ? GET : PUT;
    }

    bool useDefaultToggleIsOn = strcmp(IupGetAttribute(
         tdlg->useDefaultToggle, IUP_VALUE), IUP_ON) == 0;
    string tempLocalPath;

    if (useDefaultToggleIsOn) {
       tempLocalPath = string(tdlg->defaultPath) + 
                 tdlg->separatorChar + tdlg->remoteFileName;
    }
    else {
      tempLocalPath = tdlg->localFileName;
    }
    
    tdlg->idleTransfer = new IdleTransfer(op, tempLocalPath.c_str(),
       tdlg->remoteFileName+1, tdlg->project, tdlg->ownerId, 4*1024);

    IupSetFunction(IUP_IDLE_ACTION, (Icallback) idle_function);
    IupSetAttribute(tdlg->gauge, IUP_VALUE, "0");
    IupSetAttribute(tdlg->cancelTransferButton, IUP_TITLE, "Cancel");
    IupSetAttribute(tdlg->progressPopup, IUP_TITLE, "Transfering...");
    IupShow(tdlg->progressPopup);
  }
  catch (wioapi::Exception *ex) {
     printf("No foi possvel conexo com o servidor.\n");
     printf("Cliente: %s\n", ex->getClientMessage());
     printf("Servidor: %s\n", ex->getServerMessage());
     delete ex;
  }

  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Efetua a transferncia do arquivo em blocos e atualiza 
 * a barra de progresso
 */
int TransferDialog::idle_function() {
  // Caso o usuario dessa classe use threads, ele nao vai querer usar a funcao
  // idle do IUP. Portanto podemos usar variavel global aqui.
  TransferDialog* tdlg = (TransferDialog*)TransferDialog::getDialog();
   
  long long transferred = tdlg->idleTransfer->getTransferred();
  long long size = tdlg->idleTransfer->getSize();

  try {
    if (transferred < size) {
      static char value[32];
      sprintf(value, "%.2f", ((float)transferred/(float)size));
      IupStoreAttribute(tdlg->gauge, IUP_VALUE, value);
      tdlg->idleTransfer->transfer();
    }
    else {
      IupSetAttribute(tdlg->gauge, IUP_VALUE, "1");
      IupSetAttribute(tdlg->cancelTransferButton, IUP_TITLE, "Close");
      IupSetAttribute(tdlg->progressPopup, IUP_TITLE, "Finished");
      tdlg->cancelTransfer_cb(NULL);
    }
  }
  catch (wioapi::Exception *ex) {
     printf("No foi possvel conexo com o servidor.\n");
     printf("Cliente: %s\n", ex->getClientMessage());
     printf("Servidor: %s\n", ex->getServerMessage());
     delete ex;
  }

  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Fecha a janela principal
 */
int TransferDialog::close_cb(Ihandle * self) {
  return IUP_CLOSE;
}

/* ....................................................................... */
/**
 * Callback chamado quando o usuario escolhe usar como caminho 
 * do arquivo local o caminho default
 */
int TransferDialog::useDefault_cb(Ihandle * self, int isOn) {
  TransferDialog* tdlg = (TransferDialog*) IupGetAttribute(
        self, "TRANSFER_DIALOG");

  if (isOn) {
    IupSetAttribute(tdlg->localBrowseButton, IUP_ACTIVE, "NO");
    IupSetAttribute(tdlg->localPath, IUP_ACTIVE, "NO");
  }
  else {
    IupSetAttribute(tdlg->localBrowseButton, IUP_ACTIVE, "YES");
    IupSetAttribute(tdlg->localPath, IUP_ACTIVE, "YES");
  }
  
  tdlg->activateTransferButton();
  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Abre a janela de browse sobre o sistema de arquivos local
 */
int TransferDialog::localBrowse_cb(Ihandle* self) {
  TransferDialog* tdlg = (TransferDialog*) IupGetAttribute(
         self, "TRANSFER_DIALOG");

  Ihandle* filedlg = IupFileDlg();
  IupSetAttributes(filedlg, "DIALOGTYPE = SAVE, TITLE = \"File Save\"");

  if (!tdlg->getOnly) {
    char *operation = IupGetAttribute(tdlg->operationRadio, "VALUE");
    if (strcmp(operation, "send") == 0) {
       IupSetAttributes(filedlg, "DIALOGTYPE = LOAD, TITLE = \"File Load\"");
    }
  }

  if (tdlg->localFileName != NULL && strlen(tdlg->localFileName) != 0) {
     char* tempName = strdup(tdlg->localFileName);
     #ifdef WIN32
       char* t = strrchr(tempName, '\\');
     #else
       char* t = strrchr(tempName, tdlg->separatorChar[0]);
     #endif
     char tempChar = t[1];
     t[1] = '\0';
     IupStoreAttribute(filedlg, IUP_DIRECTORY, tempName);
     t[1] = tempChar;
     IupStoreAttribute(filedlg, IUP_FILE, t+1);
     free(tempName);
  }

  IupPopup(filedlg, IUP_CENTER, IUP_CENTER);

  char* fileName = IupGetAttribute(filedlg, IUP_VALUE);
  if (fileName != NULL && strlen(fileName) != 0) {
     if (tdlg->localFileName != NULL) free(tdlg->localFileName);
     tdlg->localFileName = strdup(fileName);
  }

  IupStoreAttribute(tdlg->localPath, IUP_VALUE, tdlg->localFileName);
  tdlg->activateTransferButton();

  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Abre a janela de browse sobre o sistema de arquivos remoto
 */
int TransferDialog::remoteBrowse_cb(Ihandle * self) {
  TransferDialog* tdlg = (TransferDialog*) IupGetAttribute(
          self, "TRANSFER_DIALOG");
  enum WioFileDlgType op = OPEN;
  if (!tdlg->getOnly) {
    char *operation = IupGetAttribute(tdlg->operationRadio, "VALUE");
    if (strcmp(operation, "send") == 0) {
      bool useDirToggleIsOn = strcmp(
        IupGetAttribute(tdlg->useDirToggle, IUP_VALUE), IUP_ON) == 0;
      op = useDirToggleIsOn ? DIR : SAVE;
    } 
  }
  //char* fileTypes = "Arquivos Textos|*.txt|Todos os Arquivos|*.*";
  char* fileTypes = "Arquivos Textos|*.txt";
  //char* fileTypes = "All files|*.*";

  if (op == DIR) {
    char* dirName = wioDirDlg(tdlg->filesystem, tdlg->project, ".", "dialog");
    if (dirName != NULL && strlen(dirName) != 0) {
       char* outputName = "output.bin";
       int nDirName = strlen(dirName);
       int nSep = strlen(tdlg->separatorChar);
       int nFileName = strlen(outputName);
       char* cstr = new char[nDirName+nSep+nFileName+1];
       strcpy(cstr, dirName);
       strcat(cstr, tdlg->separatorChar);
       strcat(cstr, outputName);
       if (tdlg->remoteFileName != NULL)
         free(tdlg->remoteFileName);
       tdlg->remoteFileName = cstr;
    } else {
      free(dirName);
    }
  } else {
    char* fileName = wioFileDlg(tdlg->filesystem, tdlg->project, op, 
      NULL, fileTypes, "dialog");
    if (fileName != NULL && strlen(fileName) != 0) {
       if (tdlg->remoteFileName != NULL)
         free(tdlg->remoteFileName);
       tdlg->remoteFileName = fileName;
    }
    else {
      free(fileName);
    }
  }
  
  IupStoreAttribute(tdlg->remotePath, IUP_VALUE, tdlg->remoteFileName);
  tdlg->activateTransferButton();
  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Cancela transferncia de arquivos
 */
int TransferDialog::cancelTransfer_cb(Ihandle * self) {
  IupSetFunction(IUP_IDLE_ACTION, (Icallback) NULL);
 
  // se self == NULL, este callback est sendo chamado a partir da 
  // idle_function, quando a transferncia  finalizada
  if (self != NULL) {
     TransferDialog* tdlg = (TransferDialog*) IupGetAttribute(
           self, "TRANSFER_DIALOG");
     IupHide(tdlg->progressPopup);
     if (tdlg->idleTransfer != NULL) delete tdlg->idleTransfer;
     tdlg->idleTransfer = NULL;
  }
  
  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Callback chamado quando o usurio escolher a operaco a ser realizada.
 */
int TransferDialog::operationRadio_cb(Ihandle * self, int isOn) {
  if (isOn) {
     TransferDialog *tdlg = (TransferDialog *) IupGetAttribute(self,
             "TRANSFER_DIALOG");
     char *operation = IupGetAttribute(tdlg->operationRadio, "VALUE");
     if (strcmp(operation, "send") == 0) {
        IupSetAttribute(tdlg->useDefaultToggle, IUP_VALUE, IUP_OFF);
        IupSetAttribute(tdlg->useDefaultToggle, IUP_ACTIVE, IUP_NO);
        IupSetAttribute(tdlg->useDirToggle, IUP_ACTIVE, IUP_YES);
     }
     else {
        IupSetAttribute(tdlg->localBrowseButton, IUP_ACTIVE, "YES");
        IupSetAttribute(tdlg->useDirToggle, IUP_VALUE, IUP_OFF);
        IupSetAttribute(tdlg->useDirToggle, IUP_ACTIVE, IUP_NO);
        IupSetAttribute(tdlg->useDefaultToggle, IUP_ACTIVE, IUP_YES);
     }
     tdlg->activateTransferButton();
  }
  return IUP_DEFAULT;
}
