/**
 * @mainpage
 * @section intro Introduo
 * A biblioteca est implementada em dois grupos de funes:
 * <ul>
 * <li> Conjunto de funes utilitrias (arquivo <code>wiofiledlg.h</code>),
 * <li> Dilogo de busca de arquivos remoto 
 *      (arquivo <code>TransferDialog.h</code>).
 * </ul>
 */

/**
 * @file wiofiledlg.h
 * Arquivo com a interface exportada com funes utilitrias
 * de busca de um path em um sistema de arquivos WIO remoto.
 * @author Aldo de Araujo Nogueira
 * @author Tatiana Almeida Souza Coelho
 * @author Camilo da Fonseca Freire
 * @author Andr Luiz Clinio
 */

#ifndef _WIOFILEDLG_H_
#define _WIOFILEDLG_H_


#include <wioapi.h>


/**
 * define que representa a declarao dos mtodos para
 * gerao de DLL's em ambiente Windows (se necessrio).
 */
#if !defined(WIOFILEDLG_API)
  #if defined(WIN32)
    #if defined(WIOFILEDLG_DLL_MAKE_EXPORT)
      #define WIOFILEDLG_API __declspec(dllexport)
    #elif defined(WIOFILEDLG_DLL_MAKE_IMPORT)
      #define WIOFILEDLG_API __declspec(dllimport)
    #else
      #define WIOFILEDLG_API
    #endif
  #else
    #define WIOFILEDLG_API
  #endif
#endif

/**
 * Especifica o tipo de caixa de dilogo de arquivos
 */
enum WIOFILEDLG_API WioFileDlgType {
  /** Modo OPEN */ OPEN,
  /** Modo SAVE */ SAVE,
  /** Modo Selea de diretrio */ DIR
};

/**
 * Abre um dilogo para escolha diretrios na rea de projetos.
 * @param server endereo do servidor
 * @param port porta do servio de WIO
 * @param user identificador do usurio
 * @param pwd senha do usurio
 * @param projectId identificador do projeto
 * @param owner identificador do dono do projeto
 * @param initialDir path o diretrio inicial.
 * @param parent identificador da janela-pai que bloquear o dilogo
 * enquanto estiver aberta.
 * @return o caminho (path) do arquivo selecionado; "" 
 *         (string vazia) caso o usurio clique em Cancel ou <code>NULL</code>
 *         caso ocorra algum erro.
 * @throws (wioapi::Exception*) se houver erro de acesso ao servidor WIO.
 */
WIOFILEDLG_API char* wioDirDlg(const char *server, const char *port, const char *user, 
const char *pwd, const char *projectId, const char* owner, 
const char *initialDir, const char *parent) throw (wioapi::Exception*);

/**
 * Abre um dilogo para escolha diretrios na rea de projetos.
 * @param initialDir path o diretrio inicial.
 * @param parent identificador da janela-pai que bloquear o dilogo
 * enquanto estiver aberta.
 * @return o caminho (path) do arquivo selecionado; "" 
 *         (string vazia) caso o usurio clique em Cancel ou <code>NULL</code>
 *         caso ocorra algum erro.
 * @throws (wioapi::Exception*) se houver erro de acesso ao servidor WIO.
 */
WIOFILEDLG_API char* wioDirDlg(wioapi::WIOFileSystem* filesystem, 
wioapi::WIOProject* project, const char* initialDir, const char* parent) 
throw (wioapi::Exception*);

/**
 * Abre um dilogo para escolha de arquivos ou diretrios na rea de projetos.
 * @param server endereo do servidor
 * @param port porta do servio de WIO
 * @param user identificador do usurio
 * @param pwd senha do usurio
 * @param projectId identificador do projeto
 * @param owner identificador do dono do projeto
 * @param type tipo de dilogo de arquivos: para abrir ou salvar arquivos
 * @param parent identificador da janela-pai que bloquear o dilogo
 * enquanto estiver aberta.
 * @return o caminho (path) do arquivo selecionado; "" 
 *         (string vazia) caso o usurio clique em Cancel ou <code>NULL</code>
 *         caso ocorra algum erro.
 * @throws (wioapi::Exception*) se houver erro de acesso ao servidor WIO.
 */
WIOFILEDLG_API char* wioFileDlg(const char* server, const char* port, const char* user, 
const char* pwd, const char* projectId, const char* owner, 
enum WioFileDlgType type, const char *defaultExt, const char *filterExt, 
const char* parent) throw (wioapi::Exception*);


/**
 * Abre um dilogo para escolha de arquivos ou diretrios na rea de projetos.
 * @param server endereo do servidor
 * @param port porta do servio de WIO
 * @param user identificador do usurio
 * @param pwd senha do usurio
 * @param projectId identificador do projeto
 * @param type tipo de dilogo de arquivos: para abrir ou salvar arquivos
 * @param parent o identificador da janela IUP pai; este bloquear 
 *        enquanto a janela estiver aberta.
 * @return o caminho do arquivo selecionado; 
 *         "" (string vazia) caso o usurio clique em <b>cancel</b> ou 
 *         <code>NULL</code> caso ocorra algum erro.
 * @throws (wioapi::Exception*) se houver erro de acesso ao servidor WIO.
 */
WIOFILEDLG_API char* wioFileDlg(const char* server, const char* port, const char* user, 
const char* pwd, const char* projectId, const char* owner, 
enum WioFileDlgType type, const char* parent) 
throw (wioapi::Exception*);


/**
 * Abre um dilogo para escolha de arquivos ou diretrios na rea de projetos.
 * @param filesystem um servidor remoto.
 * @param project um projeto remoto.
 * @param type tipo de dilogo de arquivos: para abrir ou salvar arquivos
 * @param defaultExt a extenso default a ser usada.
 * @param extFilter o filtro desejado.
 * @param parent o identificador da janela IUP pai; 
 *        este bloquear enquanto a janela estiver aberta.
 * @return o caminho do arquivo selecionado; 
 *         "" (string vazia) caso o usurio clique em <b>cancel</b> ou 
 *         <code>NULL</code> caso ocorra algum erro.
 * @throws (wioapi::Exception*) se houver erro de acesso ao servidor WIO.
 */
WIOFILEDLG_API char* wioFileDlg(wioapi::WIOFileSystem* filesystem,
wioapi::WIOProject* project, enum WioFileDlgType type, 
const char *defaultExt, const char *extFilter, const char* parent) 
throw (wioapi::Exception*);

/**
 * Abre um dilogo para escolha de arquivos ou diretrios na rea de projetos.
 * @param filesystem representa um servio de acesso  rea de projetos WIO
 * @param project o projeto aberto
 * @param type o tipo de dilogo de arquivos: para abrir ou salvar arquivos
 * @param parent o identificador da janela IUP pai. Este bloquear enquanto a
 * janela estiver aberta.
 * @return o caminho do arquivo selecionado; 
 *         "" (string vazia) caso o usurio clique em <b>cancel</b> ou 
 *         <code>NULL</code> caso ocorra algum erro.
 * @throws (wioapi::Exception*) se houver erro de acesso ao servidor WIO.
 */
WIOFILEDLG_API char* wioFileDlg(wioapi::WIOFileSystem* filesystem,
wioapi::WIOProject* project, enum WioFileDlgType type, 
const char* parent) throw (wioapi::Exception*);

#endif // _WIOFILEDLG_H_
