package csdk.v1_0.api.filesystem.project;

import java.awt.Window;

import csdk.v1_0.api.core.IContext;
import csdk.v1_0.api.filesystem.FileSelectionType;
import csdk.v1_0.api.filesystem.IFile;

/**
 * Define os mtodos disponveis para acesso da aplicao  rea de projetos
 * durante a sua execuo.
 */
public interface IProjectContext extends IContext {

  /**
   * Mostra um navegador para seleo de um arquivo de projeto em modo de
   * leitura.
   * 
   * @param fileTypes os tipos de arquivos selecionveis
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param selectionType tipo de seleo do navegador (somente arquivos ou
   *        somente diretrios).
   * @param window a janela-me do navegador de arquivos.
   * @return o arquivo selecionado ou <code>null</code> caso a operao tenha
   *         sido cancelada.
   * @throws ProjectException em caso de erro ao acessar a rea de projetos.
   */
  public IFile browseProjectFileInOpenMode(String[] fileTypes,
    boolean allowAll, FileSelectionType selectionType, Window window)
    throws ProjectException;

  /**
   * Mostra um navegador para seleo de mltiplos arquivos de projeto em modo
   * de leitura.
   * 
   * @param fileTypes os tipos de arquivos selecionveis
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param selectionType tipo de seleo do navegador (somente arquivos ou
   *        somente diretrios).
   * @param window a janela-me do navegador de arquivos.
   * @return o arquivo selecionado ou <code>null</code> caso a operao tenha
   *         sido cancelada.
   * @throws ProjectException em caso de erro ao acessar a rea de projetos.
   */
  public IFile[] browseMultipleProjectFilesInOpenMode(String[] fileTypes,
    boolean allowAll, FileSelectionType selectionType, Window window)
    throws ProjectException;

  /**
   * Mostra um navegador para seleo de um arquivo de projeto em modo de
   * escrita.
   * 
   * @param fileTypes os tipos de arquivo selecionveis
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param defaultName o nome padro do arquivo.
   * @param window a janela-me do navegador de arquivos.
   * @return o arquivo selecionado ou <code>null</code> caso a operao tenha
   *         sido cancelada.
   * @throws ProjectException em caso de erro ao acessar a rea de projetos.
   */
  public IFile browseProjectFileInSaveMode(String[] fileTypes,
    boolean allowAll, String defaultName, Window window)
    throws ProjectException;

  /**
   * Obtm o arquivo de projeto com o caminho especificado.
   * 
   * @param path o caminho.
   * @param window a janela-me da ao.
   * @return o arquivo selecionado ou <code>null</code> caso o arquivo no
   *         exista no projeto.
   * @throws ProjectException em caso de erro ao obter o arquivo.
   */
  public IFile getProjectFile(String[] path, Window window)
    throws ProjectException;

  /**
   * Cria um arquivo de projeto no diretrio especificado.
   * 
   * @param path o caminho do diretrio pai.
   * @param name o nome do arquivo.
   * @param fileType o tipo do arquivo ou "DIRECTORY_TYPE" caso seja um
   *        diretrio.
   * @param window a janela-me da ao.
   * @return o arquivo criado.
   * @throws ProjectException em caso de erro ao criar o arquivo.
   */
  public IFile createProjectFile(String[] path, String name, String fileType,
    Window window) throws ProjectException;

  /**
   * Remove um arquivo ou diretrio de projeto. Para remover um diretrio, este
   * precisa estar vazio.
   * 
   * @param path o caminho para o arquivo.
   * @param window a janela-me da ao.
   * @return <code>true</code> caso a remoo seja feita com sucesso ou
   *         <code>false</code> caso contrrio.
   * @throws ProjectException em caso de erro ao remover o arquivo.
   */
  public boolean removeProjectFile(String[] path, Window window)
    throws ProjectException;

  /**
   * Obtm o identificador do projeto corrente ou <code>null</code>, caso no
   * haja projeto aberto no momento.
   * 
   * @return o identificador do projeto ou <code>null</code>, caso no haja
   *         projeto aberto no momento.
   */
  public String getCurrentProjectId();

  /**
   * Adiciona um observador de mudanas no projeto corrente.
   * 
   * @param listener o observador.
   */
  public void addProjectObserver(IProjectObserver listener);

  /**
   * Remove um observador de mudanas do projeto corrente.
   * 
   * @param listener o observador.
   */
  public void removeProjectObserver(IProjectObserver listener);

}