package csdk.v2.fx.helper;

import java.net.URL;

import java.util.ResourceBundle;

import csdk.v2.api.core.ICSDKEnvironment;
import csdk.v2.helper.AbstractCSDKWindowApplication;
import javafx.application.Platform;
import javafx.embed.swing.JFXPanel;
import javafx.fxml.FXMLLoader;
import javafx.scene.Parent;
import javafx.scene.Scene;
import javafx.scene.control.Alert;
import javafx.scene.control.Alert.AlertType;
import javafx.stage.Modality;
import javafx.stage.Window;

/**
 * Classe abstrata para simplificar a criao de aplicaes CSDK que fazem
 * integrao entre JFX e Swing.
 *
 * @author Tecgraf/PUC-Rio
 */
public abstract class AbstractJFXApplication extends
  AbstractCSDKWindowApplication {

  /**
   * Painel FX principal da aplicao.
   */
  private JFXPanel jfxPanel;

  /**
   * Construtor.
   * 
   * @param csdkInterface ambiente CSDK
   */
  public AbstractJFXApplication(ICSDKEnvironment csdkInterface) {
    super(csdkInterface);
    jfxPanel = new JFXPanel(); // inicializa o ambiente JavaFX
    Platform.setImplicitExit(false);
    initFX();
    getApplicationFrame().add(jfxPanel);
  }

  /**
   * Faz a inicializao dos componentes grficos em uma Thread JavaFX. Cria o
   * n raiz e a cena principal, configura o estilo da cena e associa a cena ao
   * painel JFX associado ao frame da aplicao.
   */
  public void initFX() {
    Platform.runLater(new Runnable() {
      @Override
      public void run() {
        try {
          Parent root = getParentNode();
          Scene scene = new Scene(root);
          configStyle(scene);
          jfxPanel.setScene(scene);
        }
        catch (Exception e) {
          handleException(e, getApplicationFrame());
          e.printStackTrace();
        }
      }
    });
  }

  /**
   * Retorna o n pai da cena principal da aplicao. Pode-se criar este n
   * explicitamente ou tambm usar o mtodo
   * {@code AbstractJFXApplication#loadParentNode(URL, ResourceBundle, Object)}.
   * 
   * @return n pai da cena principal da aplicao
   * 
   * @throws Exception exceo na criao do n
   */
  protected abstract Parent getParentNode() throws Exception;

  /**
   * Mtodo para facilitar o carregamento dos ns de uma GUI, a partir de um
   * arquivo fxml, seu bundle e seu controller (opcional).
   * 
   * @param fxml fxml com os campos da tela
   * @param bundle arquivo de idiomas associado ao fxml
   * @param controller controlador associado, pode ser null, caso o controller
   *        seja fornecido diretamente no fxml
   * @return n lido do fxml
   * 
   * @throws Exception
   */
  protected final Parent loadParentNode(URL fxml, ResourceBundle bundle,
    Object controller) throws Exception {
    FXMLLoader loader = new FXMLLoader(fxml, bundle);
    if (controller != null) {
      loader.setController(controller);
    }
    Parent root = loader.load();
    return root;
  }

  /**
   * Define o estilo da cena. Pode ser associado CSS, ajustadas as configuraes
   * de preenchimento, etc. Comportamneto default no implementa nenhnum ajuste.
   * 
   * @param scene cena principal que ser configurada
   */
  protected void configStyle(Scene scene) {
    // default vazio
  }

  /**
   * Aplicao JavaFX deve chamar este mtodo da EDT para boto de fechamento.
   */
  @Override
  public void finishApplication() {
    if (Platform.isFxApplicationThread()) {
      // BUG 1 conhecido (porm em cenrio RMI):
      // http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=8019274
      // verificamos que tambm ocorre no cenrio usando com Thread JFX
      // Para contornar: deixamos com a aplicao a responsabilidade de chamar
      // SwingThreadDispatcher.invokeLater (para no depender de javautils)
      throw new RuntimeException(
        "Cdigo deve ser executado na EDT, porm est rodando na Thread FX!");
    }

    // aqui assumimos que estamos na EDT...

    // BUG 2 conhecido:
    // http://www.java-forum.org/thema/jframe-mit-jfxpanel-npe-bei-dispose.172437/
    // Para verificar problema: https://bugs.openjdk.java.net/browse/JDK-8089371
    // Para contornar: antes da chamada ao dispose() do JFrame  necessrio 
    // remover jfxPanel do frame (OU remover cena do jfxPanel)
    jfxPanel.setScene(null);
    super.finishApplication();
  }

  /**
   * Consulta a Window JavaFx associada  cena principal (semelhante ao owner fx
   * do dilogo principal da aplicao). Se no houver nenhuma cena associada ao
   * painelJFX retorna null.
   * 
   * @return Window associado a cena principal
   */
  protected Window getWindowStage() {
    Scene scene = jfxPanel.getScene();
    if (scene == null) {
      return null;
    }
    return scene.getWindow();
  }

  /**
   * Exibe um dilogo fx modal com o tipo, header e a mensagem recebidos como
   * parmetro.
   * 
   * @param type tipo do dilogo
   * @param header texto para o header
   * @param message mensagem de informao do dilogo
   */
  protected void showModalAlert(AlertType type, String header, String message) {
    Alert alert = new Alert(type);
    alert.initModality(Modality.WINDOW_MODAL);
    alert.initOwner(getWindowStage());
    alert.setTitle(getApplicationName());
    alert.setHeaderText(header);
    alert.setContentText(message);
    alert.setResizable(true);
    alert.show();
  }
  
}