package testealgo;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.HashMap;
import java.util.Map;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import csdk.v1_0.api.application.ApplicationException;
import csdk.v1_0.api.application.IApplication;
import csdk.v1_0.api.application.IApplicationContext;
import csdk.v1_0.api.application.IMessage;
import csdk.v1_0.api.application.IMessageSender;
import csdk.v1_0.api.command.CommandException;
import csdk.v1_0.api.command.IAlgorithmTemplate;
import csdk.v1_0.api.command.ICommandContext;
import csdk.v1_0.api.command.ICommandInfo;
import csdk.v1_0.api.command.ICommandObserver;
import csdk.v1_0.api.command.IFlowTemplate;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.filesystem.FileSelectionType;
import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.api.filesystem.local.ILocalFileSystemContext;
import csdk.v1_0.api.filesystem.project.IProjectContext;
import csdk.v1_0.helper.ContextNotAvailableException;
import csdk.v1_0.helper.command.AlgorithmTemplate;
import csdk.v1_0.helper.command.FlowTemplate;
import csdk.v1_0.helper.command.TemplateHelper;
import csdk.v1_0.helper.command.parameters.BooleanParameterValue;
import csdk.v1_0.helper.command.parameters.DoubleListParameterValue;
import csdk.v1_0.helper.command.parameters.DoubleParameterValue;
import csdk.v1_0.helper.command.parameters.EnumerationListParameterValue;
import csdk.v1_0.helper.command.parameters.EnumerationParameterValue;
import csdk.v1_0.helper.command.parameters.FileListParameterValue;
import csdk.v1_0.helper.command.parameters.FileParameterValue;
import csdk.v1_0.helper.command.parameters.FormulaParameterValue;
import csdk.v1_0.helper.command.parameters.IParameterValue;
import csdk.v1_0.helper.command.parameters.IntegerListParameterValue;
import csdk.v1_0.helper.command.parameters.IntegerParameterValue;
import csdk.v1_0.helper.command.parameters.ParameterValueConvertionException;
import csdk.v1_0.helper.command.parameters.RawParameterValue;
import csdk.v1_0.helper.command.parameters.TableParameterValue;
import csdk.v1_0.helper.command.parameters.TextListParameterValue;
import csdk.v1_0.helper.command.parameters.TextParameterValue;

/**
 * Aplicao de teste simples.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class TesteAlgo implements IApplication {

  /**
   * Interface padro para o ambiente CSDK.
   */
  private final ICSDKEnvironment csdkInterface;

  /**
   * Contexto da aplicao.
   */
  private final IApplicationContext appContext;

  /**
   * Contexto de execuo da aplicao.
   */
  private final ICommandContext commandContext;

  /**
   * Contexto de acesso a projetos.
   */
  private final IProjectContext projectContext;

  /**
   * Contexto de acesso ao sistema de arquivos.
   */
  private final ILocalFileSystemContext localFileSystemContext;

  /**
   * Janela principal da aplicao.
   */
  private final JFrame mainFrame;

  /**
   * Construtor padro.
   * 
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  public TesteAlgo(final ICSDKEnvironment csdkInterface) {
    this.csdkInterface = csdkInterface;
    this.appContext = csdkInterface.getContext(IApplicationContext.class);
    this.commandContext = csdkInterface.getContext(ICommandContext.class);
    this.projectContext = csdkInterface.getContext(IProjectContext.class);
    this.localFileSystemContext = csdkInterface.getContext(ILocalFileSystemContext.class);
    this.mainFrame = new JFrame();
    this.mainFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
    this.mainFrame.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        csdkInterface.finishApplication();
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canEndApplication() {
    int result =
      JOptionPane.showConfirmDialog(mainFrame, appContext
        .getString("confirm.message"), appContext.getApplicationName(),
        JOptionPane.YES_NO_OPTION);
    return result == JOptionPane.YES_OPTION;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onMessageReceived(IMessage message, IMessageSender sender) {
    // No faz nada
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationEnd() throws ApplicationException {
    mainFrame.dispose();
  }

  /**
   * Inicializa a interface da aplicao.
   */
  private void initUI() {
    String algoButtonName = appContext.getString("execute.algo.button");
    JButton algoButton = new JButton(algoButtonName);
    algoButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent arg0) {
        try {
          executeAlgorithm();
        }
        catch (Exception e) {
          csdkInterface.handleException(e, mainFrame);
        }
      }
    });
    algoButton.setEnabled(commandContext != null);
    String flowButtonName = appContext.getString("execute.flow.button");
    JButton flowButton = new JButton(flowButtonName);
    flowButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent arg0) {
        try {
          executeFlow();
        }
        catch (CommandException e) {
          csdkInterface.handleException(e, mainFrame);
        }
      }
    });
    flowButton.setEnabled(commandContext != null);
    String flowFromFileButtonName =
      appContext.getString("execute.flow.from.file.button");
    JButton flowFromFileButton = new JButton(flowFromFileButtonName);
    flowFromFileButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent arg0) {
        try {
          executeFlowFromFile();
        }
        catch (CommandException e) {
          csdkInterface.handleException(e, mainFrame);
        }
      }
    });
    flowFromFileButton.setEnabled(commandContext != null);
    JPanel panel = new JPanel();
    panel.add(algoButton);
    panel.add(flowButton);
    panel.add(flowFromFileButton);
    mainFrame.add(panel);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationStart() throws ApplicationException {
    initUI();
    mainFrame.pack();
    mainFrame.setVisible(true);
  }

  /**
   * Executa um algoritmo simples.
   * 
   * @throws CommandException em caso de erro ao submeter o comando.
   * @throws ParameterValueConvertionException em caso de erro na
   *         parametrizao.
   */
  protected void executeAlgorithm() throws CommandException,
    ParameterValueConvertionException {
    TemplateHelper helper = new TemplateHelper();
    FileParameterValue fileValue =
      new FileParameterValue("entrada.txt", "TEXT");
    helper.addParameterValue("ARQUIVO_DE_ENTRADA", fileValue);
    FileParameterValue fileValue2 =
      new FileParameterValue("entrada2.txt", "TEXT");
    FileParameterValue fileValue3 =
      new FileParameterValue("entrada3.txt", "TEXT");
    FileListParameterValue fileListValue = new FileListParameterValue();
    fileListValue.addValue(fileValue);
    fileListValue.addValue(fileValue2);
    fileListValue.addValue(fileValue3);
    helper.addParameterValue("LISTA_DE_ARQUIVOS_DE_ENTRADA", fileListValue);
    FileParameterValue dirValue = new FileParameterValue("dados", "DIR");
    helper.addParameterValue("DIRETORIO_DE_ENTRADA", dirValue);
    FileListParameterValue fileDirListValue = new FileListParameterValue();
    fileDirListValue.addValue(fileValue);
    fileDirListValue.addValue(fileValue2);
    fileDirListValue.addValue(dirValue);
    helper.addParameterValue("LISTA_DE_ENTRADAS", fileDirListValue);
    helper.addParameterValue("ARQUIVO_DE_SAIDA", fileValue);
    helper.addParameterValue("DIRETORIO_DE_SAIDA", dirValue);
    helper.addParameterValue("LISTA_DE_ARQUIVOS_DE_SAIDA", fileListValue);
    BooleanParameterValue booleanValue = new BooleanParameterValue(true);
    helper.addParameterValue("BOOLEANO", booleanValue);
    IntegerParameterValue integerValue = new IntegerParameterValue(3);
    helper.addParameterValue("INTEIRO", integerValue);
    EnumerationParameterValue enumValue =
      new EnumerationParameterValue("item2");
    helper.addParameterValue("ENUMERACAO", enumValue);
    EnumerationListParameterValue enumListValue =
      new EnumerationListParameterValue();
    enumListValue.addValue("item1");
    enumListValue.addValue("item2");
    helper.addParameterValue("ENUMERACAO_MULTIPLA", enumListValue);
    FormulaParameterValue formulaValue = new FormulaParameterValue("A + B");
    helper.addParameterValue("FORMULA", formulaValue);
    IntegerListParameterValue intListValue =
      new IntegerListParameterValue(new Integer[] { 1, 2, 3 });
    helper.addParameterValue("LISTA_DE_INTEIROS", intListValue);
    DoubleParameterValue doubleValue = new DoubleParameterValue(0.5);
    helper.addParameterValue("REAL", doubleValue);
    DoubleListParameterValue doubleListValue =
      new DoubleListParameterValue(new Double[] { 1.5, 2.5, 3.5 });
    helper.addParameterValue("LISTA_DE_REAIS", doubleListValue);
    TextParameterValue textValue = new TextParameterValue("abc");
    helper.addParameterValue("TEXTO", textValue);
    TextListParameterValue textListValue =
      new TextListParameterValue(new String[] { "aaa", "bbb", "ccc" });
    helper.addParameterValue("LISTA_DE_TEXTO", textListValue);
    TableParameterValue tableValue = new TableParameterValue();
    IParameterValue<?>[] row1 =
      { new DoubleParameterValue(1.0), new BooleanParameterValue(false),
          new IntegerParameterValue(1), new TextParameterValue("linha1"),
          new EnumerationParameterValue("item1") };
    IParameterValue<?>[] row2 =
      { new DoubleParameterValue(2.0), new BooleanParameterValue(true),
          new IntegerParameterValue(2), new TextParameterValue("linha2"),
          new EnumerationParameterValue("item2") };
    tableValue.addRow(row1);
    tableValue.addRow(row2);
    helper.addParameterValue("TABELA", tableValue);
    RawParameterValue rawValue = new RawParameterValue(askForParameterValue());
    helper.addParameterValue("EXIT_CODE", rawValue);
    Map<String, String> parameterValues = helper.generateParameterValuesMap();
    IAlgorithmTemplate info =
      new AlgorithmTemplate("AlgoritmoTesteParametros", "1.0.0",
        parameterValues);
    commandContext.executeAlgorithm(info, "teste CSDK", null, mainFrame,
      new CommandObserver());
  }

  /**
   * Pede para o usurio informar o valor de um parmetro do algoritmo.
   * 
   * @return o valor do parmetro.
   */
  private String askForParameterValue() {
    String result =
      JOptionPane.showInputDialog(mainFrame, "Cdigo de sada desejado:");
    if (result == null || result.isEmpty()) {
      result = "0";
    }
    return result;
  }

  /**
   * Executa um fluxo de algoritmos.
   * 
   * @throws CommandException em caso de erro ao submeter o comando.
   */
  protected void executeFlow() throws CommandException {
    Map<Integer, IAlgorithmTemplate> algosInfo =
      new HashMap<Integer, IAlgorithmTemplate>();
    IFile file = null;
    try {
      String[] fileTypes = new String[] { "TEXT" };
      if (projectContext != null) {
        file =
          projectContext.browseProjectFileInOpenMode(fileTypes, true,
            FileSelectionType.FILES_ONLY, mainFrame);
      }
      else if (localFileSystemContext != null) {
        localFileSystemContext.browseLocalFileInOpenMode(fileTypes, true, null,
          FileSelectionType.FILES_ONLY, mainFrame);
      }
      else {
        throw new ContextNotAvailableException(ILocalFileSystemContext.class);
      }
    }
    catch (Exception e) {
      csdkInterface.handleException(e, mainFrame);
    }
    if (file == null) {
      return;
    }
    HashMap<String, String> parameterValues1 = new HashMap<String, String>();
    parameterValues1.put("ENTRADA1", file.getStringPath());
    parameterValues1.put("SLEEP", "10");
    IAlgorithmTemplate algoInfo1 =
      new AlgorithmTemplate("AlgoritmoTeste", "1.0.0", parameterValues1);
    algosInfo.put(1, algoInfo1);
    HashMap<String, String> parameterValues2 = new HashMap<String, String>();
    parameterValues2.put("SLEEP", "2");
    IAlgorithmTemplate algoInfo2 =
      new AlgorithmTemplate("AlgoritmoTeste", "1.0.0", parameterValues2);
    algosInfo.put(2, algoInfo2);
    IFlowTemplate info = new FlowTemplate("FluxoSimples", "1.0.0", algosInfo);
    commandContext.executeFlow(info, "teste fluxo CSDK", null, mainFrame,
      new CommandObserver());
  }

  /**
   * Executa um fluxo de algoritmos.
   * 
   * @throws CommandException em caso de erro ao submeter o comando.
   */
  protected void executeFlowFromFile() throws CommandException {
    Map<Integer, IAlgorithmTemplate> algosInfo =
      new HashMap<Integer, IAlgorithmTemplate>();
    IFile file = null;
    try {
      String[] fileTypes = new String[] { "FLX" };
      if (projectContext != null) {
        file =
          projectContext.browseProjectFileInOpenMode(fileTypes, true,
            FileSelectionType.FILES_ONLY, mainFrame);
      }
      else if (localFileSystemContext != null) {
        file =
          localFileSystemContext.browseLocalFileInOpenMode(fileTypes, true, null,
            FileSelectionType.FILES_ONLY, mainFrame);
      }
      else {
        throw new ContextNotAvailableException(ILocalFileSystemContext.class);
      }
    }
    catch (Exception e) {
      csdkInterface.handleException(e, mainFrame);
    }
    if (file == null) {
      return;
    }
    HashMap<String, String> parameterValues1 = new HashMap<String, String>();
    parameterValues1.put("SLEEP", "10");
    IAlgorithmTemplate algoInfo1 =
      new AlgorithmTemplate("AlgoritmoTeste", "1.0.0", parameterValues1);
    algosInfo.put(1, algoInfo1);
    HashMap<String, String> parameterValues2 = new HashMap<String, String>();
    parameterValues2.put("SLEEP", "20");
    IAlgorithmTemplate algoInfo2 =
      new AlgorithmTemplate("AlgoritmoTeste", "1.0.0", parameterValues2);
    algosInfo.put(2, algoInfo2);
    IFlowTemplate info = new FlowTemplate(file, algosInfo);
    commandContext.executeFlow(info, "teste fluxo com arquivo CSDK", null,
      mainFrame, new CommandObserver());
  }

  /**
   * Observador de comandos.
   */
  private final class CommandObserver implements ICommandObserver {
    /**
     * {@inheritDoc}
     */
    @Override
    public void onCommandEnd(final ICommandInfo cmdInfo) {
      SwingUtilities.invokeLater(new Runnable() {
        @Override
        public void run() {
          CommandReportFrame frame = new CommandReportFrame(cmdInfo);
          frame.setVisible(true);
        }
      });
    }
  }

}
