/*
 * Detalhes da ltima alterao:
 * 
 * $Author: vfusco $ $Date: 2009-02-18 18:33:43 -0300 (Wed, 18 Feb 2009) $
 * $Revision: 88595 $
 */
package tecgraf.ftc_1_2.client;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;

import tecgraf.ftc_1_2.common.exception.FailureException;
import tecgraf.ftc_1_2.common.exception.FileLockedException;
import tecgraf.ftc_1_2.common.exception.InvalidProtocolVersionException;
import tecgraf.ftc_1_2.common.exception.MaxClientsReachedException;
import tecgraf.ftc_1_2.common.exception.PermissionException;
import tecgraf.ftc_1_2.common.logic.ErrorCode;
import tecgraf.ftc_1_2.common.logic.Operation;
import tecgraf.ftc_1_2.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_2.common.logic.ProtocolVersion;
import tecgraf.ftc_1_2.utils.ByteBufferUtils;

/**
 * Representa uma implementao de um canal de arquivo remoto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class RemoteFileChannelImpl implements RemoteFileChannel {
  /**
   * A mquina do servidor de arquivos.
   */
  private String host;
  /**
   * A porta do servidor de arquivos.
   */
  private int port;
  /**
   * A chave de acesso ao arquivo.  usada para verificar se o usurio possui
   * permisso de acesso ao arquivo.
   */
  private byte[] key;
  /**
   * O identificador do arquivo.
   */
  private byte[] identifier;
  /**
   * Indica se  permitida a escrita no arquivo.
   */
  private boolean writable;
  /**
   * Indica se o arquivo est aberto.
   */
  private boolean open = false;
  /**
   * Indica se o arquivo foi aberto como somente leitura.
   */
  private boolean readOnly;
  /**
   * Canal de comunicao com o servidor de arquivos.
   */
  private SocketChannel channel;
  /**
   * <i>Buffer</i> utilizado na comunicao com o servidor de arquivos.
   */
  private ByteBuffer buffer;

  /**
   * <i>bufferSize</i> determina o tamanho do buffer utilizado.
   */
  private int bufferSize = 1024 * 1024;

  /**
   * Cria um arquivo remoto.
   * 
   * @param identifier O identificador do arquivo.
   * @param writable Indica se a escrita  permitida no arquivo.
   * @param host A mquina do servidor de arquivos.
   * @param port A porta do servidor de arquivos.
   * @param key A chave de acesso ao arquivo.
   */
  public RemoteFileChannelImpl(byte[] identifier, boolean writable,
    String host, int port, byte[] key) {
    this.host = host;
    this.port = port;
    this.key = key;

    this.identifier = identifier;
    this.writable = writable;
  }

  /**
   * {@inheritDoc}
   * 
   * @throws InvalidProtocolVersionException
   */
  public void open(boolean readOnly) throws PermissionException,
    FileNotFoundException, FailureException, MaxClientsReachedException,
    InvalidProtocolVersionException {
    if (readOnly == false && this.writable == false) {
      throw new PermissionException(
        "O arquivo no pode ser aberto para escrita.");
    }
    try {
      this.channel =
        SocketChannel.open(new InetSocketAddress(this.host, this.port));
      this.channel.socket().setTcpNoDelay(true);
    }
    catch (IOException e) {
      throw new FailureException(e);
    }

    this.buffer = ByteBuffer.allocate(bufferSize);

    protocolVersionHandshake();

    authenticate();

    Operation operation =
      readOnly ? Operation.OPEN_READ_ONLY : Operation.OPEN_READ_WRITE;

    ErrorCode errorCode = null;
    try {
      this.buffer.put(operation.getCode());
      ByteBufferUtils.writeBytes(this.buffer, this.channel,
        PrimitiveTypeSize.BYTE.getSize(), this.identifier);
      errorCode = readReturnCode();
    }
    catch (IOException e) {
      this.release();
      throw new FailureException(e);
    }

    if (!ErrorCode.OK.equals(errorCode)) {
      this.release();

      if (errorCode.equals(ErrorCode.FILE_NOT_FOUND)) {
        throw new FileNotFoundException("O arquivo no existe.");
      }
      else if (errorCode.equals(ErrorCode.NO_PERMISSION)) {
        throw new PermissionException("Sem permisso para abrir o arquivo.");
      }
      else if (errorCode.equals(ErrorCode.FAILURE)) {
        throw new FailureException(
          "Falha no servidor ao tentar abrir o arquivo.");
      }
      throw new IllegalStateException("Cdigo de erro invlido " + errorCode);
    }

    this.open = true;
    this.readOnly = readOnly;
  }

  /**
   * Metodo responsavel por realizar o handshake inicial de verso do protocolo.
   * 
   * @throws FailureException
   * @throws PermissionException
   * @throws MaxClientsReachedException
   * @throws InvalidProtocolVersionException
   * 
   */
  private void protocolVersionHandshake() throws FailureException,
    PermissionException, MaxClientsReachedException,
    InvalidProtocolVersionException {
    ErrorCode errorCode = null;
    try {
      long idAndVersion = ProtocolVersion.PROTOCOL_IDENTIFICATION;
      idAndVersion <<= 32;
      int major = ProtocolVersion.MAJOR_VERSION;
      int minor = ProtocolVersion.MINOR_VERSION;
      int patch = ProtocolVersion.PATCH_VERSION;
      idAndVersion |= ((major << 16) | (minor << 8) | (patch));

      ByteBufferUtils.writeLong(this.buffer, this.channel, idAndVersion);
      errorCode = readReturnCode();
    }
    catch (IOException e) {
      this.release();
      throw new FailureException(e);
    }

    if (!ErrorCode.OK.equals(errorCode)) {
      this.release();

      if (errorCode.equals(ErrorCode.INVALID_VERSION)) {
        throw new InvalidProtocolVersionException(
          "Servidor no suporta esta verso do protocolo.");
      }
      else if (errorCode.equals(ErrorCode.MAX_CLIENTS_REACHED)) {
        throw new MaxClientsReachedException(
          "Nmero mximo de clientes no servidor atingido.");
      }
      else if (errorCode.equals(ErrorCode.FAILURE)) {
        throw new FailureException(
          "Falha ao negociar protocolo com o servidor.");
      }
      throw new IllegalStateException("Cdigo de erro invlido " + errorCode);
    }
  }

  /**
   * Metodo responsavel por realizar a autenticao
   * 
   * @throws FailureException
   * @throws PermissionException
   * @throws MaxClientsReachedException
   */
  private void authenticate() throws FailureException, PermissionException,
    MaxClientsReachedException {
    ErrorCode errorCode = null;
    try {
      ByteBufferUtils.writeBytes(this.buffer, this.channel, this.key);
      errorCode = readReturnCode();
    }
    catch (IOException e) {
      this.release();
      throw new FailureException(e);
    }

    if (!ErrorCode.OK.equals(errorCode)) {
      this.release();

      if (errorCode.equals(ErrorCode.INVALID_KEY)) {
        throw new PermissionException("Chave de acesso invlida.");
      }
      else if (errorCode.equals(ErrorCode.FAILURE)) {
        throw new FailureException(
          "Falha no servidor ao tentar se autenticar com a chave fornecida.");
      }
      throw new IllegalStateException("Cdigo de erro invlido " + errorCode);
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean isOpen() {
    return this.open;
  }

  /**
   * {@inheritDoc}
   */
  public void close() throws FailureException {
    if (this.channel == null) {
      throw new IllegalStateException(
        "No  possvel fechar um arquivo que no foi aberto.");
    }
    this.open = false;
    ErrorCode errorCode = ErrorCode.FAILURE;
    try {
      ByteBufferUtils.writeByte(this.buffer, this.channel,
        Operation.CLOSE.getCode());
      errorCode = readReturnCode();
    }
    catch (IOException e) {
      this.release();
      throw new FailureException(e);
    }

    this.release();

    if (ErrorCode.FAILURE.equals(errorCode)) {
      throw new FailureException("Falha no fechamento do canal do arquivo.");
    }
  }

  /**
   * Metodo privado que le o codigo de retorno enviado pelo servidor
   * 
   * @return Codigo de retorno
   * @throws IOException
   */
  private ErrorCode readReturnCode() throws IOException {
    byte code = ByteBufferUtils.readByte(this.buffer, this.channel);
    ErrorCode errorCode = ErrorCode.valueOf(code);
    this.buffer.clear();
    return errorCode;
  }

  /**
   * Libera os recursos alocados pelo arquivo.
   * 
   */
  private void release() {
    try {
      this.channel.close();
    }
    catch (IOException e) {
      // No fazer nada mesmo
      //e.printStackTrace();
    }
    finally {
      this.open = false;
      this.buffer = null;
      this.channel = null;
    }
  }

  /**
   * {@inheritDoc}
   */
  public void setSize(long size) throws PermissionException, FailureException {
    if (this.readOnly) {
      throw new PermissionException(
        "Arquivo foi aberto somente para a leitura.");
    }
    this.buffer.put(Operation.SET_SIZE.getCode());
    try {
      ByteBufferUtils.writeLong(this.buffer, this.channel,
        PrimitiveTypeSize.BYTE.getSize(), size);
      ErrorCode errorCode = readReturnCode();
      if (!ErrorCode.OK.equals(errorCode)) {
        throw new FailureException("Falha ao alterar o tamanho do arquivo.");
      }
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @throws FailureException Se houver falha na comunicao
   */
  public long getPosition() throws FailureException {
    try {
      ByteBufferUtils.writeByte(this.buffer, this.channel,
        Operation.GET_POSITION.getCode());

      ErrorCode errorCode = readReturnCode();

      if (!errorCode.equals(ErrorCode.OK))
        throw new FailureException("Falha ao ler posio do arquivo.");

      long pos = ByteBufferUtils.readLong(this.buffer, this.channel);

      return pos;
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  public void setPosition(long position) throws FailureException {
    try {
      this.buffer.put(Operation.SET_POSITION.getCode());
      ByteBufferUtils.writeLong(this.buffer, this.channel,
        PrimitiveTypeSize.BYTE.getSize(), position);
      ErrorCode errorCode = readReturnCode();
      if (!ErrorCode.OK.equals(errorCode)) {
        throw new FailureException("Falha ao alterar o tamanho do arquivo.");
      }
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @throws FailureException
   */
  public long getSize() throws FailureException {
    try {
      ByteBufferUtils.writeByte(this.buffer, this.channel,
        Operation.GET_SIZE.getCode());

      ErrorCode errorCode = readReturnCode();

      if (!errorCode.equals(ErrorCode.OK))
        throw new FailureException("Falha ao ler tamanho do arquivo");

      long size = ByteBufferUtils.readLong(this.buffer, this.channel);

      return size;
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  public int read(byte[] target) throws FailureException {
    return this.read(target, 0, target.length, -1);
  }

  /**
   * {@inheritDoc}
   */
  public int read(byte[] target, long position) throws FailureException {
    return this.read(target, 0, target.length, position);
  }

  /**
   * {@inheritDoc}
   */
  public int read(byte[] target, int offset, int length)
    throws FailureException {
    return this.read(target, offset, length, -1);
  }

  /**
   * {@inheritDoc}
   */
  public int read(byte[] target, int offset, int length, long position)
    throws FailureException {
    if ((offset < 0) || (length < 0))
      throw new FailureException("Parametros do READ invalidos.");

    this.buffer.put(Operation.READ.getCode());
    this.buffer.putLong(position);
    this.buffer.putLong(length);
    this.buffer.flip();
    try {
      this.channel.write(this.buffer);
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
    finally {
      this.buffer.clear();
    }

    int totalBytesRead = 0;
    while (totalBytesRead < length) {
      try {
        ErrorCode errorCode = readReturnCode();
        if (errorCode.equals(ErrorCode.END_OF_FILE)) {
          if (totalBytesRead == 0)
            return -1;
          return totalBytesRead;
        }
        else if (!errorCode.equals(ErrorCode.OK)) {
          throw new FailureException("Falha ao tentar ler arquivo");
        }

        long chunkSize = ByteBufferUtils.readLong(this.buffer, this.channel);

        ByteBuffer dstBuffer = ByteBuffer.wrap(target, offset, length);

        int bytesRead = 0;
        int readCount = 0;
        while (bytesRead < chunkSize) {
          readCount = this.channel.read(dstBuffer);
          if (readCount < 0)
            throw new FailureException("Falha ao tentar ler dados do canal");
          bytesRead += readCount;
        }
        totalBytesRead += bytesRead;
      }
      catch (IOException e) {
        throw new FailureException(e);
      }
    }

    return totalBytesRead;
  }

  /**
   * {@inheritDoc}
   */
  public int write(byte[] source) throws PermissionException, FailureException,
    FileLockedException {
    return this.write(source, 0, source.length, -1);
  }

  /**
   * {@inheritDoc}
   */
  public int write(byte[] source, long position) throws PermissionException,
    FailureException, FileLockedException {
    return this.write(source, 0, source.length, position);
  }

  /**
   * {@inheritDoc}
   */
  public int write(byte[] source, int offset, int length)
    throws PermissionException, FailureException, FileLockedException {
    return this.write(source, offset, length, -1);
  }

  /**
   * {@inheritDoc}
   */
  public int write(byte[] source, int offset, int length, long position)
    throws PermissionException, FailureException, FileLockedException {
    if (this.readOnly) {
      throw new PermissionException(
        "Arquivo foi aberto somente para a leitura.");
    }
    this.buffer.put(Operation.WRITE.getCode());
    this.buffer.putLong(position);

    ErrorCode errorCode = ErrorCode.FAILURE;
    try {
      ByteBufferUtils.writeLong(this.buffer, this.channel,
        (PrimitiveTypeSize.BYTE.getSize() + PrimitiveTypeSize.LONG.getSize()),
        length);
      errorCode = readReturnCode();
    }
    catch (IOException e) {
      throw new FailureException(e);
    }

    if (ErrorCode.FILE_LOCKED.equals(errorCode)) {
      throw new FileLockedException("Arquivo reservado para outro usurio.");
    }
    else if (ErrorCode.READ_ONLY.equals(errorCode)) {
      throw new PermissionException("Arquivo esta aberto somente para leitura.");
    }
    else if (!ErrorCode.OK.equals(errorCode)) {
      throw new FailureException("Falha ao tentar escrever no arquivo");
    }

    ByteBuffer dstBuffer = ByteBuffer.wrap(source, offset, length);
    try {
      int bytesWritten = 0;
      do {
        bytesWritten += this.channel.write(dstBuffer);
      } while (bytesWritten < length);
      return bytesWritten;
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  public long transferTo(long position, long count, OutputStream outputStream)
    throws FailureException {
    this.buffer.put(Operation.READ.getCode());
    this.buffer.putLong(position);
    this.buffer.putLong(count);
    this.buffer.flip();
    try {
      this.channel.write(this.buffer);
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
    finally {
      this.buffer.clear();
    }

    long bytesWrittenTotal = 0;
    long currentChunkSize = 0;
    int chunkReadBytes = 0;
    int bytesRead = 0;

    while (bytesWrittenTotal < count) {
      try {
        ErrorCode errorCode = readReturnCode();
        if (errorCode.equals(ErrorCode.END_OF_FILE)) {
          this.buffer.clear();
          if (bytesWrittenTotal == 0)
            return -1;
          return bytesWrittenTotal;
        }
        else if (!errorCode.equals(ErrorCode.OK)) {
          throw new FailureException("Falha ao tentar ler arquivo");
        }

        currentChunkSize = ByteBufferUtils.readLong(this.buffer, this.channel);

      }
      catch (IOException e) {
        throw new FailureException(e);
      }

      chunkReadBytes = 0;
      while (chunkReadBytes < currentChunkSize) {
        this.buffer.clear();
        this.buffer.limit((int) (currentChunkSize - chunkReadBytes));
        try {
          bytesRead = this.channel.read(this.buffer);
          if (bytesRead < 0)
            throw new FailureException("Falha ao tentar ler dados do canal");

          outputStream.write(this.buffer.array(), 0, bytesRead);

        }
        catch (IOException e) {
          throw new FailureException(e);
        }
        bytesWrittenTotal += bytesRead;
        chunkReadBytes += bytesRead;
      }
      this.buffer.clear();
    }
    this.buffer.clear();

    return bytesWrittenTotal;
  }

  /**
   * {@inheritDoc}
   */
  public long transferTo(long position, long count, RemoteFileChannel target)
    throws FailureException, PermissionException, FileLockedException {
    this.buffer.put(Operation.READ.getCode());
    this.buffer.putLong(position);
    this.buffer.putLong(count);
    this.buffer.flip();
    try {
      this.channel.write(this.buffer);
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
    finally {
      this.buffer.clear();
    }

    long bytesWrittenTotal = 0;
    long currentChunkSize = 0;
    long chunkReadBytes = 0;
    int bytesRead = 0;

    while (bytesWrittenTotal < count) {
      try {
        ErrorCode errorCode = readReturnCode();
        if (errorCode.equals(ErrorCode.END_OF_FILE)) {
          if (bytesWrittenTotal == 0)
            return -1;
          return bytesWrittenTotal;
        }
        else if (!errorCode.equals(ErrorCode.OK)) {
          throw new FailureException("Falha ao tentar ler arquivo");
        }

        currentChunkSize = ByteBufferUtils.readLong(this.buffer, this.channel);

      }
      catch (IOException e) {
        throw new FailureException(e);
      }

      chunkReadBytes = 0;
      while (chunkReadBytes < currentChunkSize) {
        this.buffer.clear();
        this.buffer.limit((int) (currentChunkSize - chunkReadBytes));
        try {
          bytesRead = this.channel.read(this.buffer);
          if (bytesRead < 0)
            throw new FailureException("Falha ao tentar ler dados do canal");
          buffer.flip();

          target.write(this.buffer.array(), 0, bytesRead);

        }
        catch (IOException e) {
          throw new FailureException(e);
        }
        bytesWrittenTotal += bytesRead;
        chunkReadBytes += bytesRead;
      }
      this.buffer.clear();
    }
    this.buffer.clear();

    return bytesWrittenTotal;
  }

  /**
   * {@inheritDoc}
   */
  public long transferFrom(InputStream source, long position, long count)
    throws PermissionException, FailureException, FileLockedException {
    if (this.readOnly) {
      throw new PermissionException(
        "Arquivo foi aberto somente para a leitura.");
    }
    this.buffer.put(Operation.WRITE.getCode());
    this.buffer.putLong(position);
    ErrorCode errorCode = ErrorCode.FAILURE;
    try {
      ByteBufferUtils.writeLong(this.buffer, this.channel,
        (PrimitiveTypeSize.BYTE.getSize() + PrimitiveTypeSize.LONG.getSize()),
        count);
      errorCode = readReturnCode();
    }
    catch (IOException e) {
      throw new FailureException(e);
    }

    if (ErrorCode.FILE_LOCKED.equals(errorCode)) {
      throw new FileLockedException("Arquivo reservado para outro usurio.");
    }
    byte[] bufferArray = this.buffer.array();
    long bytesReadTotal = 0;
    while (bytesReadTotal < count) {
      int len = bufferArray.length;
      if (bufferArray.length > count - bytesReadTotal) {
        len = (int) (count - bytesReadTotal);
      }
      int bytesRead;
      try {
        bytesRead = source.read(bufferArray, 0, len);
      }
      catch (IOException e) {
        throw new FailureException(e);
      }
      if (bytesRead == -1) {
        break;
      }
      bytesReadTotal += bytesRead;
      this.buffer.limit(bytesRead);
      int bytesWrittenTotal = 0;
      while (this.buffer.hasRemaining()) {
        try {
          int bytesWritten;
          bytesWritten = this.channel.write(this.buffer);
          if (bytesWritten < 0) {
            throw new IllegalStateException();
          }
          bytesWrittenTotal += bytesWritten;
        }
        catch (IOException e) {
          this.buffer.clear();
          throw new FailureException(e);
        }
      }
      this.buffer.clear();
    }
    return bytesReadTotal;
  }

  /**
   * {@inheritDoc}
   */
  public long transferFrom(RemoteFileChannel source, long position, long count)
    throws PermissionException, FailureException, FileLockedException {
    if (this.readOnly) {
      throw new PermissionException(
        "Arquivo foi aberto somente para a leitura.");
    }
    this.buffer.put(Operation.WRITE.getCode());
    this.buffer.putLong(position);
    ErrorCode errorCode = ErrorCode.FAILURE;
    try {
      ByteBufferUtils.writeLong(this.buffer, this.channel,
        (PrimitiveTypeSize.BYTE.getSize() + PrimitiveTypeSize.LONG.getSize()),
        count);
      errorCode = readReturnCode();
    }
    catch (IOException e) {
      throw new FailureException(e);
    }

    if (ErrorCode.FILE_LOCKED.equals(errorCode)) {
      throw new FileLockedException("Arquivo reservado para outro usurio.");
    }
    byte[] bufferArray = this.buffer.array();
    long bytesReadTotal = 0;
    while (bytesReadTotal < count) {
      int len = bufferArray.length;
      if (bufferArray.length > count - bytesReadTotal) {
        len = (int) (count - bytesReadTotal);
      }
      int bytesRead = source.read(bufferArray, 0, len);
      bytesReadTotal += bytesRead;
      this.buffer.limit(bytesRead);
      int bytesWrittenTotal = 0;
      while (this.buffer.hasRemaining()) {
        try {
          int bytesWritten;
          bytesWritten = this.channel.write(this.buffer);
          if (bytesWritten < 0) {
            throw new IllegalStateException();
          }
          bytesWrittenTotal += bytesWritten;
        }
        catch (IOException e) {
          //e.printStackTrace();
          this.buffer.clear();
          throw new FailureException(e);
        }
      }
      this.buffer.clear();
    }
    return bytesReadTotal;
  }

  /**
   * {@inheritDoc}
   * 
   * @throws FailureException
   */
  public void keepAlive() throws FailureException {
    try {
      ByteBufferUtils.writeByte(this.buffer, this.channel,
        Operation.KEEP_ALIVE.getCode());
      ErrorCode errorCode = readReturnCode();

      if (!ErrorCode.OK.equals(errorCode)) {
        throw new FailureException("Erro na operao Keep Alive");
      }
    }
    catch (IOException e) {
      throw new FailureException(e);
    }
  }

  /**
   * @param bufferSize O tamanho do buffer em bytes.
   */
  public void setBufferSize(int bufferSize) {
    this.bufferSize = bufferSize;
  }

  /**
   * @return O tamanho do ByteBuffer Utilizado
   */
  public int getBufferSize() {
    return bufferSize;
  }

}
