/*
 * Detalhes da ltima alterao:
 * 
 * $Author$ $Date$ $Revision$
 */
package tecgraf.ftc_1_2.server.states;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_2.common.logic.ErrorCode;
import tecgraf.ftc_1_2.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_2.server.Session;

/**
 * Operao para definir a nova posio do arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class SetPositionState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a nova posio do arquivo j foi lida.
     */
    POSITION_READ,
    /**
     * Estado que indica que o cdigo de erro j foi enviado.
     */
    ERROR_CODE_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;

  /**
   * A nova posio do arquivo.
   */
  private long position;

  /**
   * O cdigo de erro.
   */
  private ErrorCode errorCode;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria a operao para definir a nova posio do arquivo.
   */
  public SetPositionState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado para definir a posio de um arquivo.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler do canal.");
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.position = buffer.getLong();
        buffer.clear();
        this.currentState = InternalState.POSITION_READ;
        FileChannel fileChannel = session.getFileChannel();
        try {
          fileChannel.position(this.position);
          this.errorCode = ErrorCode.OK;
        }
        catch (IOException e) {
          this.errorCode = ErrorCode.FAILURE;
        }
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case POSITION_READ:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(this.errorCode.getCode());
        buffer.flip();
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao escrever no canal.");
          return false;
        }
        buffer.clear();
        this.currentState = InternalState.ERROR_CODE_SENT;
        session.setCurrentState(new GetOperationState());

        if (logger.isLoggable(Level.FINER))
          logger.finer("Cdigo " + this.errorCode + " enviado.");
      default:
        return true;
    }
  }
}
