/*
 * $Id$
 */
package tecgraf.ftc_1_2.server.states;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_2.common.exception.InvalidArraySize;
import tecgraf.ftc_1_2.common.logic.ErrorCode;
import tecgraf.ftc_1_2.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_2.server.AccessKey;
import tecgraf.ftc_1_2.server.FileChannelRequestInfo;
import tecgraf.ftc_1_2.server.FileServer;
import tecgraf.ftc_1_2.server.Session;

/**
 * Operao de obteno e validao de uma chave de acesso.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ValidateKeyState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que o tamanho da chave j foi recebido.
     */
    KEY_SIZE_RECEIVED,
    /**
     * Estado que indica que a chave j foi recebida.
     */
    KEY_RECEIVED,
    /**
     * Estado que indica que o cdigo de erro j foi enviado.
     */
    ERROR_CODE_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;
  /**
   * O tamanho da chave de acesso.
   */
  private byte keySize;
  /**
   * A chave de acesso.
   */
  private byte[] key;
  /**
   * O cdigo de erro.
   */
  private ErrorCode errorCode;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria a operao de obteno e validao de uma chave de acesso.
   */
  public ValidateKeyState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de validao de chave.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler dados do canal");
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.keySize = buffer.get();
        buffer.clear();
        this.currentState = InternalState.KEY_SIZE_RECEIVED;

        if (logger.isLoggable(Level.FINER))
          logger.finer("Tamanho da chave lido " + this.keySize);
      case KEY_SIZE_RECEIVED:
        buffer.limit(this.keySize);
        try {

          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler dados do canal");
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.key = new byte[this.keySize];
        buffer.get(this.key);
        buffer.clear();
        this.currentState = InternalState.KEY_RECEIVED;
        FileServer fileServer = session.getFileServer();
        AccessKey accessKey = null;

        try {
          accessKey = new AccessKey(this.key);
        }
        catch (InvalidArraySize e) {
          this.errorCode = ErrorCode.INVALID_KEY;
        }

        FileChannelRequestInfo fileChannelInfo =
          fileServer.getFileChannelInfo(accessKey);

        if (fileChannelInfo == null) {
          this.errorCode = ErrorCode.INVALID_KEY;
        }
        else {
          this.errorCode = ErrorCode.OK;
          session.setFileChannelInfo(fileChannelInfo);
        }
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case KEY_RECEIVED:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(this.errorCode.getCode());
        buffer.flip();
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao escrever no canal");
          return false;
        }
        buffer.clear();
        this.currentState = InternalState.ERROR_CODE_SENT;
        if (this.errorCode.equals(ErrorCode.OK)) {
          session.setCurrentState(new GetOperationState());
        }
        else {
          session.setCurrentState(null);
        }
        if (logger.isLoggable(Level.FINER))
          logger.finer("Cdigo " + this.errorCode + " enviado.");
      default:
        return true;
    }
  }
}
