/*
 * $Id$
 */
package tecgraf.ftc_1_2.server.states.v1_1;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_2.common.logic.Operation;
import tecgraf.ftc_1_2.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_2.server.Session;
import tecgraf.ftc_1_2.server.states.State;

/**
 * Responsvel por obter a operao solicitada pelo cliente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class GetOperationState implements State {

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria o estado de espera por operaes.
   */
  public GetOperationState() {
    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de ler operaes.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    byte operationCode;
    buffer.limit(PrimitiveTypeSize.BYTE.getSize());
    try {
      if (channel.read(buffer) > 0)
        session.markLastActivity();
      if (buffer.hasRemaining()) {
        return true;
      }
      buffer.flip();
      operationCode = buffer.get();
    }
    catch (IOException e) {
      session.getFileServer().exceptionRaised(e,
        session.getFileChannelInfo().getFileId());
      return false;
    }
    finally {
      buffer.clear();
    }
    Operation operation = Operation.valueOf(operationCode);
    if (operation == null) {
      if (logger.isLoggable(Level.FINER))
        logger.finer("Operao desconhecida " + operationCode);
    }
    else {

      if (logger.isLoggable(Level.FINER))
        logger.finer("Operao obtida " + operation);

      session.setCurrentState(this.getState(operation));
    }
    return true;
  }

  /**
   * Obtm o estado relacionado a uma operao.
   * 
   * @param operation A operao.
   * 
   * @return O estado.
   */
  private State getState(Operation operation) {
    switch (operation) {
      case OPEN_READ_ONLY:
        return new OpenReadOnlyState();
      case OPEN_READ_WRITE:
        return new OpenReadWriteState();
      case CLOSE:
        return new CloseState();
      case SET_SIZE:
        return new SetSizeState();
      case GET_POSITION:
        return new GetPositionState();
      case SET_POSITION:
        return new SetPositionState();
      case GET_SIZE:
        return new GetSizeState();
      case READ:
        return new ReadState();
      case WRITE:
        return new WriteState();
      default:
        return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    return true;
  }
}
