/*
 * Detalhes da ltima alterao:
 * 
 * $Author$ $Date$ $Revision$
 */
package tecgraf.ftc_1_2.server.states.v1_1;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_2.common.logic.ErrorCode;
import tecgraf.ftc_1_2.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_2.server.Session;
import tecgraf.ftc_1_2.server.states.State;

/**
 * Operao para truncar o arquivo para um novo tamanho.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class SetSizeState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que o novo tamanho do arquivo j foi lido.
     */
    SIZE_READ,
    /**
     * Estado que indica que o cdigo de erro j foi enviado.
     */
    ERROR_CODE_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;

  /**
   * O cdigo de erro.
   */
  private ErrorCode errorCode;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger =
    Logger.getLogger("tecgraf.ftc");

  /**
   * Cria a operao para ajustar o arquivo para um novo tamanho.
   */
  public SetSizeState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado que altera o tamanho de um arquivo.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        long size = buffer.getLong();
        buffer.clear();
        this.currentState = InternalState.SIZE_READ;
        FileChannel fileChannel = session.getFileChannel();
        try {

          if (logger.isLoggable(Level.FINER)) {
            logger.finer("Atual tamanho: " + fileChannel.size());
            logger.finer("Novo tamanho: " + size);
          }

          if (size > fileChannel.size()) {
            ByteBuffer src = ByteBuffer.wrap(new byte[] { 0 });
            long currentPosition = fileChannel.position();
            fileChannel.position(size - 1);
            fileChannel.write(src);
            fileChannel.position(currentPosition);
          }
          else {
            fileChannel.truncate(size);
          }
          this.errorCode = ErrorCode.OK;
        }
        catch (IOException e) {
          e.printStackTrace();
          this.errorCode = ErrorCode.FAILURE;
        }
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case SIZE_READ:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(this.errorCode.getCode());
        buffer.flip();
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        finally {
          buffer.clear();
        }
        this.currentState = InternalState.ERROR_CODE_SENT;
        session.setCurrentState(new GetOperationState());

        if (logger.isLoggable(Level.FINER))
          logger.finer("Cdigo " + this.errorCode + " enviado.");

      default:
        return true;
    }
  }
}
