package tecgraf.ftc_1_3.server.states;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_3.common.logic.ErrorCode;
import tecgraf.ftc_1_3.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_3.server.Session;

/**
 * Operao que mantem uma conexao ativa.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class KeepAliveState implements State {

  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    RETURN_MSG_WRITTEN
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria a operao de obteno do tamanho do arquivo.
   */
  public KeepAliveState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de obteno do tamanho do arquivo.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(ErrorCode.OK.getCode());
        buffer.flip();
        currentState = InternalState.RETURN_MSG_WRITTEN;
      case RETURN_MSG_WRITTEN:
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao escrever no canal.");
          return false;
        }
        buffer.clear();
        session.setCurrentState(new GetOperationState());

        if (logger.isLoggable(Level.FINER))
          logger.finer("Cdigo OK enviado.");
      default:
        return true;
    }
  }
}
