/*
 * Detalhes da ltima alterao:
 * 
 * $Author: vfusco $ $Date: 2009-02-18 18:33:43 -0300 (Wed, 18 Feb 2009) $
 * $Revision: 88595 $
 */
package tecgraf.ftc_1_3.server.states;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_3.common.logic.ErrorCode;
import tecgraf.ftc_1_3.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_3.server.FileChannelRequestInfo;
import tecgraf.ftc_1_3.server.FileServer;
import tecgraf.ftc_1_3.server.Session;
import tecgraf.ftc_1_3.utils.IOUtils;

/**
 * Operao para leitura de dados a partir de uma determinada posio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ReadState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a posio a partir da qual os dados sero lidos j
     * foi lida.
     */
    POSITION_READ,
    /**
     * Estado que indica que a quantidade de bytes que sero lidos j foi lida.
     */
    BYTE_COUNT_READ,
    /**
     * Estado que indica que o cabealho de resposta ja foi montado.
     */
    CHUNK_HEADER_WRITTEN,
    /**
     * Estado que indica que o cabealho de resposta ao comando ja foi enviado.
     */
    CHUNK_HEADER_SENT,
    /**
     * Estado que indica que todos os bytes solicitados j foram enviados.
     */
    BYTES_SENT;
  }

  /**
   * Tamanho maximo de cada chunk de dados
   */
  static long CHUNK_SIZE = 2048;

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;

  /**
   * A posio a partir da qual os dados sero lidos do arquivo.
   */
  private long position;
  /**
   * Quantidade de bytes que sero lidos do arquivo.
   */
  private long count;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Tamanho maximo a ser transmitido na chamada do metodo transferTo do
   * fileChannel.
   */
  public static final long MAX_BYTES = Integer.MAX_VALUE;

  /**
   * A quantidade de bytes enviados ao cliente.
   */
  private long bytesSent = 0;

  /**
   * Informa quantos bytes do chunk atual ja foram enviados.
   */
  private long chunkBytesSent = 0;

  /**
   * Informa o tamanho do chunk atual.
   */
  private long currentChunkSize = 0;

  /**
   * Tamanho do arquivo no momento da leitura.
   */
  private long fileSize = 0;

  /**
   * Variavel que indica que o final do arquivo foi atingido.
   */
  private boolean endOfFile = false;

  /**
   * Cria a operao para leitura de dados a partir de uma determinada posio.
   */
  public ReadState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de leitura.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    FileChannel fileChannel = session.getFileChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler do canal.");
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        this.position = buffer.getLong();
        buffer.clear();

        if (this.position < 0) {
          try {
            this.position = fileChannel.position();
          }
          catch (Exception e1) {
            if (logger.isLoggable(Level.FINER))
              logger.finer("Erro ao ler posio do arquivo.");
            return false;
          }
        }
        this.currentState = InternalState.POSITION_READ;

        if (logger.isLoggable(Level.FINER))
          logger.finer("Position " + this.position);
      case POSITION_READ:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler do canal.");
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.count = buffer.getLong();
        buffer.clear();
        this.currentState = InternalState.BYTE_COUNT_READ;

        try {
          this.fileSize = fileChannel.size();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler tamanho do arquivo.");
          return false;
        }

        if (logger.isLoggable(Level.FINER))
          logger.finer("Quantidade pedida " + this.count);
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    SocketChannel channel = session.getChannel();
    ByteBuffer buffer = session.getBuffer();
    FileChannel fileChannel = session.getFileChannel();
    FileChannelRequestInfo fileInfo = session.getFileChannelInfo();

    switch (this.currentState) {
      case BYTE_COUNT_READ:
        buffer.clear();
        long bytesLeftOnFile = fileSize - (position + bytesSent);
        if (bytesLeftOnFile <= 0) {
          logger.finer("Final do arquivo atingido enviando EOF");

          buffer.limit(PrimitiveTypeSize.BYTE.getSize());
          buffer.put(ErrorCode.END_OF_FILE.getCode());
          buffer.flip();
          endOfFile = true;
        }
        else {
          long missingBytes = count - bytesSent;
          currentChunkSize =
            (missingBytes > CHUNK_SIZE) ? CHUNK_SIZE : missingBytes;
          currentChunkSize =
            (currentChunkSize > bytesLeftOnFile) ? bytesLeftOnFile
              : currentChunkSize;

          if (logger.isLoggable(Level.FINER))
            logger.finer("Enviando chunk de tamanho " + currentChunkSize);

          buffer.limit(PrimitiveTypeSize.BYTE.getSize()
            + PrimitiveTypeSize.LONG.getSize());
          buffer.put(ErrorCode.OK.getCode());
          buffer.putLong(currentChunkSize);
          buffer.flip();
        }

        this.currentState = InternalState.CHUNK_HEADER_WRITTEN;

      case CHUNK_HEADER_WRITTEN:
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          buffer.clear();
          session.getFileServer().exceptionRaised(e,
            session.getFileChannelInfo().getFileId());
          return false;
        }

        if (buffer.hasRemaining())
          return true;

        buffer.clear();
        if (endOfFile) {
          session.setCurrentState(new GetOperationState());
          return true;
        }
        this.currentState = InternalState.CHUNK_HEADER_SENT;
      case CHUNK_HEADER_SENT:
        long bytesToread = currentChunkSize - chunkBytesSent;
        long bytesWritten = 0;
        try {
          if (fileInfo.useTransferTo()
            && ((count <= MAX_BYTES) || (!FileServer.PLATAFORM_HAS_TRANSFERTO_BUG))) {

            if (logger.isLoggable(Level.FINEST))
              logger.finest("Using TransferTo");

            bytesWritten =
              fileChannel.transferTo(this.position + this.bytesSent,
                bytesToread, channel);
          }
          else {
            bytesWritten =
              IOUtils.transferToNonBlock(fileChannel, this.position
                + this.bytesSent, bytesToread, channel, buffer);
            buffer.clear();
          }

          if (bytesWritten > 0) {
            session.markLastActivity();
            this.bytesSent += bytesWritten;
            fileChannel.position(this.position + bytesSent);
          }

        }
        catch (IOException e) {
          buffer.clear();
          session.getFileServer().exceptionRaised(e,
            session.getFileChannelInfo().getFileId());
          return false;
        }

        if (logger.isLoggable(Level.FINEST))
          logger.finest("Enviados " + bytesWritten);

        // Se todos os bytes desse chunk foram enviados, comecar um novo chunk.
        this.chunkBytesSent += bytesWritten;
        if (this.chunkBytesSent == currentChunkSize) {
          chunkBytesSent = 0;
          this.currentState = InternalState.BYTE_COUNT_READ;
        }

        // Se todos os bytes pedidos ja foram enviados, esperar nova operacao
        if (this.bytesSent == this.count) {
          buffer.clear();
          this.currentState = InternalState.BYTES_SENT;
          session.setCurrentState(new GetOperationState());
        }
      default:
        return true;
    }
  }
}
