/*
 * $Id$
 */
package tecgraf.ftc_1_3.server.states.v1_1;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_3.common.logic.ErrorCode;
import tecgraf.ftc_1_3.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_3.server.ChannelClosedReason;
import tecgraf.ftc_1_3.server.Session;
import tecgraf.ftc_1_3.server.states.State;

/**
 * Operao para fechamento de arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class CloseState implements State {

  /**
   * Codigo de erro a ser retornado.
   */
  private ErrorCode errorCode = ErrorCode.FAILURE;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria a operao de fechamento de arquivo.
   */
  public CloseState() {
    this(ErrorCode.FAILURE);
  }

  /**
   * Cria a operao de fechamento de arquivo.
   * 
   * @param error
   */
  public CloseState(ErrorCode error) {
    errorCode = error;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de fechamento.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    if (session.closeFileChannel(ChannelClosedReason.REQUESTED_BY_CLIENT)) {
      errorCode = ErrorCode.OK;
    }
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    buffer.limit(PrimitiveTypeSize.BYTE.getSize());
    buffer.put(errorCode.getCode());
    buffer.flip();
    try {
      if (channel.write(buffer) > 0)
        session.markLastActivity();
    }
    catch (IOException e) {
      session.getFileServer().exceptionRaised(e,
        session.getFileChannelInfo().getFileId());
      return false;
    }
    finally {
      buffer.clear();
    }
    try {
      channel.close();
    }
    catch (IOException e) {
      session.getFileServer().exceptionRaised(e,
        session.getFileChannelInfo().getFileId());
    }
    session.setCurrentState(null);
    return true;
  }
}
