/*
 * $Id: OpenState.java 88595 2009-02-18 21:33:43Z vfusco $
 */
package tecgraf.ftc_1_3.server.states.v1_1;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_3.common.exception.FailureException;
import tecgraf.ftc_1_3.common.exception.PermissionException;
import tecgraf.ftc_1_3.common.logic.ErrorCode;
import tecgraf.ftc_1_3.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_3.server.FileChannelRequestInfo;
import tecgraf.ftc_1_3.server.FileServer;
import tecgraf.ftc_1_3.server.Session;
import tecgraf.ftc_1_3.server.states.State;

/**
 * Operao de abertura de um arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class OpenState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  protected enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que o tamanho do identificador do arquivo j foi
     * recebido.
     */
    IDENTIFIER_SIZE_RECEIVED,
    /**
     * Estado que indica que o identificador do arquivo j foi recebido.
     */
    IDENTIFIER_RECEIVED,
    /**
     * Estado que indica que o cdigo de erro j foi enviado.
     */
    ERROR_CODE_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;
  /**
   * O tamanho do identificador do arquivo.
   */
  private byte fileIdSize;
  /**
   * O identificador do arquivo.
   */
  private byte[] fileId;
  /**
   * O canal do arquivo.
   */
  private FileChannel fileChannel;
  /**
   * O cdigo de erro.
   */
  private ErrorCode errorCode;

  /**
   * Indica se o arquivo ser aberto somente para leitura ou se ser aberto para
   * leitura e gravao.
   */
  private boolean readOnly;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria uma operao de abertura de arquivo.
   * 
   * @param readOnly Indica se o arquivo ser aberto somente para leitura ou se
   *        ser aberto para leitura e gravao.
   */
  protected OpenState(boolean readOnly) {
    this.readOnly = readOnly;
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de abertura de arquivo. Somente leitura? "
        + readOnly);
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.fileIdSize = buffer.get();
        buffer.clear();
        this.currentState = InternalState.IDENTIFIER_SIZE_RECEIVED;

        if (logger.isLoggable(Level.FINER))
          logger.finer("Tamanho do identificador lido " + this.fileIdSize);

      case IDENTIFIER_SIZE_RECEIVED:
        buffer.limit(this.fileIdSize);
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.fileId = new byte[this.fileIdSize];
        buffer.get(this.fileId);
        buffer.clear();
        this.currentState = InternalState.IDENTIFIER_RECEIVED;

        if (logger.isLoggable(Level.FINER))
          logger.finer("Identificador lido " + this.fileId);

        FileChannelRequestInfo fileChannelInfo = session.getFileChannelInfo();
        if (this.fileId.equals(fileChannelInfo.getFileId())) {
          this.errorCode = ErrorCode.NO_PERMISSION;
        }
        else {
          FileServer fileServer = session.getFileServer();
          try {
            this.fileChannel =
              fileServer.getFileProvider().createFileChannel(
                fileChannelInfo.getRequester(), fileChannelInfo.getFileId(),
                this.readOnly);
            if (this.fileChannel == null) {
              this.errorCode = ErrorCode.FILE_NOT_FOUND;
            }
            else {
              this.errorCode = ErrorCode.OK;
            }
          }
          catch (FailureException e) {
            this.errorCode = ErrorCode.FAILURE;
          }
          catch (PermissionException e) {
            this.errorCode = ErrorCode.NO_PERMISSION;
          }
        }
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case IDENTIFIER_RECEIVED:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(this.errorCode.getCode());
        buffer.flip();
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        buffer.clear();
        this.currentState = InternalState.ERROR_CODE_SENT;

        if (logger.isLoggable(Level.FINER))
          logger.finer("Cdigo " + this.errorCode + " enviado.");

        if (this.errorCode.equals(ErrorCode.OK)) {
          session.setFileChannel(this.fileChannel);
        }
        else {
          return false;
        }
        session.setCurrentState(new GetOperationState());
      default:
        return true;
    }
  }
}
