/**
 * $Id: ConfigurationManager.java 108737 2010-08-05 21:07:49Z costa $
 */
package tecgraf.javautils.configurationmanager;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.*;

import tecgraf.javautils.configurationmanager.Logger.Mode;

/**
 * Representa um gerenciador de configuraes. Essa classe  o ponto de acesso
 * nico para obteno de configurao a partir de uma determinada classe.
 * 
 * Uma classe tem acesso no somente  suas propriedades, mas tambm s
 * propriedades de seus ancestrais (superclasses).
 * 
 * Existem 2 {@link Mode modos de funcionamento} para o gerenciador de
 * configuraes: o modo {@link Mode#VERBOSE verbose} e o modo
 * {@link Mode#SILENT silencioso}. No modo verbose, praticamente todas as
 * operaes no gerenciador geram uma mensagem na sada de erro, No modo
 * silencioso, nada  impresso na sada de erro. O modo padro  {link
 * #MODE_PROPERTY_VALUE}. Para trocar de modo, crie um arquivo de configurao
 * para o gerente de configuraes com a chave {@link #MODE_PROPERTY_NAME}. Os
 * valores possveis so: {@link Mode#VERBOSE} e {@link Mode#SILENT}.
 * 
 * @author Tecgraf
 */
public final class ConfigurationManager {

  /** Separador de recursos. */
  private static final char RESOURCE_NAME_SEPARATOR = '/';

  /** Separador de pacotes. */
  private static final char PACKAGE_SEPARATOR = '.';

  /** Extenso do arquivo de properties. */
  private static final String CONFIGURATION_FILE_EXTENSION = ".properties";

  /** Prefixo do arquivo de configurao. */
  private static final String CONFIGURATION_PREFIX = RESOURCE_NAME_SEPARATOR
    + "conf" + RESOURCE_NAME_SEPARATOR;

  /** Extenso padro. */
  private static final String DEFAULT_EXTENSION_PREFIX = CONFIGURATION_PREFIX
    + "ext" + RESOURCE_NAME_SEPARATOR;

  /** Nome do modo. */
  private static final String MODE_PROPERTY_NAME = "mode";

  /** Modo padro. */
  private static final Mode MODE_PROPERTY_VALUE = Mode.SILENT;

  /** A instncia nica do gerente de configuraes. */
  private static ConfigurationManager instance;

  /** Mapa da chave. */
  static Map<String, String> cache;

  /**
   * Lista de prefixos de classes que <b>no</b> devem ser includas na
   * hierarquia de configuraes para uma determinada classe. Esta lista  usada
   * para evitar que armazenemos desnecessariamente configuraes para classes
   * internas da linguagem.
   */
  private static final List<String> CLASS_PREFIXES_BLACKLIST = Arrays.asList(
    "java.", "javax.", "com.", "org.");

  /**
   * Um mapa de configuraes. Esse mapa tem como chave uma classe (
   * {@link Class}) e como valor uma configurao ({@link Configuration}).
   */
  private final Map<Class<?>, Configuration> configurationMap;

  /**
   * Representa o prefixo para obteno de propriedades de extenso (instncias
   * do framework).
   */
  private final String extensionPrefix;

  /** Construtor. */
  private ConfigurationManager() {
    this(DEFAULT_EXTENSION_PREFIX);
  }

  /**
   * Construtor.
   * 
   * @param extensionPrefix prefixo para obteno de propriedades de extenso.
   */
  private ConfigurationManager(String extensionPrefix) {
    this.configurationMap = new HashMap<>();
    this.extensionPrefix = extensionPrefix;

    try {
      Configuration configuration =
        getConfiguration(ConfigurationManager.class);
      Logger.Mode mode =
        configuration.getOptionalEnumerationProperty(MODE_PROPERTY_NAME,
          Mode.class, MODE_PROPERTY_VALUE);
      Logger.getInstance().setMode(mode);
    }
    catch (ConfigurationManagerException e) {
      log(e, "Erro ao tentar carregar as propriedades do ConfigurationManager.");
    }
    log("Modo: %s.", Logger.getInstance().getMode());
    log("Prefixo de extenso: %s.", extensionPrefix);
  }

  /**
   * Obtm a configurao pertencente  uma determinada classe.
   * 
   * @param ownerClass classe dona da configurao.
   * 
   * @return configurao da classe, ou uma configurao vazia caso a
   *         configurao no exista.
   * 
   * @throws ConfigurationManagerException caso ocorra algum problema ao
   *         carregar as configuraes.
   * @throws IllegalArgumentException caso a classe dona seja nula.
   */
  public Configuration getConfiguration(Class<?> ownerClass)
    throws ConfigurationManagerException {
    if (ownerClass == null) {
      throw new IllegalArgumentException(
        "A classe dona da configurao no pode ser nula.");
    }
    log("Solicitada a configurao da classe %s.", ownerClass.getName());
    Configuration currentConfiguration = configurationMap.get(ownerClass);
    if (currentConfiguration != null) {
      /*
       * j temos uma configurao para a classe corrente, assumimos que as
       * configuraes para a hierarquia de superclasses j foi criada
       */
      return currentConfiguration;
    }

    Stack<Class<?>> classStack = new Stack<>();
    /*
     * criamos uma pilha com a hierarquia de classes (o topo possui a raiz da
     * hierarquia)
     */
    for (Class<?> currentClass = ownerClass; currentClass != null; currentClass =
      currentClass.getSuperclass()) {
      String className = currentClass.getName();
      /*
       * s criamos um cache para classes que no pertenam a determinados
       * pacotes
       */
      boolean skip = false;
      for (String forbiddenClass : CLASS_PREFIXES_BLACKLIST) {
        if (className.startsWith(forbiddenClass)) {
          skip = true;
          break;
        }
      }
      if (!skip) {
        classStack.push(currentClass);
      }
    }
    Configuration previousConfiguration = null;
    /*
     * percorremos a hierarquia do topo at a classe corrente
     */
    while (!classStack.isEmpty()) {
      Class<?> currentClass = classStack.pop();
      log("Tentando obter a configurao da classe %s "
        + "para atender a solicitao de configurao para a classe %s.",
        currentClass.getName(), ownerClass.getName());
      currentConfiguration = this.configurationMap.get(currentClass);
      if (currentConfiguration == null) {
        /*
         * no temos um configurador para a classe, lemos as propriedades
         * correspondentes de um arquivo e o criamos
         */
        log("A configurao da classe %s no est no cache.", currentClass
          .getName());
        currentConfiguration =
          new Configuration(currentClass, previousConfiguration);
        Properties configurationProperties =
          this.loadProperties(currentClass, CONFIGURATION_PREFIX);
        if (configurationProperties != null) {
          log("Realizando o merge da configurao da classe %s "
            + "utilizando o prefixo %s.", currentClass.getName(),
            CONFIGURATION_PREFIX);
          currentConfiguration.merge(configurationProperties);
          log("O merge da configurao da classe %s "
            + "utilizando o prefixo %s foi feito.", currentClass.getName(),
            CONFIGURATION_PREFIX);
        }
        Properties extensionProperties =
          this.loadProperties(currentClass, this.extensionPrefix);
        if (extensionProperties != null) {
          /*
           * existiam propriedades "de extenso", adicionamos as mesmas s
           * propriedades. As extenses tm prioridade (sobrescrevem as
           * propriedades "originais").
           */
          log("Realizando o merge da configurao da classe %s "
            + "utilizando o prefixo %s.", currentClass.getName(),
            this.extensionPrefix);
          currentConfiguration.merge(extensionProperties);
          log("O merge da configurao da classe %s "
            + "utilizando o prefixo %s foi feito.", currentClass.getName(),
            this.extensionPrefix);
        }
        this.configurationMap.put(currentClass, currentConfiguration);
        log("A configurao da classe %s adicionada ao cache.", currentClass
          .getName());
      }
      previousConfiguration = currentConfiguration;
    }
    if (currentConfiguration != null) {
      log("A configurao da classe %s foi encontrada.", ownerClass.getName());
    }
    else {
      log("A configurao da classe %s no foi encontrada.", ownerClass
        .getName());
    }
    return currentConfiguration;
  }

  /**
   * Cria a instncia nica do gerente de configuraes.
   * 
   * @throws IllegalStateException Caso j exista instncia criada.
   */
  public static void createInstance() {
    if (instance != null) {
      throw new IllegalStateException("J existe uma instncia criada.");
    }
    instance = new ConfigurationManager();
  }

  /**
   * Cria a instncia nica do gerente de configuraes.
   * 
   * @param extensionPrefix O prefixo para obteno de propriedades de extenso.
   * 
   * @throws IllegalStateException Caso j exista instncia criada.
   */
  public static void createInstance(String extensionPrefix) {
    if (instance != null) {
      throw new IllegalStateException("J existe uma instncia criada.");
    }
    instance = new ConfigurationManager(extensionPrefix);
  }

  /**
   * Obtm a instncia nica do gerente de configuraes.
   * 
   * @return O gerente de configuraes.
   * 
   * @throws IllegalStateException Caso no exista instncia criada.
   */
  public static ConfigurationManager getInstance() {
    if (instance == null) {
      throw new IllegalStateException("No existe nenhuma instncia criada.");
    }
    return instance;
  }

  /**
   * Realiza o log das informaes na sada padro.
   * 
   * @param throwable A causa.
   * @param message A mensagem.
   * @param args Os argumentos da mensagem.
   */
  private void log(Throwable throwable, String message, Object... args) {
    Logger.getInstance().log(throwable, message, args);
  }

  /**
   * Realiza o log das informaes na sada padro.
   * 
   * @param message A mensagem.
   * @param args Os argumentos da mensagem.
   */
  private void log(String message, Object... args) {
    Logger.getInstance().log(message, args);
  }

  /**
   * Cria as propriedades a partir de um caminho de arquivo.
   * 
   * @param ownerClass classe dona do arquivo de propriedades.
   * @param prefix prefixo para busca do arquivo.
   * 
   * @return propriedades carregadas, ou null, caso a propriedade no seja
   *         encontrada.
   * 
   * @throws ConfigurationManagerException caso ocorra um erro ao carregar o
   *         arquivo.
   */
  private Properties loadProperties(Class<?> ownerClass, String prefix)
    throws ConfigurationManagerException {
    log("Carregando as propriedades da classe %s utilizando o prefixo %s.",
      ownerClass.getName(), prefix);
    StringBuilder strBuilder = new StringBuilder(prefix);
    /*
     * garantimos que o prefixo ser corretamente usado mesmo que no termine
     * com '/'
     */
    if (prefix.charAt(prefix.length() - 1) != RESOURCE_NAME_SEPARATOR) {
      strBuilder.append(RESOURCE_NAME_SEPARATOR);
    }
    strBuilder.append(ownerClass.getName().replace(PACKAGE_SEPARATOR,
      RESOURCE_NAME_SEPARATOR));
    strBuilder.append(CONFIGURATION_FILE_EXTENSION);

    String propertiesFile = strBuilder.toString();
    log("Verificando se o arquivo %s pode ser encontrado.", propertiesFile);

    InputStream inputStream = getInputStreamForResource(propertiesFile);
    if (inputStream == null) {
      return null;
    }
    log("Carregandos as propriedades de %s.", propertiesFile);
    Properties properties = new Properties();
    try {
      properties.load(inputStream);
    }
    catch (IOException e) {
      ConfigurationManagerException exception =
        new ConfigurationManagerException(e,
          "Erro ao carregar as propriedades de {0}.", propertiesFile);
      throw exception;
    }
    log("As propriedades do arquivo %s foram carregadas.", propertiesFile);

    StringBuilder message = new StringBuilder();
    final String fmt = "A seguir uma listagem de todas as propriedades do arquivo %s. ";
    message.append(String.format(fmt, propertiesFile));
    message.append("Cada propriedade ser exibida com seguinte formato:\n");
    message.append("[chave]=[valor]\n");
    message.append("Os colchetes delimitam a chave e o valor. ");
    message.append("A chave e o valor so separados por '='.\n");
    final Set<Map.Entry<Object, Object>> entries = properties.entrySet();
    for (Map.Entry<Object, Object> entry : entries) {
      final Object key = entry.getKey();
      final Object value = entry.getValue();
      message.append(String.format("[%s]=[%s]%n", key, value));
    }
    log(message.toString());

    log( "As propriedades da classe %s utilizando o prefixo %s foram carregadas.",
      ownerClass.getName(), prefix);
    return properties;
  }

  /**
   * Obtm um {@link InputStream} para um arquivo de propriedades, seja este um
   * path absoluto local ou uma URL. Os protocolos aceitos para a URL so:
   * https, file e ftp.
   * 
   * @param propertiesFile path absoluto ou URL.
   * @return stream para o arquivo de propriedades ou <code>null</code> em caso
   *         de erro (URL/path invlido, erro de I/O etc.)
   */
  private InputStream getInputStreamForResource(String propertiesFile) {
    if (propertiesFile.matches("^(https?|file|ftp):.*")) {
      try {
        return new URL(propertiesFile).openStream();
      }
      catch (Exception e) {
        log(e, "Erro acessando a URL {0}", propertiesFile);
        return null;
      }
    }
    InputStream inputStream =
      ConfigurationManager.class.getResourceAsStream(propertiesFile);
    if (inputStream == null) {
      log("O arquivo %s no foi encontrado.", propertiesFile);
    }
    return inputStream;
  }
}
