package tecgraf.javautils.configurationmanager;

import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Log rudimentar usado pelo {@link ConfigurationManager} para exibir detalhes
 * do seu funcionamento.
 * 
 * Existem 2 {@link Mode modos de funcionamento}: {@link Mode#SILENT silencioso}
 * e {@link Mode#VERBOSE verbose}. No modo silencioso nada  escrito e no modo
 * verbose todas as chamadas ao log geram mensagens na sada de erro. O modo
 * padro  verbose.
 * 
 * @author Tecgraf
 */
public class Logger {

  /** A instncia nica desta classe. */
  private static Logger instance;

  /** O modo de funcionamento atual. */
  private Mode mode;

  /** Construtor. */
  private Logger() {
    setMode(Mode.SILENT);
  }

  /**
   * Obtm a instncia nica desta classe.
   * 
   * @return instncia.
   */
  public static Logger getInstance() {
    if (instance == null) {
      instance = new Logger();
    }
    return instance;
  }

  /**
   * Obtm o modo de funcionamento.
   * 
   * @return O modo.
   */
  public Mode getMode() {
    return mode;
  }

  /**
   * Realiza o log de informaes.
   * 
   * @param message mensagem.
   * @param args argumentos da mensagem.
   */
  public void log(String message, Object... args) {
    if (mode.equals(Mode.VERBOSE)) {
      Date date = new Date();
      SimpleDateFormat dateFormat =
        new SimpleDateFormat("yyyy/MM/dd - HH:mm:ss");
      System.err.printf("LOG: [%s] %s%n", dateFormat.format(date),
        String.format(message, args));
    }
  }

  /**
   * Realiza o log de informaes.
   * 
   * @param throwable exceo a ser registrada.
   * @param message mensagem.
   * @param args argumentos da mensagem.
   */
  public void log(Throwable throwable, String message, Object... args) {
    if (mode.equals(Mode.VERBOSE)) {
      log(message, args);
      Throwable cause = throwable;
      while ((cause != null) && (cause != cause.getCause())) {
        cause.printStackTrace();
        cause = cause.getCause();
      }
    }
  }

  /**
   * Habilita o modo de funcionamento.
   * 
   * @param mode modo de funcionamento (No aceita {@code null}).
   */
  void setMode(Mode mode) {
    if (mode == null) {
      throw new IllegalArgumentException("Mode no pode ser nulo.");
    }
    this.mode = mode;
  }

  /**
   * Modo de funcionamento do {@link Logger}.
   * 
   * @author Tecgraf
   */
  public enum Mode {

    /** Modo silencioso */
    SILENT,

    /** Modo verbose */
    VERBOSE
  }
}
