package busexplorer.panel.offers;

import busexplorer.utils.Utils;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Vector;

/**
 * Classe que detm as informaes locais da oferta para apresentao em
 * tabelas.
 * 
 * @author Tecgraf
 */
public class OfferWrapper {
  /** Objeto descritor da oferta */
  private ServiceOfferDesc desc;

  /** Identificador da entidade */
  private final String entity;
  /** Nome da interface */
  private final Vector<String> interfaces;
  /** Data da oferta */
  private final Date date;
  /** Nome do componente */
  private final String name;
  /** Verso do componente */
  private final String version;

  /**
   * /** Construtor.
   * 
   * @param desc descritor da oferta
   */
  public OfferWrapper(ServiceOfferDesc desc) {
    this.desc = desc;
    this.entity = Utils.getProperty(desc, "openbus.offer.entity");
    this.interfaces = Utils.getProperties(desc, "openbus.component.interface");
    int year = Integer.parseInt(Utils.getProperty(desc, "openbus.offer.year"));
    // precisa decrementar o mes em 1
    int month =
      Integer.parseInt(Utils.getProperty(desc, "openbus.offer.month")) - 1;
    int day = Integer.parseInt(Utils.getProperty(desc, "openbus.offer.day"));
    int hour = Integer.parseInt(Utils.getProperty(desc, "openbus.offer.hour"));
    int min = Integer.parseInt(Utils.getProperty(desc, "openbus.offer.minute"));
    int sec = Integer.parseInt(Utils.getProperty(desc, "openbus.offer.second"));
    Calendar calendar = Calendar.getInstance();
    calendar.set(year, month, day, hour, min, sec);
    this.date = calendar.getTime();
    this.name = Utils.getProperty(desc, "openbus.component.name");
    String version = Utils.getProperty(desc, "openbus.component.version.major");
    version += "." + Utils.getProperty(desc, "openbus.component.version.minor");
    version += "." + Utils.getProperty(desc, "openbus.component.version.patch");
    this.version = version;
  }

  /**
   * Compara um objeto  instncia de {@link OfferWrapper}.
   *
   * O mtodo no leva em considerao o objeto descritor da oferta.
   * 
   * @param o Objeto a ser comparado.
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof OfferWrapper)) {
      return false;
    }
    OfferWrapper other = (OfferWrapper) o;
    return entity.equals(other.entity) && interfaces.equals(other.interfaces) &&
      date.equals(other.date);
  }

  /**
   * Cdigo hash do objeto.
   *
   * @return Cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    return entity.hashCode() ^ interfaces.hashCode() ^ date.hashCode();
  }

  /**
   * Recupera o identificador da oferta.
   * 
   * @return o identificador.
   */
  public String getEntityId() {
    return entity;
  }

  /**
   * Obtm nome da interface da oferta.
   * 
   * @return nome da interface.
   */
  public Vector<String> getInterfaces() {
    return interfaces;
  }

  /**
   * Obtm data de incio da oferta.
   * 
   * @return String que representa a data de incio da oferta.
   */
  public Date getDate() {
    return date;
  }

  /**
   * Recupera o prprio objeto descritor de oferta.
   * 
   * @return o objeto descritor.
   */
  public ServiceOfferDesc getDescriptor() {
    return desc;
  }
  
  /**
   * Recupera o nome do componente da oferta.
   * 
   * @return o nome do componente.
   */
  public String getName() {
    return name;
  }
  
  /**
   * Recupera a verso do componente da oferta.
   * 
   * @return a string da verso do componente.
   */
  public String getVersion() {
    return version;
  }

  /**
   * Mtodo utilitrio para converter lista de {@link ServiceOfferDesc} para
   * {@link OfferWrapper}
   * 
   * @param offers a lista de {@link ServiceOfferDesc}
   * @return a lista de {@link OfferWrapper}
   */
  public static List<OfferWrapper> convertToInfo(List<ServiceOfferDesc> offers) {
    List<OfferWrapper> list = new ArrayList<OfferWrapper>();
    for (ServiceOfferDesc offer : offers) {
      list.add(new OfferWrapper(offer));
    }
    return list;
  }

}
