package br.pucrio.tecgraf.soma.websocketnotifier.infrastructure.persistence.message;

import br.pucrio.tecgraf.soma.serviceapi.configuration.ServiceConfiguration;
import br.pucrio.tecgraf.soma.websocketnotifier.application.configuration.Constants.Config;
import br.pucrio.tecgraf.soma.websocketnotifier.model.WebsocketNotifierInfo;
import br.pucrio.tecgraf.soma.websocketnotifier.model.WebsocketNotifierStatus;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import io.confluent.kafka.serializers.AbstractKafkaAvroSerDeConfig;
import io.confluent.kafka.serializers.KafkaAvroDeserializer;
import io.confluent.kafka.serializers.KafkaAvroDeserializerConfig;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.event.EventListener;
import org.springframework.messaging.MessagingException;
import org.springframework.stereotype.Service;

import java.util.*;

import static org.apache.kafka.clients.consumer.ConsumerConfig.*;


@Service
public class KafkaEventReader extends Observable {

  private final Logger logger = LoggerFactory.getLogger(JsonObject.class);

  private final String KAFKA_SERVER;
  private final String KAFKA_SCHEMA_REGISTRY_URL;
  private final List<String> KAFKA_TOPICS;
  private final List<String> KAFKA_GROUP;

  @Autowired
  public KafkaEventReader(ServiceConfiguration serviceConfiguration) {
    this.KAFKA_SERVER = serviceConfiguration.getValue(Config.KAFKA_SERVER_ADDRESS.option.getLongName());
    this.KAFKA_SCHEMA_REGISTRY_URL = serviceConfiguration.getValue(Config.KAFKA_SCHEMA_REGISTRY_URL.option.getLongName());
    this.KAFKA_TOPICS = Arrays.asList(serviceConfiguration.getValue(Config.KAFKA_TOPIC.option.getLongName()).split(";"));
    this.KAFKA_GROUP = Arrays.asList(serviceConfiguration.getValue(Config.KAFKA_CONSUMER_GROUP.option.getLongName()).split(";"));
  }

  @EventListener(ApplicationReadyEvent.class)
  public void run() {
    if (KAFKA_TOPICS.size() != KAFKA_GROUP.size()) {
      String msg =
              
                      "List arguments value of --%s and --%s must have the same length".formatted(
                      Config.KAFKA_TOPIC.option.getLongName(),
                      Config.KAFKA_CONSUMER_GROUP.option.getLongName());
      logger.error(msg);
      throw new IllegalArgumentException(msg);
    }
    List<Consumer<String, Object>> consumers = new ArrayList<>();
    for (int i = 0; i < KAFKA_TOPICS.size(); i++) {
      consumers.add(buildKafkaConsumer(KAFKA_SERVER, KAFKA_SCHEMA_REGISTRY_URL, KAFKA_TOPICS.get(i), KAFKA_GROUP.get(i)));
    }
    while (!Thread.currentThread().isInterrupted()) {
      for (Consumer<String, Object> consumer: consumers) {
        readRecords(consumer);
      }
    }
  }

  protected Properties buildProperties(String kafkaServer, String schemaRegistryUrl, String group) {
    Properties properties = new Properties();
    properties.setProperty(BOOTSTRAP_SERVERS_CONFIG, kafkaServer);
    properties.setProperty(KEY_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class.getName());
    properties.setProperty(VALUE_DESERIALIZER_CLASS_CONFIG, KafkaAvroDeserializer.class.getName());
    properties.setProperty(AbstractKafkaAvroSerDeConfig.SCHEMA_REGISTRY_URL_CONFIG, schemaRegistryUrl);
    properties.setProperty(KafkaAvroDeserializerConfig.SPECIFIC_AVRO_READER_CONFIG, "false");
    properties.setProperty(GROUP_ID_CONFIG, group);
    properties.setProperty(ENABLE_AUTO_COMMIT_CONFIG, "false");
    properties.setProperty(AUTO_COMMIT_INTERVAL_MS_CONFIG, "1000");
    properties.setProperty(AUTO_OFFSET_RESET_CONFIG, "earliest");
    return properties;
  }

  protected void readRecords(Consumer<String, Object> consumer) {
    ConsumerRecords<String, Object> records = consumer.poll(100);
    if(!records.isEmpty()) {
      logger.debug("readRecords Got {} records from Kafka", records.count());
    }
    for (TopicPartition partition : records.partitions()) {
      List<ConsumerRecord<String, Object>> partitionRecords = records.records(partition);
      for (ConsumerRecord<String, Object> record : partitionRecords) {
        readRecord(consumer, record, WebsocketNotifierStatus.LIVE.name());
      }
    }
  }

  public List<JsonObject> readRecordsFromOffset(Consumer<String, Object> consumer, WebsocketNotifierInfo wsInfo, long startPositionOffset) {
    List<JsonObject> recordFromOffset = new ArrayList<>();
    if(wsInfo.getOffset() == 0){
      // it is a new user and that does not need to receive update events after first connection
      return recordFromOffset;
    }
    consumer.poll(100);
    Set<TopicPartition> assignment = consumer.assignment();
    for (TopicPartition tp : assignment) {
      consumer.seek(tp, startPositionOffset);
      ConsumerRecords<String, Object> records = consumer.poll(1000);
      if(!records.isEmpty()) {
        logger.info("readRecordsFromOffset Got {} records from topic {} Kafka", records.count(), tp.topic());
      }
      for (TopicPartition partition : records.partitions()) {
        List<ConsumerRecord<String, Object>> partitionRecords = records.records(partition);
        for (ConsumerRecord<String, Object> record : partitionRecords) {
          JsonObject eventMessage = this.parseKafkaRecord(record, this.getEventStatus(record, wsInfo));
          recordFromOffset.add(eventMessage);
        }
      }
    }
    return recordFromOffset;
  }

  private String getEventStatus(ConsumerRecord<String, Object> record, WebsocketNotifierInfo wsInfo){
    if(record.offset() <= wsInfo.getOffset()) {
      return WebsocketNotifierStatus.OLD.name();
    } else {
      return WebsocketNotifierStatus.LOST.name();
    }
  }

  private JsonObject parseKafkaRecord(ConsumerRecord<String, Object> record, String eventStatus){
    JsonObject eventMessage = (new JsonParser()).parse(record.value().toString()).getAsJsonObject();
    eventMessage.addProperty("topic_name", record.topic());
    eventMessage.addProperty("offset", record.offset());
    eventMessage.addProperty("eventStatus", eventStatus);
    return eventMessage;
  }

  public void readRecord(Consumer<String, Object> consumer, ConsumerRecord<String, Object> record, String eventStatus) {
    try {
      setChanged();
      JsonObject eventMessage = this.parseKafkaRecord(record, eventStatus);
      logger.debug("readRecord got Job {}", eventMessage);
      notifyObservers(eventMessage);
    } catch (MessagingException me) {
      logger.error("Error trying send message to job-info", me);
    } catch (Exception e) {
      logger.error("Unrecoverable error.", e);
    } finally {
      try{
        consumer.commitSync();
      } catch (Exception e) {
        logger.error("Unrecoverable error during commitSync.", e);
      }
    }
  }

  public Consumer<String, Object> buildKafkaConsumer(String kafkaServer, String schemaRegistryUrl, String topic, String group) {
    Consumer<String, Object> consumer = new KafkaConsumer<>(buildProperties(kafkaServer, schemaRegistryUrl, group));
    consumer.subscribe(Collections.singletonList(topic));
    return consumer;
  }


}
