/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 * 
 * file: GridCoverageDataStoreFactory.java
 * created: 23/10/2008
 */
package org.geotools.data.gridcoverage;

import java.awt.RenderingHints.Key;
import java.io.IOException;
import java.io.Serializable;
import java.net.URL;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.geotools.coverage.grid.io.AbstractGridCoverage2DReader;
import org.geotools.coverage.grid.io.AbstractGridFormat;
import org.geotools.coverage.grid.io.GridFormatFactorySpi;
import org.geotools.coverage.grid.io.GridFormatFinder;
import org.geotools.coverage.grid.io.UnknownFormat;
import org.geotools.data.DataStore;
import org.geotools.data.DataStoreFactorySpi;
import org.geotools.data.DataStoreFinder;
import org.geotools.data.gridcoverage.exception.DataAccessGridCoverageException;

/**
 * Specifies a factory for creating a {@link GridCoverageDataStore} based on
 * appropriate connection parameters. This factory is used as a
 * {@link DataStoreFactorySpi} to allow GridCoverageDataStores to be
 * instantiated through the use of {@link DataStoreFinder}. In practice, the
 * connection parameters necessary for instantiating a GridCoverageDataStore
 * are:
 * <ul>
 * <li>the URL of the coverage data
 * <li>the FORMAT of the coverage data (e.g., "GeoTIFF", "WorldImage")
 * <i>(optional)</i>
 * <ul>
 * If no FORMAT parameter is specified, the factory will try its best to
 * instantiate an appropriate coverage reader based on the provided URL. In that
 * case, it should be noted that URLs with a .tif extension are generally
 * assumed to be in GeoTIFF format. As such, the instantiation of a
 * GridCoverageDataStore for a .tif/.tfw world image pair requires the
 * specification of a FORMAT parameter with the value "WorldImage".
 * 
 * @author milton
 * @since TDK 3.0.0
 */
public class GridCoverageDataStoreFactory implements DataStoreFactorySpi
{
    public static final Param URLP = new Param("url", URL.class, "url to a GridCoverage file");
    public static final Param FORMATP = new Param("format", String.class, "name of an AbstractGridFormat");

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataStoreFactorySpi#createDataStore(java.util.Map)
     */
    @Override
    public DataStore createDataStore(Map<String, Serializable> params) throws IOException
    {
        // checks for a invalid parameters
        if (params == null)
            throw new NullPointerException("params can't be null");
        if (params.size() == 0)
            throw new IllegalArgumentException("params can't be empty");

        // looks up URL parameter
        URL url = (URL) URLP.lookUp(params);

        // searches for an appropriate AbstractGridFormat
        AbstractGridFormat gridFormat = getFormat(params);
        if (gridFormat instanceof UnknownFormat)
            throw new IOException("Could not find an appropriate AbstractGridFormat for parameters '"
                    + params.toString() + "'");

        // used the format to retrieve a reader for the URL
        AbstractGridCoverage2DReader reader = gridFormat.getReader(url);

        // uses reader to instantiate a new GridCoverageDataStore
        try
        {
            return new DefaultGridCoverageDataStore(reader);
        }
        catch (DataAccessGridCoverageException e)
        {
            throw new IOException("Could not instantiate a datastore with reader '" + reader.toString() + "'");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.geotools.data.DataStoreFactorySpi#createNewDataStore(java.util.Map)
     * 
     * This method is not supported for GridCoverageDataStores, which can only
     * manipulate pre-existing coverages located at a specified URL.
     */
    @Override
    public DataStore createNewDataStore(Map<String, Serializable> params) throws IOException
    {
        throw new UnsupportedOperationException();
    }

    /**
     * Retrieves an available AbstractGridFormat whose name matches a specified
     * format name. If no format is found, then UnknownFormat is returned.
     * 
     * @param formatName the name that identifies the format (e.g.,
     *            "WorldImage")
     * @return the corresponding AbstractGridFormat, or UnknownFormat if no
     *         available appropriate formats were found.
     * @since TDK 3.0.0
     */
    protected AbstractGridFormat getFormat(String formatName)
    {
        // retrieves all GridFormatFactory objects available
        Set<GridFormatFactorySpi> factories = GridFormatFinder.getAvailableFormats();

        // iterates through the factories looking for an appropriate one
        for (GridFormatFactorySpi factory : factories)
        {
            // creates a format based on the factory and checks if:
            // - it is an instance of AbstractGridFormat
            // - its name matches the specified format name
        	AbstractGridFormat format = factory.createFormat();
            if (format.getName().equalsIgnoreCase(formatName))
                return format;
        }
        return new UnknownFormat();
    }

    /**
     * Retrieves an available AbstractGridFormat based on the specified
     * parameters. If the FORMAT parameter was specified, then it is used to
     * retrieve the appropriate format. If not, then an appropriate format is
     * searched based on the URL parameter. If no format is found, then
     * UnknownFormat is returned.
     * 
     * @param params parameters specifying a GridCoverageDataStore
     * @return an AbstractGridFormat appropriate for the given parameters, or
     *         UnknownFormat if none is found.
     * @since TDK 3.0.0
     */
    protected AbstractGridFormat getFormat(Map<String, Serializable> params)
    {
        // retrieves the mandatory URL parameter
        URL url = null;
        try
        {
            url = (URL) URLP.lookUp(params);
        }
        catch (IOException e)
        {
            return new UnknownFormat();
        }

        try
        {
            // looks for a format based on the FORMAT parameter
            String formatName = (String) FORMATP.lookUp(params);
            return getFormat(formatName);
        }
        catch (IOException ioe)
        {
            // if FORMAT was not specified, searches for a format based on the
            // URL parameter
            try
            {
                return (AbstractGridFormat) GridFormatFinder.findFormat(url);
            }
            catch (Exception e)
            {
                // DEBUG *** This code exists only because
                // GridFormatFinder.findFormat is wrongly throwing
                // IllegalArgumentExceptions
                return new UnknownFormat();
            }

        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataAccessFactory#canProcess(java.util.Map)
     */
    @Override
    public boolean canProcess(Map<String, Serializable> params)
    {
        // checks for a invalid parameters
        if (params == null)
            throw new NullPointerException("params can't be null");
        if (params.size() == 0)
            throw new IllegalArgumentException("params can't be empty");

        // retrieves format based on the specified parameters
        AbstractGridFormat gridFormat = getFormat(params);

        // returns whether a valid grid format was found
        if (gridFormat instanceof UnknownFormat)
            return false;
        else
            return true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataAccessFactory#getDescription()
     */
    @Override
    public String getDescription()
    {
        return "GridCoverage raster files";
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataAccessFactory#getDisplayName()
     */
    @Override
    public String getDisplayName()
    {
        return "GridCoverage";
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataAccessFactory#getParametersInfo()
     */
    @Override
    public Param[] getParametersInfo()
    {
        return new Param[] { URLP, FORMATP };
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataAccessFactory#isAvailable()
     */
    @Override
    public boolean isAvailable()
    {
        try
        {
            // checks for the availability of the necessary classes for this
            // factory:
            // - GridFormatFinder - to search for an appropriate grid format
            // - AbstractGridCoverage2DReader - to instantiate a reader for the
            // grid format
            // - GridCoverageDataStoreImpl - to instantiate an actual
            // GridCoverageDataStore
            GridFormatFinder.class.getName();
            AbstractGridCoverage2DReader.class.getName();
            DefaultGridCoverageDataStore.class.getName();
        }
        catch (Exception e)
        {
            return false;
        }
        return true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.factory.Factory#getImplementationHints()
     */
    @Override
    public Map<Key, ?> getImplementationHints()
    {
        return Collections.emptyMap();
    }

}
