/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 * 
 *    (C) 2004-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.renderer.lite;

import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Hashtable;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.batik.bridge.BridgeContext;
import org.apache.batik.bridge.DocumentLoader;
import org.apache.batik.bridge.GVTBuilder;
import org.apache.batik.bridge.UserAgent;
import org.apache.batik.bridge.UserAgentAdapter;
import org.apache.batik.dom.svg.SAXSVGDocumentFactory;
import org.apache.batik.gvt.GraphicsNode;
import org.apache.batik.util.XMLResourceDescriptor;
import org.geotools.renderer.style.SVGGraphicFactory;
import org.geotools.styling.Displacement;
import org.geotools.styling.ExternalGraphic;
import org.geotools.styling.Graphic;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.svg.SVGDocument;


/**
 * Turns SVG vector drawings into buffered images geotools can use for rendering
 * 
 * @author James
 * @source $URL:
 *         http://svn.geotools.org/geotools/trunk/gt/module/render/src/org/geotools/renderer/lite/SVGGlyphRenderer.java $
 * @deprecated, use {@link SVGGraphicFactory} instead
 */
public class SVGGlyphRenderer implements GlyphRenderer {
    private static Hashtable<URL,Document> cache = new Hashtable<URL,Document>(10);
    private static Hashtable<URL,GraphicsNode> _gvtCache = new Hashtable<URL, GraphicsNode>();

    private static final java.util.List<String> formats = java.util.Collections
            .unmodifiableList(java.util.Arrays.asList(new String[] { "image/svg", "image/svg+xml", "image/svg-xml" }));

    /** The logger for the rendering module. */
    private static final Logger LOGGER = org.geotools.util.logging.Logging
            .getLogger("org.geotools.rendering");

    // static {
    // do register our xml reader wrapper against batik so that we can use
    // jaxp instead of the hard-coded xerces implementation
    // XMLResourceDescriptor.setXMLParserClassName(BatikXMLReader.class.getName());
    // }

    /**
     * mini cache to stop re-loading SVG files. Dont know how effective this is,
     * but...
     */
    private Document getDocument(URL url) throws Exception {
        if (cache.contains(url))
            return (Document) cache.get(url);

        String parser = XMLResourceDescriptor.getXMLParserClassName();
        SAXSVGDocumentFactory f = new SAXSVGDocumentFactory(parser);

        Document doc = f.createDocument(url.toString());
        cache.put(url, doc);
        return doc;
    }

    /** Creates a new instance of SVGGlyphRenderer */
    public SVGGlyphRenderer() {
    }

    public boolean canRender(String format) {
    	return formats.contains(format.toLowerCase());
        //return (format.toLowerCase().equals("image/svg+xml"));
    }

    public java.util.List<String> getFormats() {
        return formats;
    }

    public java.awt.image.BufferedImage render(org.geotools.styling.Graphic graphic,
            org.geotools.styling.ExternalGraphic eg, Object feature, int height) {
        try {
            BufferedImage img;
            URL svgfile = eg.getLocation();
            InternalTranscoder magic = new InternalTranscoder();

            if (height > 0)
                magic.addTranscodingHint(InternalTranscoder.KEY_HEIGHT, new Float(height));

            Document inputDoc = getDocument(svgfile);
            // TranscoderInput in = new TranscoderInput(svgfile .openStream());
            magic.transcode(inputDoc);
            img = magic.getImage();
            return img;
        } catch (java.io.IOException mue) {
            LOGGER.log(Level.WARNING, "Unable to load external svg file", mue);
            return null;
        } catch (Exception te) {
            LOGGER.log(Level.WARNING, "Unable to load external svg file", te);
            return null;
        }
    }

	public void render(Graphics2D g, Graphic graphic, ExternalGraphic eg,
			Object feature, int height, float centerX, float centerY)	
	{
		
		
		try
		{
			URL svgLocation = eg.getLocation();
		
			GraphicsNode node = getGraphicNode(svgLocation);
			
			Rectangle2D bounds = getSvgDocBounds(svgLocation);
			
			if(bounds == null)
			{
				bounds =  node.getBounds();
			}
			
			double newWidth = bounds.getWidth();
			double newHeight = bounds.getHeight();
			if(height > 0)
			{
				double shapeAspectRatio = (bounds.getHeight() > 0 && bounds.getWidth() > 0) ? bounds.getWidth()/bounds.getHeight() : 1.0;
				newWidth = shapeAspectRatio * height;
				newHeight = height;
			}
			
			//save the old transform;
			AffineTransform oldTransform = node.getTransform();
			if(oldTransform == null) oldTransform = new AffineTransform();
			
			AffineTransform transform = new AffineTransform(oldTransform);
			
			if(graphic.getRotation() != null)
			{
				Object value = graphic.getRotation().evaluate(feature);
				double rotation = 0.0;
				if(value instanceof Number)
				{
					rotation = ((Number)value).doubleValue();
				}
				transform.rotate(Math.toRadians(rotation));
			}
			
			
			transform.translate(centerX-(newWidth/2.0), centerY-(newHeight/2.0));
			
			Displacement displacement = graphic.getDisplacement();
			if(displacement != null)
			{
				double displacementX = 0.0;
				double displacementY = 0.0;
				if(displacement.getDisplacementX() != null)
				{
					Object value = displacement.getDisplacementX().evaluate(feature);
					if(value instanceof Number)
					{
						displacementX = ((Number)value).doubleValue();
					}				
				}
				if(displacement.getDisplacementY() != null)
				{
					Object value = displacement.getDisplacementY().evaluate(feature);
					if(value instanceof Number)
					{
						displacementY = ((Number)value).doubleValue();
					}	
				}
				transform.translate(displacementX, displacementY);
			}
			
			
			
			transform.scale(newWidth/bounds.getWidth(), newHeight/bounds.getHeight());
			node.setTransform(transform);
			node.paint(g);
			
			node.setTransform(oldTransform);
			
		}catch(IOException e)
		{
			LOGGER.log(Level.WARNING, "Unable to load external svg file", e);
		}
		catch (URISyntaxException e)
		{
			LOGGER.log(Level.WARNING, "Unable to load external svg file", e);
		}
		
		
		
		
	}

	private Rectangle2D getSvgDocBounds(URL svgLocation) {

		SVGDocument doc = (SVGDocument) cache.get(svgLocation);

		NodeList list = doc.getElementsByTagName("svg");
		Node svgNode = list.item(0);
		
		NamedNodeMap attrbiutes = svgNode.getAttributes();
		Node widthNode = attrbiutes.getNamedItem("width");
		Node heightNode = attrbiutes.getNamedItem("height");
		
		if(widthNode != null && heightNode != null)
		{
			double width = Double.parseDouble(widthNode.getNodeValue());
			double height = Double.parseDouble(heightNode.getNodeValue());
			return new Rectangle2D.Double(0.0,0.0,width,height);
		}

		return null;
	}

	private GraphicsNode getGraphicNode(URL svgLocation) throws IOException, URISyntaxException {
		
		if(_gvtCache.contains(svgLocation)) return _gvtCache.get(svgLocation);
		
		String xmlParser = XMLResourceDescriptor.getXMLParserClassName();
		SAXSVGDocumentFactory df = new SAXSVGDocumentFactory(xmlParser);
		SVGDocument doc = df.createSVGDocument(svgLocation.toURI().toString());
		
		cache.put(svgLocation, doc);
		
		UserAgent userAgent = new UserAgentAdapter();
		DocumentLoader loader = new DocumentLoader(userAgent);
		BridgeContext ctx = new BridgeContext(userAgent, loader);
		ctx.setDynamic(true);
		GVTBuilder builder = new GVTBuilder();
		GraphicsNode rootNode = builder.build(ctx, doc);
		_gvtCache.put(svgLocation, rootNode);
		
		return rootNode;
	}

}
