package tecgraf.openbus;

import java.util.Properties;

import org.omg.CORBA.ORB;
import org.omg.PortableInterceptor.Current;

import tecgraf.openbus.core.v2_0.services.access_control.CallChain;
import tecgraf.openbus.core.v2_0.services.access_control.LoginRegistry;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferRegistry;
import tecgraf.openbus.exception.InvalidPropertyValue;

/**
 * Permite controlar o contexto das chamadas de um {@link ORB} para acessar
 * informaes que identificam essas chamadas em barramentos OpenBus.
 * <p>
 * O contexto de uma chamada pode ser definido pela linha de execuo atual do
 * programa em que executa uma chamada, o que pode ser a thread em execuo ou
 * mais comumente o {@link Current} do padro CORBA. As informaes acessveis
 * atravs do {@link OpenBusContext} se referem basicamente  identificao da
 * origem das chamadas, ou seja, nome das entidades que autenticaram os acessos
 * ao barramento que originaram as chamadas.
 * <p>
 * A identifcao de chamadas no barramento  controlada atravs do
 * OpenBusContext atravs da manipulao de duas abstraes representadas pelas
 * seguintes interfaces:
 * <ul>
 * <li> {@link Connection}: Representa um acesso ao barramento, que  usado tanto
 * para fazer chamadas como para receber chamadas atravs do barramento. Para
 * tanto a conexo precisa estar autenticada, ou seja, logada. Cada chamada
 * feita atravs do ORB  enviada com as informaes do login da conexo
 * associada ao contexto em que a chamada foi realizada. Cada chamada recebida
 * tambm deve vir atravs de uma conexo logada, que deve ser o mesmo login com
 * que chamadas aninhadas a essa chamada original devem ser feitas.
 * <li> {@link CallChain}: Representa a identicao de todos os acessos ao
 * barramento que originaram uma chamada recebida. Sempre que uma chamada 
 * recebida e executada,  possvel obter um CallChain atravs do qual 
 * possvel inspecionar as informaes de acesso que originaram a chamada
 * recebida.
 * </ul>
 * 
 * @author Tecgraf
 */
public interface OpenBusContext {

  /**
   * Recupera o ORB associado ao ConnectionManager.
   * 
   * @return o ORB
   */
  org.omg.CORBA.ORB orb();

  /**
   * Callback a ser chamada para determinar a conexo a ser utilizada para
   * receber cada chamada.
   * <p>
   * Esse atributo  utilizado para definir um objeto que implementa uma
   * interface de callback a ser chamada sempre que a conexo receber uma do
   * barramento. Essa callback deve devolver a conexo a ser utilizada para para
   * receber a chamada. A conexo utilizada para receber a chamada ser a nica
   * conexo atravs do qual novas chamadas aninhadas  chamada recebida podero
   * ser feitas (veja a operao {@link OpenBusContext#joinChain}).
   * <p>
   * Se o objeto de callback for definido como <code>null</code> ou devolver
   * <code>null</code>, a conexo padro  utilizada para receber achamada, caso
   * esta esteja definida.
   * <p>
   * Caso esse atributo seja <code>null</code>, nenhum objeto de callback 
   * chamado na ocorrncia desse evento e ???
   * 
   * @param callback Objeto que implementa a interface de callback a ser chamada
   *        ou <code>null</code> caso nenhum objeto deva ser chamado na
   *        ocorrncia desse evento.
   */
  void onCallDispatch(CallDispatchCallback callback);

  /**
   * Recupera a callback a ser chamada sempre que a conexo receber uma do
   * barramento.
   * 
   * @return a callback ou <code>null</code> caso ela no exista.
   */
  CallDispatchCallback onCallDispatch();

  /**
   * Cria uma conexo para um barramento. O barramento  indicado por um nome ou
   * endereo de rede e um nmero de porta, onde os servios ncleo daquele
   * barramento esto executando.
   * 
   * @param host Endereo ou nome de rede onde os servios ncleo do barramento
   *        esto executando.
   * @param port Porta onde os servios ncleo do barramento esto executando.
   * 
   * @return Conexo criada.
   */
  Connection createConnection(String host, int port);

  /**
   * Cria uma conexo para um barramento. O barramento  indicado por um nome ou
   * endereo de rede e um nmero de porta, onde os servios ncleo daquele
   * barramento esto executando.
   * 
   * @param host Endereo ou nome de rede onde os servios ncleo do barramento
   *        esto executando.
   * @param port Porta onde os servios ncleo do barramento esto executando.
   * @param props Lista opcional de propriedades que definem algumas
   *        configuraes sobre a forma que as chamadas realizadas ou validadas
   *        com essa conexo so feitas. A seguir so listadas as propriedades
   *        vlidas:
   *        <ul>
   *        <li>access.key: chave de acesso a ser utiliza internamente para a
   *        gerao de credenciais que identificam as chamadas atravs do
   *        barramento. A chave deve ser uma chave privada RSA de 2048 bits (256
   *        bytes). Quando essa propriedade no  fornecida, uma chave de acesso
   *         gerada automaticamente.
   *        <li>legacy.disable: desabilita o suporte a chamadas usando protocolo
   *        OpenBus 1.5. Por padro o suporte est habilitado. Valores esperados
   *        so <code>true</code> ou <code>false</code>.
   *        <li>legacy.delegate: indica como  preenchido o campo 'delegate' das
   *        credenciais enviadas em chamadas usando protocolo OpenBus 1.5. H
   *        duas formas possveis (o padro  'caller'):
   *        <ul>
   *        <li>caller: o campo 'delegate'  preenchido sempre com a entidade do
   *        campo 'caller' da cadeia de chamadas.
   *        <li>originator: o campo 'delegate'  preenchido sempre com a
   *        entidade que originou a cadeia de chamadas, que  o primeiro login
   *        do campo 'originators' ou o campo 'caller' quando este  vazio.
   *        </ul>
   *        </ul>
   * 
   * @return Conexo criada.
   * 
   * @throws InvalidPropertyValue O valor de uma propriedade no  vlido.
   */
  Connection createConnection(String host, int port, Properties props)
    throws InvalidPropertyValue;

  /**
   * Define a conexo padro a ser usada nas chamadas.
   * <p>
   * Define uma conexo a ser utilizada em chamadas sempre que no houver uma
   * conexo especfica definida no contexto atual, como  feito atravs da
   * operao {@link OpenBusContext#setCurrentConnection(Connection)
   * setRequester}. Quando <code>conn</code>  <code>null</code> nenhuma conexo
   * fica definida como a conexo padro.
   * 
   * @param conn Conexo a ser definida como conexo padro.
   * 
   * @return Conexo definida como conexo padro anteriormente, ou
   *         <code>null</code> se no havia conexo padro definida
   *         anteriormente.
   */
  Connection setDefaultConnection(Connection conn);

  /**
   * Devolve a conexo padro.
   * <p>
   * Veja operao {@link OpenBusContext#setDefaultConnection
   * setDefaultConnection}.
   * 
   * @return Conexo definida como conexo padro.
   */
  Connection getDefaultConnection();

  /**
   * Define a conexo associada ao contexto corrente.
   * <p>
   * Define a conexo a ser utilizada em todas as chamadas feitas no contexto
   * atual. Quando <code>conn</code>  <code>null</code> o contexto passa a
   * ficar sem nenhuma conexo associada.
   * 
   * @param conn Conexo a ser associada ao contexto corrente.
   * 
   * @return Conexo definida como a conexo corrente anteriormente, ou null se
   *         no havia conexo definida ateriormente.
   */
  Connection setCurrentConnection(Connection conn);

  /**
   * Devolve a conexo associada ao contexto corrente.
   * <p>
   * Devolve a conexo associada ao contexto corrente, que pode ter sido
   * definida usando a operao {@link OpenBusContext#setCurrentConnection} ou
   * {@link OpenBusContext#setDefaultConnection}.
   * 
   * @return Conexo associada ao contexto corrente, ou <code>null</code> caso
   *         no haja nenhuma conexo associada.
   */
  Connection getCurrentConnection();

  /**
   * Devolve a cadeia de chamadas  qual a execuo corrente pertence.
   * <p>
   * Caso a contexto corrente (e.g. definido pelo {@link Current}) seja o
   * contexto de execuo de uma chamada remota oriunda do barramento dessa
   * conexo, essa operao devolve um objeto que representa a cadeia de
   * chamadas do barramento que esta chamada faz parte. Caso contrrio, devolve
   * <code>null</code>.
   * 
   * @return Cadeia da chamada em execuo.
   */
  CallerChain getCallerChain();

  /**
   * Associa uma cadeia de chamadas ao contexto corrente.
   * <p>
   * Associa uma cadeia de chamadas ao contexto corrente (e.g. definido pelo
   * {@link Current}), de forma que todas as chamadas remotas seguintes neste
   * mesmo contexto sejam feitas como parte dessa cadeia de chamadas.
   * 
   * @param chain Cadeia de chamadas a ser associada ao contexto corrente.
   */
  void joinChain(CallerChain chain);

  /**
   * Associa a cadeia de chamadas obtida em {@link #getCallerChain()} ao
   * contexto corrente.
   * <p>
   * Associa a cadeia de chamadas obtida em {@link #getCallerChain()} ao
   * contexto corrente (e.g. definido pelo {@link Current}), de forma que todas
   * as chamadas remotas seguintes neste mesmo contexto sejam feitas como parte
   * dessa cadeia de chamadas.
   */
  void joinChain();

  /**
   * Faz com que nenhuma cadeia de chamadas esteja associada ao contexto
   * corrente.
   * <p>
   * Remove a associao da cadeia de chamadas ao contexto corrente (e.g.
   * definido pelo {@link Current}), fazendo com que todas as chamadas seguintes
   * feitas neste mesmo contexto deixem de fazer parte da cadeia de chamadas
   * associada previamente. Ou seja, todas as chamadas passam a iniciar novas
   * cadeias de chamada.
   */
  void exitChain();

  /**
   * Devolve a cadeia de chamadas associada ao contexto corrente.
   * <p>
   * Devolve um objeto que representa a cadeia de chamadas associada ao contexto
   * corrente (e.g. definido pelo {@link Current}) nesta conexo. A cadeia de
   * chamadas informada foi associada previamente pela operao
   * {@link #joinChain(CallerChain) joinChain}. Caso o contexto corrente no
   * tenha nenhuma cadeia associada, essa operao devolve <code>null</code>.
   * 
   * @return Cadeia de chamadas associada ao contexto corrente ou
   *         <code>null</code> .
   */
  CallerChain getJoinedChain();

  /**
   * Referncia ao servio ncleo de registro de logins do barramento
   * referenciado no contexto atual.
   * 
   * @return o servio de registro de logins.
   */
  LoginRegistry getLoginRegistry();

  /**
   * Referncia ao servio ncleo de registro de ofertas do barramento
   * referenciado no contexto atual.
   * 
   * @return o servio de registro de ofertas.
   */
  OfferRegistry getOfferRegistry();
}
