package tecgraf.openbus.core;

import java.util.Properties;

import tecgraf.openbus.exception.InvalidPropertyValue;

/**
 * Enumerao que define todas as possveis propriedades do domnio da
 * biblioteca de acesso, com os seus valores padres e um local nico para
 * recuperar e verificar validade do valor atribudo s propriedades
 * 
 * @author Tecgraf
 */
enum OpenBusProperty {

  /**
   * Chave da propriedade que define se o suporte legado deve ser habilitado ou
   * no. Os valores possveis so: <code>true</code> e <code>false</code>, onde
   * o padro  <code>true</code>
   */
  LEGACY_DISABLE("legacy.disable", "false"),
  /**
   * Chave da propriedade que define como o campo delegate do suporte legado
   * deve ser construdo a partir de uma credencial 2.0. Os valores possveis
   * so: "originator" e "caller", onde o padro  "caller".
   */
  LEGACY_DELEGATE("legacy.delegate", "caller");

  /** Nome da propriedade */
  private final String key;
  /** Valor padro da propriedade */
  private final String defaultValue;

  /**
   * Construtor.
   * 
   * @param key nome da propriedade.
   * @param value valor padro.
   */
  private OpenBusProperty(String key, String value) {
    this.key = key;
    this.defaultValue = value;
  }

  /**
   * Recupera a propriedade na lista de propriedades passada. Caso a mesma no
   * esteja especificada na lista, retorna-se o seu valor padro.
   * 
   * @param props a lista de propriedades.
   * @return o valor da propriedade.
   * @throws InvalidPropertyValue
   */
  String getProperty(Properties props) throws InvalidPropertyValue {
    String value = props.getProperty(this.key, this.defaultValue);
    value = value.toLowerCase();
    switch (this) {
      case LEGACY_DISABLE:
        if (value.equals("true") || value.equals("false")) {
          return value;
        }
        break;

      case LEGACY_DELEGATE:
        if (value.equals("caller") || value.equals("originator")) {
          return value;
        }
        break;

      default:
        return null;
    }
    throw new InvalidPropertyValue(this.key, value);
  }
}
